"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const main_1 = require("./main");
const app = new cdk.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new cdk.Stack(app, 'demo-stack', { env });
const orderTask = new ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
const zoneName = 'svc.local';
const internalAlbRecordName = 'internal';
const externalAlbRecordName = 'external';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
orderTask.addContainer('order', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'order',
        versionNum: '1.0',
    },
});
const customerTask = new ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
const productTask = new ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
const svc = new index_1.AlbFargateServices(stack, 'Service', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        // The order service with both external/internal access
        {
            listenerPort: 80,
            accessibility: main_1.LoadBalancerAccessibility.EXTERNAL_ONLY,
            task: orderTask,
            desiredCount: 2,
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            // The customer service(internal only)
            accessibility: main_1.LoadBalancerAccessibility.INTERNAL_ONLY,
            listenerPort: 8080,
            task: customerTask,
            desiredCount: 1,
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        // The produce service(internal only)
        { listenerPort: 9090, task: productTask, desiredCount: 1, accessibility: main_1.LoadBalancerAccessibility.INTERNAL_ONLY },
    ],
    route53Ops: {
        zoneName,
        externalAlbRecordName,
        internalAlbRecordName,
    },
});
// create a dummy sg
const dummySg = new ec2.SecurityGroup(stack, 'DummySG', {
    vpc: svc.vpc,
});
// allow all traffic from dummy sg to all the services
for (let i = 0; i < svc.service.length; i++) {
    svc.service[i].connections.allowFrom(dummySg, ec2.Port.allTraffic());
}
//# sourceMappingURL=data:application/json;base64,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