"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Monitoring = void 0;
const cw = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const cdk_watchful_1 = require("cdk-watchful");
const web_canary_1 = require("./web-canary");
/**
 * Construct hub monitoring.
 *
 * This construct uses cdk-watchful to automatically generate a dashboard and a
 * set of standard alarms for common resources.
 *
 * Components should use the APIs of this module to add watchful monitors,
 * and add canaries and alarms as needed.
 */
class Monitoring extends core_1.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        this.alarmActions = props.alarmActions;
        this.watchful = new cdk_watchful_1.Watchful(this, 'Watchful', {
            // alarms that come from watchful are all considered normal severity
            alarmActionArns: ((_a = this.alarmActions) === null || _a === void 0 ? void 0 : _a.normalSeverity) ? [this.alarmActions.normalSeverity] : [],
            alarmActions: ((_b = this.alarmActions) === null || _b === void 0 ? void 0 : _b.normalSeverityAction) ? [this.alarmActions.normalSeverityAction] : [],
        });
        this.highSeverityDashboard = new cw.Dashboard(this, 'HighSeverityDashboard');
    }
    /**
     * Adds a high-severity alarm. If this alarm goes off, the action specified in `highSeverityAlarmActionArn`
     * @param alarm
     */
    addHighSeverityAlarm(title, alarm) {
        var _a, _b;
        const highSeverityActionArn = (_a = this.alarmActions) === null || _a === void 0 ? void 0 : _a.highSeverity;
        if (highSeverityActionArn) {
            alarm.addAlarmAction({
                bind: () => ({ alarmActionArn: highSeverityActionArn }),
            });
        }
        const highSeverityAction = (_b = this.alarmActions) === null || _b === void 0 ? void 0 : _b.highSeverityAction;
        if (highSeverityAction) {
            alarm.addAlarmAction(highSeverityAction);
        }
        this.highSeverityDashboard.addWidgets(new cw.AlarmWidget({
            alarm,
            title,
            width: 24,
        }));
    }
    addLowSeverityAlarm(_title, alarm) {
        var _a, _b;
        const normalSeverityActionArn = (_a = this.alarmActions) === null || _a === void 0 ? void 0 : _a.normalSeverity;
        if (normalSeverityActionArn) {
            alarm.addAlarmAction({ bind: () => ({ alarmActionArn: normalSeverityActionArn }) });
        }
        const normalSeverityAction = (_b = this.alarmActions) === null || _b === void 0 ? void 0 : _b.normalSeverityAction;
        if (normalSeverityAction) {
            alarm.addAlarmAction(normalSeverityAction);
        }
    }
    /**
     * Adds a canary that pings a certain URL and raises an alarm in case the URL
     * responds with an error over 80% of the times.
     *
     * Canary failures are always considered high-severity (at this point).
     *
     * @param name The name of the canary
     * @param url The URL to ping
     */
    addWebCanary(name, url) {
        const canary = new web_canary_1.WebCanary(this, `WebCanary${name.replace(/[^A-Z0-9]/ig, '')}`, {
            url,
            displayName: name,
        });
        this.addHighSeverityAlarm(`${name} Canary`, canary.alarm);
    }
}
exports.Monitoring = Monitoring;
//# sourceMappingURL=data:application/json;base64,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