import * as cloudfront from '@aws-cdk/aws-cloudfront';
import * as s3 from '@aws-cdk/aws-s3';
import { Construct } from '@aws-cdk/core';
import { Domain } from '../api';
import { PackageStats } from '../backend/package-stats';
import { Monitoring } from '../monitoring';
import { PreloadFile } from '../preload-file';
import { WebappConfigProps } from './config';
export interface PackageLinkConfig {
    /**
     * The name of the link, appears before the ":" on the website
     */
    readonly linkLabel: string;
    /**
     * The location of the value inside the constructHub.packageLinks
     * key of a module's package.json
     */
    readonly configKey: string;
    /**
     * optional text to display as the hyperlink text
     *
     * @default the url of the link
     */
    readonly linkText?: string;
    /**
     * allowList of domains for this link
     *
     * @default all domains allowed
     */
    readonly allowedDomains?: string[];
}
/**
 * Configuration for packages to feature on the home page.
 */
export interface FeaturedPackages {
    /**
     * Grouped sections of packages on the homepage.
     */
    readonly sections: FeaturedPackagesSection[];
}
/**
 * Customization options for one section of the home page.
 */
export interface FeaturedPackagesSection {
    /**
     * The name of the section (displayed as a header).
     */
    readonly name: string;
    /**
     * Show the N most recently updated packages in this section.
     * Cannot be used with `showPackages`.
     */
    readonly showLastUpdated?: number;
    /**
     * Show an explicit list of packages.
     * Cannot be used with `showLastUpdated`.
     */
    readonly showPackages?: FeaturedPackagesDetail[];
}
/**
 * Customization options for a specific package on the home page.
 */
export interface FeaturedPackagesDetail {
    /**
     * The name of the package.
     */
    readonly name: string;
    /**
     * An additional comment to include with the package.
     */
    readonly comment?: string;
}
/**
 * Enable/disable features for the web app.
 */
export interface FeatureFlags {
    readonly homeRedesign?: boolean;
    readonly searchRedesign?: boolean;
    [key: string]: any;
}
/**
 * A category of packages.
 */
export interface Category {
    /**
     * The title on the category button as it appears in the Construct Hub home page.
     */
    readonly title: string;
    /**
     * The URL that this category links to. This is the full path to the link that
     * this category button will have. You can use any query options such as
     * `?keywords=`, `?q=`, or a combination thereof.
     *
     * @example "/search?keywords=monitoring"
     */
    readonly url: string;
}
export interface WebAppProps extends WebappConfigProps {
    /**
     * Connect to a domain.
     * @default - uses the default CloudFront domain.
     */
    readonly domain?: Domain;
    /**
     * Monitoring system.
     */
    readonly monitoring: Monitoring;
    /**
     * The bucket containing package data.
     */
    readonly packageData: s3.Bucket;
    /**
     * Manages the `stats.json` file object.
     */
    readonly packageStats?: PackageStats;
    /**
     * JavaScript file which will be loaded before the webapp
     */
    readonly preloadScript?: PreloadFile;
}
export declare class WebApp extends Construct {
    readonly baseUrl: string;
    readonly bucket: s3.Bucket;
    readonly distribution: cloudfront.Distribution;
    constructor(scope: Construct, id: string, props: WebAppProps);
}
