"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlpsSpecRestApi = void 0;
const fs = require("fs");
const apigw = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class AlpsSpecRestApi extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.operationIdLambdaMapping = props === null || props === void 0 ? void 0 : props.operationIdLambdaMapping;
        const apiRole = new iam.Role(scope, 'apiRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        apiRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['lambda:InvokeFunction'],
        }));
        const specFile = props.alpsSpecFile;
        // const specFile = 'src/todo-alps.yaml';
        // convert ALPS yaml to OAS JSON. WORKAROUND as currently alps-unified is an js cli tool
        let oasSpecJSON = unified(specFile);
        const region = cdk.Stack.of(this).region;
        const accountId = cdk.Stack.of(this).account;
        const endpointsJson = oasSpecJSON.paths;
        for (const endpoint of Object.keys(endpointsJson)) {
            console.log(`endpoint: ${JSON.stringify(endpoint)}`);
            for (const method of Object.values(endpointsJson[endpoint])) {
                console.log(`method ${JSON.stringify(method)}`);
                // validate OpenApi Spec
                const methodKey = Object.keys(method)[0];
                if (method.operationId === undefined || method.operationId === null) {
                    throw new Error(`Endpoint ${endpoint} with Method ${methodKey}`);
                }
                let methodExt = method;
                let lambdaName = method.operationId;
                // override lambdaName if specified in operationIdLambdaMapping
                if (this.operationIdLambdaMapping && this.operationIdLambdaMapping[method.operationId] !== undefined) {
                    lambdaName = this.operationIdLambdaMapping[method.operationId];
                }
                methodExt['x-amazon-apigateway-integration'] = {
                    uri: `arn:aws:apigateway:${region}:lambda:path/2015-03-31/functions/arn:aws:lambda:${region}:${accountId}:function:${lambdaName}/invocations`,
                    passthroughBehavior: 'when_no_match',
                    httpMethod: 'POST',
                    type: 'aws_proxy',
                    credentials: apiRole.roleArn,
                };
                console.log(`methodExt: ${JSON.stringify(methodExt)}`);
            }
        }
        console.log(`specJson after edit: ${JSON.stringify(oasSpecJSON)}`);
        const api = new apigw.SpecRestApi(this, 'SpecRestApi', {
            // restApiName: 'Alps Rest Api Gw',
            apiDefinition: apigw.ApiDefinition.fromInline(oasSpecJSON),
        });
        api;
    }
}
exports.AlpsSpecRestApi = AlpsSpecRestApi;
function unified(alpSpec) {
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const { execSync } = require('child_process');
    execSync(`node_modules/unified/src/index.js -f ${alpSpec} -t oas -o tmp/oas.yaml`);
    execSync('node_modules/unified/src/index.js -f tmp/oas.yaml -t json -o tmp/oas.json');
    const tmpOasFileString = fs.readFileSync('tmp/oas.json', { encoding: 'utf-8' });
    // console.log(`tmpOasFileString: ${tmpOasFileString}`);
    return JSON.parse(tmpOasFileString);
}
;
//# sourceMappingURL=data:application/json;base64,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