"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const shared_1 = require("../shared");
const KEY_FORMAT_REGEX = new RegExp(`^${shared_1.constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`);
// Capture groups:                                                   ┗━━━━━━━━1━━━━━━━━┛  ┗━━2━━┛
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    var _a;
    const BUCKET_NAME = shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    if (!event.rebuild) {
        console.log('Loading existing catalog...');
        const data = await shared_1.aws.s3().getObject({ Bucket: BUCKET_NAME, Key: shared_1.constants.CATALOG_KEY }).promise()
            .catch((err) => err.code !== 'NoSuchKey'
            ? Promise.reject(err)
            : Promise.resolve({ /* no data */}));
        if (data.Body != null) {
            const catalog = JSON.parse(data.Body.toString('utf-8'));
            for (const info of catalog.packages) {
                if (!packages.has(info.name)) {
                    packages.set(info.name, new Map());
                }
                packages.get(info.name).set(info.major, info);
            }
        }
    }
    else {
        console.log('Rebuild requested, ignoring existing catalog...');
    }
    console.log('Listing existing objects...');
    for await (const object of relevantObjects(BUCKET_NAME)) {
        // Ensure we don't already have a more recent version tracked for this package-major.
        const [, packageName, versionStr] = object.Key.match(KEY_FORMAT_REGEX);
        const version = new semver_1.SemVer(versionStr);
        const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
        if (found != null && version.compare(found.version) <= 0) {
            console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
            continue;
        }
        console.log(`Registering ${packageName}@${version}`);
        // Donwload the tarball to inspect the `package.json` data therein.
        const data = await shared_1.aws.s3().getObject({ Bucket: BUCKET_NAME, Key: object.Key }).promise();
        const manifest = await new Promise((ok, ko) => {
            zlib_1.gunzip(Buffer.from(data.Body), (err, tar) => {
                if (err) {
                    return ko(err);
                }
                tar_stream_1.extract()
                    .on('entry', (header, stream, next) => {
                    if (header.name !== 'package/package.json') {
                        // Not the file we are looking for, skip ahead (next run-loop tick).
                        return setImmediate(next);
                    }
                    const chunks = new Array();
                    return stream
                        .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                        .once('end', () => {
                        ok(Buffer.concat(chunks));
                        next();
                    })
                        .resume();
                })
                    .once('finish', () => {
                    ko(new Error('Could not find package/package.json in tarball!'));
                })
                    .write(tar, (writeErr) => {
                    if (writeErr) {
                        ko(writeErr);
                    }
                });
            });
        });
        // Add the PackageInfo into the working set
        const metadata = JSON.parse(manifest.toString('utf-8'));
        const major = new semver_1.SemVer(metadata.version).major;
        if (!packages.has(metadata.name)) {
            packages.set(metadata.name, new Map());
        }
        packages.get(metadata.name).set(major, {
            author: metadata.author,
            description: metadata.description,
            keywords: metadata.keywords,
            languages: metadata.jsii.targets,
            license: metadata.license,
            major,
            metadata: metadata.jsii.metadata,
            name: metadata.name,
            time: metadata.time,
            version: metadata.version,
        });
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`Registered ${catalog.packages.length} package major versions`);
    // Upload the result to S3 and exit.
    return shared_1.aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: shared_1.constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'text/json',
        Metadata: {
            'Build-Process': event.rebuild ? 'FROM_SCRATCH' : 'INCREMENTAL',
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 */
async function* relevantObjects(bucket) {
    var _a, _b;
    const request = { Bucket: bucket, Prefix: shared_1.constants.STORAGE_KEY_PREFIX };
    do {
        const result = await shared_1.aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(shared_1.constants.PACKAGE_KEY_SUFFIX))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
//# sourceMappingURL=data:application/json;base64,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