"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebApp = void 0;
const path = require("path");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const r53 = require("@aws-cdk/aws-route53");
const r53targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
class WebApp extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.bucket = new s3.Bucket(this, 'WebsiteBucket', { blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL });
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: { origin: new origins.S3Origin(this.bucket) },
            domainNames: props.domain ? [props.domain.zone.zoneName] : undefined,
            certificate: props.domain ? props.domain.cert : undefined,
            defaultRootObject: 'index.html',
            errorResponses: [404, 403].map(httpStatus => ({
                httpStatus,
                responseHttpStatus: 200,
                responsePagePath: '/index.html',
            })),
        });
        const jsiiObjOrigin = new origins.HttpOrigin('awscdk.io');
        this.distribution.addBehavior('/packages/*', jsiiObjOrigin);
        this.distribution.addBehavior('/index/packages.json', jsiiObjOrigin);
        // if we use a domain, and A records with a CloudFront alias
        if (props.domain) {
            // IPv4
            new r53.ARecord(this, 'ARecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // IPv6
            new r53.AaaaRecord(this, 'AaaaRecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
        }
        // "website" contains the static react app
        const webappDir = path.join(__dirname, '..', '..', 'website');
        new s3deploy.BucketDeployment(this, 'DeployWebsite', {
            sources: [s3deploy.Source.asset(webappDir)],
            destinationBucket: this.bucket,
            distribution: this.distribution,
        });
        new core_1.CfnOutput(this, 'DomainName', {
            value: this.distribution.domainName,
            exportName: 'ConstructHubDomainName',
        });
        // add a canary that pings our home page and alarms if it returns errors.
        props.monitoring.addWebCanary('Home Page', `https://${this.distribution.domainName}`);
    }
}
exports.WebApp = WebApp;
//# sourceMappingURL=data:application/json;base64,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