"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheckType = exports.RoutingPolicy = exports.DnsRecordType = exports.Service = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const alias_target_instance_1 = require("./alias-target-instance");
const cname_instance_1 = require("./cname-instance");
const ip_instance_1 = require("./ip-instance");
const namespace_1 = require("./namespace");
const non_ip_instance_1 = require("./non-ip-instance");
const servicediscovery_generated_1 = require("./servicediscovery.generated");
class ServiceBase extends core_1.Resource {
}
/**
 * Define a CloudMap Service
 */
class Service extends ServiceBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicediscovery_ServiceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const namespaceType = props.namespace.type;
        // Validations
        if (namespaceType === namespace_1.NamespaceType.HTTP && (props.routingPolicy || props.dnsRecordType)) {
            throw new Error('Cannot specify `routingPolicy` or `dnsRecord` when using an HTTP namespace.');
        }
        if (props.healthCheck && props.customHealthCheck) {
            throw new Error('Cannot specify both `healthCheckConfig` and `healthCheckCustomConfig`.');
        }
        if (namespaceType === namespace_1.NamespaceType.DNS_PRIVATE && props.healthCheck) {
            throw new Error('Cannot specify `healthCheckConfig` for a Private DNS namespace.');
        }
        if (props.routingPolicy === RoutingPolicy.MULTIVALUE
            && props.dnsRecordType === DnsRecordType.CNAME) {
            throw new Error('Cannot use `CNAME` record when routing policy is `Multivalue`.');
        }
        // Additional validation for eventual attachment of LBs
        // The same validation happens later on during the actual attachment
        // of LBs, but we need the property for the correct configuration of
        // routingPolicy anyway, so might as well do the validation as well.
        if (props.routingPolicy === RoutingPolicy.MULTIVALUE
            && props.loadBalancer) {
            throw new Error('Cannot register loadbalancers when routing policy is `Multivalue`.');
        }
        if (props.healthCheck
            && props.healthCheck.type === HealthCheckType.TCP
            && props.healthCheck.resourcePath) {
            throw new Error('Cannot specify `resourcePath` when using a `TCP` health check.');
        }
        // Set defaults where necessary
        const routingPolicy = (props.dnsRecordType === DnsRecordType.CNAME) || props.loadBalancer
            ? RoutingPolicy.WEIGHTED
            : RoutingPolicy.MULTIVALUE;
        const dnsRecordType = props.dnsRecordType || DnsRecordType.A;
        if (props.loadBalancer
            && (!(dnsRecordType === DnsRecordType.A
                || dnsRecordType === DnsRecordType.AAAA
                || dnsRecordType === DnsRecordType.A_AAAA))) {
            throw new Error('Must support `A` or `AAAA` records to register loadbalancers.');
        }
        const dnsConfig = props.namespace.type === namespace_1.NamespaceType.HTTP
            ? undefined
            : {
                dnsRecords: renderDnsRecords(dnsRecordType, props.dnsTtl),
                namespaceId: props.namespace.namespaceId,
                routingPolicy,
            };
        const healthCheckConfigDefaults = {
            type: HealthCheckType.HTTP,
            failureThreshold: 1,
            resourcePath: props.healthCheck && props.healthCheck.type !== HealthCheckType.TCP
                ? '/'
                : undefined,
        };
        const healthCheckConfig = props.healthCheck && { ...healthCheckConfigDefaults, ...props.healthCheck };
        const healthCheckCustomConfig = props.customHealthCheck;
        // Create service
        const service = new servicediscovery_generated_1.CfnService(this, 'Resource', {
            name: props.name,
            description: props.description,
            dnsConfig,
            healthCheckConfig,
            healthCheckCustomConfig,
            namespaceId: props.namespace.namespaceId,
        });
        this.serviceName = service.attrName;
        this.serviceArn = service.attrArn;
        this.serviceId = service.attrId;
        this.namespace = props.namespace;
        this.dnsRecordType = dnsRecordType;
        this.routingPolicy = routingPolicy;
    }
    static fromServiceAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicediscovery_ServiceAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromServiceAttributes);
            }
            throw error;
        }
        class Import extends ServiceBase {
            constructor() {
                super(...arguments);
                this.namespace = attrs.namespace;
                this.serviceId = attrs.serviceId;
                this.serviceArn = attrs.serviceArn;
                this.dnsRecordType = attrs.dnsRecordType;
                this.routingPolicy = attrs.routingPolicy;
                this.serviceName = attrs.serviceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Registers an ELB as a new instance with unique name instanceId in this service.
     */
    registerLoadBalancer(id, loadBalancer, customAttributes) {
        return new alias_target_instance_1.AliasTargetInstance(this, id, {
            service: this,
            dnsName: loadBalancer.loadBalancerDnsName,
            customAttributes,
        });
    }
    /**
     * Registers a resource that is accessible using values other than an IP address or a domain name (CNAME).
     */
    registerNonIpInstance(id, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicediscovery_NonIpInstanceBaseProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.registerNonIpInstance);
            }
            throw error;
        }
        return new non_ip_instance_1.NonIpInstance(this, id, {
            service: this,
            ...props,
        });
    }
    /**
     * Registers a resource that is accessible using an IP address.
     */
    registerIpInstance(id, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicediscovery_IpInstanceBaseProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.registerIpInstance);
            }
            throw error;
        }
        return new ip_instance_1.IpInstance(this, id, {
            service: this,
            ...props,
        });
    }
    /**
     * Registers a resource that is accessible using a CNAME.
     */
    registerCnameInstance(id, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicediscovery_CnameInstanceBaseProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.registerCnameInstance);
            }
            throw error;
        }
        return new cname_instance_1.CnameInstance(this, id, {
            service: this,
            ...props,
        });
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "@aws-cdk/aws-servicediscovery.Service", version: "1.157.0" };
function renderDnsRecords(dnsRecordType, dnsTtl = core_1.Duration.minutes(1)) {
    const ttl = dnsTtl.toSeconds();
    if (dnsRecordType === DnsRecordType.A_AAAA) {
        return [{
                type: DnsRecordType.A,
                ttl,
            }, {
                type: DnsRecordType.AAAA,
                ttl,
            }];
    }
    else {
        return [{ type: dnsRecordType, ttl }];
    }
}
var DnsRecordType;
(function (DnsRecordType) {
    /**
     * An A record
     */
    DnsRecordType["A"] = "A";
    /**
     * An AAAA record
     */
    DnsRecordType["AAAA"] = "AAAA";
    /**
     * Both an A and AAAA record
     */
    DnsRecordType["A_AAAA"] = "A, AAAA";
    /**
     * A Srv record
     */
    DnsRecordType["SRV"] = "SRV";
    /**
     * A CNAME record
     */
    DnsRecordType["CNAME"] = "CNAME";
})(DnsRecordType = exports.DnsRecordType || (exports.DnsRecordType = {}));
var RoutingPolicy;
(function (RoutingPolicy) {
    /**
     * Route 53 returns the applicable value from one randomly selected instance from among the instances that you
     * registered using the same service.
     */
    RoutingPolicy["WEIGHTED"] = "WEIGHTED";
    /**
     * If you define a health check for the service and the health check is healthy, Route 53 returns the applicable value
     * for up to eight instances.
     */
    RoutingPolicy["MULTIVALUE"] = "MULTIVALUE";
})(RoutingPolicy = exports.RoutingPolicy || (exports.RoutingPolicy = {}));
var HealthCheckType;
(function (HealthCheckType) {
    /**
     * Route 53 tries to establish a TCP connection. If successful, Route 53 submits an HTTP request and waits for an HTTP
     * status code of 200 or greater and less than 400.
     */
    HealthCheckType["HTTP"] = "HTTP";
    /**
     * Route 53 tries to establish a TCP connection. If successful, Route 53 submits an HTTPS request and waits for an
     * HTTP status code of 200 or greater and less than 400.  If you specify HTTPS for the value of Type, the endpoint
     * must support TLS v1.0 or later.
     */
    HealthCheckType["HTTPS"] = "HTTPS";
    /**
     * Route 53 tries to establish a TCP connection.
     * If you specify TCP for Type, don't specify a value for ResourcePath.
     */
    HealthCheckType["TCP"] = "TCP";
})(HealthCheckType = exports.HealthCheckType || (exports.HealthCheckType = {}));
//# sourceMappingURL=data:application/json;base64,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