from django.apps import apps
from django.conf import settings
from django.core.management.commands.makemigrations import Command as OrigCommand
from django.db.migrations.loader import MigrationLoader
from pathlib import Path


LATEST_MIGRATIONS_DIR_NAME = 'latest_migrations'


README_TEMPLATE = """\
The contents of this directory is autogenerated and shouldn't be modified
manually. It stores the latest migration of each of the installed applications.
When two git branches are merged, and they contain different sets of migrations,
a merge migration has to be applied on top of them. This is enforced by putting
the name of the latest migration to the file in this directory, and if two
branches have different migrations, there will be a conflict in this file which
will stop git from doing an unsafe merge.
"""


APP_FILE_TEMPLATE = """\
# This file is autogenerated by makemigrations.
# If you have a merge conflict in this file, it means you need to run:
#     manage.py makemigrations --merge
# in order to resolve the conflict between migrations.
{migration_name}
"""


class Command(OrigCommand):
    """Cause a merge conflict in git if a merge migration needs to be created."""

    def handle(self, *args, **kwargs):
        super().handle(*args, **kwargs)

        loader = MigrationLoader(None, ignore_no_migrations=True)

        # Find the latest migration of every application.
        latest_migrations = {}
        for migration in loader.disk_migrations.values():
            if latest_migrations.get(migration.app_label, '') < migration.name:
                latest_migrations[migration.app_label] = migration.name

        # Create the directory for storing latest migration names.
        latest_migrations_dir = Path(settings.BASE_DIR, LATEST_MIGRATIONS_DIR_NAME)
        latest_migrations_dir.mkdir(exist_ok=True)

        # Clear the directory to remove applications not installed anymore,
        # and create a README explaining what's happening here.
        for lock_file in latest_migrations_dir.iterdir():
            if lock_file.is_file():
                lock_file.unlink()
        with (latest_migrations_dir / 'README').open('wt') as readme_file:
            readme_file.write(README_TEMPLATE)

        # Store the latest migrations information to the files.
        for app in apps.get_app_configs():
            # Use Python module name as an identifier here, as labels are
            # guaranteed to be unique throughout INSTALLED_APPS, but they don't
            # protect from the situation where one replaces some application
            # with a different one that has the same label.
            with (latest_migrations_dir / app.name).open('wt') as lock_file:
                lock_file.write(APP_FILE_TEMPLATE.format(migration_name=latest_migrations.get(app.label, "")))
