# fast-bitrix24
Высокоуровневый API для Python 3.7+ для быстрого получения данных от Битрикс24 через REST API.

## Основная функциональность
### Удобство кода
- Высокоуровневые списочные методы для сокращения количества необходимого кода. Большинство операций занимают только одну строку кода. Обработка параллельных запросов, упаковка запросов в батчи и многое другое убрано "под капот".
- Позволяет задавать параметры запроса именно в таком виде, как они приведены в [документации к Bitrix24 REST API](https://dev.1c-bitrix.ru/rest_help/index.php).
- Выполнение запросов автоматически сопровождается прогресс-баром из пакета `tqdm`, иллюстрирующим не только количество обработанных элементов, но и прошедшее и оставшееся время выполнения запроса.

### Высокая скорость обмена данными
- На больших списках скорость обмена данными с сервером достигает тысяч элементов в секунду.
- Использование асинхронных запросов через `asyncio` и `aiohttp` позволяет экономить время, делая несколько запросов к серверу параллельно.
- Автоматическая упаковка запросов в батчи сокращает количество требуемых запросов к серверу и ускоряет обмен данными.

### Избежание отказов сервера
- Скорость отправки запросов к серверу Битрикс контролируется для избежания ошибки HTTP `503 Service unanavilable`
- Размера батча контролируется для избежания ошибки HTTP `414 URI too long`

## Начало
Установите модуль через `pip`:
```shell
pip install fast_bitrix24
```

Далее в python:

```python
from fast_bitrix24 import Bitrix

# замените на ваш вебхук для доступа к Bitrix24
webhook = "https://your_domain.bitrix24.ru/rest/1/your_code/"
b = Bitrix(webhook)
```

Методы полученного объекта `b` в дальнейшем используются для взаимодействия с сервером Битрикс24.

## Использование

### `get_all()`

Чтобы получить полностью список сущностей, используйте метод ```get_all()```:

```python
# список пользователей
users = b.get_all('user.get')
```

Метод `get_all()` возвращает список, где каждый элемент списка является словарем, описывающим одну сущность из запрощенного списка.

Вы также можете использовать параметр ```params```, чтобы кастомизировать запрос:

```python
# список сделок в работе, включая пользовательские поля
deals = b.get_all('crm.deal.get', params={
    'select': ['*', 'UF_*'],
    'filter': {'CLOSED': 'N'}
})
```

### `get_by_ID()`
Если у вас есть список ID сущностей, то вы можете получить их при помощи метода `get_by_ID()`:

```python
'''
получим список всех контактов, привзяанных к сдлекам, в виде
[
    (ID сделки1, [контакт1, контакт2, ...]), 
    (ID сделки2, [контакт1, контакт2, ...]), 
    ...
]
'''

contacts = b.get_by_ID('crm.deal.contact.item.get',
    [d['ID'] for d in deals])
```
Метод `get_by_ID()` возвращает список кортежей вида `(ID, result)`, где `result` - ответ сервера относительно этого `ID`.

### `call()`
Чтобы создавать, изменять или удалять список сущностей, используйте метод `call()`:

```python
# вставим в начало названия всех сделок их ID
tasks = [
    {
        'ID': d['ID'],
        'NAME': f'{d["ID"]}: {d["NAME"]}'
    }
    for d in deals
]

b.call('crm.deal.update', tasks)
```
Метод `call()` возвращает список ответов сервера по каждому элементу переданного списка. 

## Подробнее о классе `Bitrix`
Объект класса `Bitrix ` создаётся, чтобы через него выполнять все запросы к серверу Битрикс24. Внутри объекта ведётся учёт скорости отправки запросов к серверу, поэтому важно, чтобы все запросы приложения отправлялись из одного экземпляра `Bitrix`.

Под капотом все методы класса используют параллельные запросы и автоматическое построение батчей, чтобы ускорить получение данных. 

Все методы будут поднимать исключения класса `aiohttp.ClientError`, если сервер Битрикс вернул HTTP-ошибку, и `RuntimeError`, если код ответа был `200`, но ошибка сдержалась в теле ответа сервера (_да, иногда бывает и такое!_).

### Метод ` __init__(self, webhook: str, custom_pool_size: int = 50, requests_per_second: float = 2, cautious: bool = False, autobatch: bool = True, verbose: bool = True):`
Создаёт экземпляр объекта `Bitrix`.

#### Параметры 
* `webhook: str` - URL вебхука, полученного от сервера Битрикс.
    
* `custom_pool_size: int = 50` - размер пула запросов. По умолчанию 50 запросов - это размер, указанный в официальной документации Битрикс24 на июль 2020 г. (https://dev.1c-bitrix.ru/rest_help/rest_sum/index.php)

* `requests_per_second: float = 2` - скорость отправки запросов. По умолчанию 2 запроса в секунду - предельная скорость, согласно официальной документации.

* `cautious: bool = False` - стартовать, считая, что пул запросов уже исчерпан, и нужно контролировать скорость запросов с первого запроса.
    
* `autobatch: bool = True` - автоматически объединять списки запросов в батчи для ускорения обмена с сервером.

* `verbose: bool = True` - показывать прогрессбар при выполнении запроса.

### Метод `get_all(self, method: str, params: dict = None)`
Получить полный список сущностей по запросу method.

`get_all()` самостоятельно обратывает постраничные ответы сервера, чтобы вернуть полный список.

#### Параметры
* `method: str` - метод REST API для запроса к серверу.

* `params: dict` - параметры для передачи методу. Используется именно тот формат, который указан в документации к REST API Битрикс24.

Возвращает полный список сущностей, имеющихся на сервере, согласно заданным методу и параметрам.

### Метод `get_by_ID(self, method: str, ID_list: list, params: dict = None)`
Получить список сущностей по запросу `method` и списку ID.

Используется для случаев, когда нужны не все сущности, имеющиеся в базе, а конкретный список поименованных ID, либо в REST API отсутствует способ  получения сущностей одним вызовом.

Например, чтобы получить все контакты, привязанные к сделкам в работе, нужно выполнить следующий код:

```python
deals = b.get_all('crm.deal.get', params={
    'filter': {'CLOSED': 'N'}
})

contacts = b.get_by_ID('crm.deal.contact.item.get',
    [d['ID'] for d in deals])
```

#### Параметры

* `method: str` - метод REST API для запроса к серверу.

* `ID_list: list` - список ID, в отношении которых будут выполняться запросы.

* `params: dict` - параметры для передачи методу. Используется именно тот формат, который указан в документации к REST API Битрикс24.

Возвращает список кортежей вида:

```
[
    (ID, <результат запроса>), 
    (ID, <результат запроса>), 
    ...
]
```

Вторым элементом каждого кортежа будет результат выполнения запроса относительно этого ID. Это может быть, например, список связанных сущностей или пустой список, если не найдено ни одной привязанной сущности.

### Метод `call(self, method: str, item_list: list of dicts)`

Вызвать метод REST API по списку.

#### Параметры
* `method: str` - метод REST API

* `item_list: list of dicts` - список параметров вызываемого метода

`call()` вызывает `method`, последовательно подставляя в параметры запроса все элементы `item_list` и возвращает список ответов сервера для каждого из отправленных запросов.

### Метод `set_requests_per_second(self, requests_per_second: float)`

Установить скорость запросов к серверу Битрикса, равную 
`requests_per_second`. Может использоваться для понижения скорости
запросов припроведении большого количества операций, нагружающих сервер
(например, создание лидов или сделок), из-за чего он может возвращать
ошибку `500 Internal Server Error`.

Обратите внимание, что класс `Bitrix` замеряет время, прошедшее между
вызовами его списочных методов, чтобы определить, какое количество запросов
"накопилось" в пуле запросов с момента последнего вызова. Для этого расчета
используется как раз установленная скорость в запросах в секунду.
Изменение этого параметра соответственно изменит этот расчет - понижение
`requests_per_second` понизит расчетную скорость накопления запросов в пуле.

#### Параметры
- `requests_per_second` - новая скорость в запросах в секунду

Возвращает `None`.

## Как связаться с автором
- e-mail: leshchenko@gmail.com
- telegram: https://t.me/leshchenko1979