# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['reactant', 'reactant.orm', 'reactant.renderer']

package_data = \
{'': ['*'], 'reactant': ['templates/*']}

install_requires = \
['Jinja2>=3.0.1,<4.0.0',
 'black>=21.8b0,<22.0',
 'click>=8.0.1,<9.0.0',
 'pydantic>=1.8.2,<2.0.0']

entry_points = \
{'console_scripts': ['reactant = reactant.run:runner']}

setup_kwargs = {
    'name': 'reactant',
    'version': '0.5.0',
    'description': 'Generate code for models, views, and urls based on Python type annotations.',
    'long_description': '<p align="center">\n    <a href="https://pypi.org/project/reactant">\n        <img width="1200" src="https://raw.githubusercontent.com/neil-vqa/reactant/main/reactant-logo-banner.png">\n    </a>\n</p>\n\nGenerate code for *models, views, and urls* based on Python type annotations. Powered by [pydantic](https://github.com/samuelcolvin/pydantic/). Influenced by [SQLModel](https://github.com/tiangolo/sqlmodel).\n\n*reactant* aims to give usable and sensible code defaults. It does **not enforce** a particular application structure. Instead, it follows the default/minimal/common structure of the supported frameworks, and the developer has the freedom of changing it (if necessary) to fit it to their application. Contributions are warmly welcomed if you believe a particular structure is widely used and can benefit from code generation.\n\n## Supported Frameworks\n\n*reactant* currently generates code for the following:\n\n**Django REST** (Django 3+, DRF 3.12+)\n\n(in Django\'s *default* project structure i.e. by *apps*)\n\n- [X] models\n- [X] views (class-based API views, filename=*views_class.py*)\n- [X] views (function-based API views, filename=*views_func.py*)\n- [X] serializers\n- [X] urls (from class-based API views, filename=*urls_class.py*)\n- [X] urls (from function-based API views, filename=*urls_func.py*)\n\n**Flask** (Flask 2+)\n\n(package structure, for building REST APIs)\n\n- [ ] views\n\n**SQLAlchemy** (SQLAlchemy 1.4+)\n\n- [X] classes mapped by Declarative Mapping\n- [X] classes mapped by Classical Mapping\n\n**Peewee** (Peewee 3.14+)\n\n- [X] models\n\n## Installation\n\n```cli\npip install reactant\n```\n\nNOTE: *reactant* isn\'t shipped with the supported frameworks. You MUST install `django`, `sqlalchemy`, `peewee`, or `flask` separately (be it in a virtual environment or globally).\n\n## Get Started\n\nCreate *reactant* models by inheriting from `Reactant` subclasses: `DjangoORM`, `SQLAlchemyORM`, `PeeweeORM`. The example below uses `DjangoORM`. Your choice will determine what code and files will be generated.\n\n```python\n# generate.py\n\nfrom typing import Optional\nfrom reactant import DjangoORM, Field, generate\nfrom datetime import date\n\n\nclass RocketEngine(DjangoORM):\n    name: str = Field(max_length=32, title="engine_name") # adding additional arguments\n    manufacturer: Optional[str]\n    power_cycle: Optional[str] = "gas-generator" # setting a default\n    thrust_weight_ratio: Optional[int] = None\n\n\nclass LaunchVehicle(DjangoORM):\n    name: str = Field(max_length=32)\n    country: str = Field("USA", max_length=32) # setting a default in the Field function \n    status: str\n    total_launches: Optional[int]\n    first_flight: Optional[date]\n    engine: str = Field(foreign_key="RocketEngine") # specifying a relationship field\n\n# Don\'t forget this block.\nif __name__ == "__main__":\n    generate()\n\n```\n\nYou can give ORM-specific arguments to the `Field` function. For more details, please read the [The Field function](#the-field-function) section.\n\nDon\'t forget `generate()`. Run by specifying the file: `reactant generate.py`\n\nTip: You can control what code and files will be generated by passing keyword arguments to `generate()`. Read the [The generate function](#the-generate-function) section for the details. \n\n```cli\n$ reactant generate.py\n\nRunning generate.py\nFound 2 Django reactants.\nDjango models.py finished rendering.\nDjango serializers.py finished rendering.\nDjango views_class.py finished rendering.\nDjango urls_class.py finished rendering.\nDjango views_func.py finished rendering.\nDjango urls_func.py finished rendering.\nSuccess! Please check "reactant_products/django" directory.\n```\n\n**BOOM!** With just the above code, the models, views, serializers, and urls (the *products*, for Django atleast) are generated. See images of the code below.\n\n## Sample Code Generated\n\n### Django REST\n\n![dj_shot_01](screenshots/dj_01n.png)\n![dj_shot_02](screenshots/dj_02n_2.png)\n![dj_shot_03](screenshots/dj_03n_2.png)\n![dj_shot_04](screenshots/dj_04n.png)\n\n## The Field function\n\nThe *reactant* `Field` function comes from pydantic. It can be used to provide extra information about the field. Currently though, *reactant* `Field` can only support the following pydantic native arguments:\n\n- `default` (the positional argument)\n- `max_length`\n- `title`\n\nOther than the listed native arguments, *reactant* `Field` can accept arguments that are specific to an ORM and filters out those that are not supported by the same ORM. For example if you are generating Django files, Django ORM model fields can accept an `error_messages` argument. You can give this argument to the *reactant* `Field` and the generated Django models will include this argument. However, if you give made-up arguments or arguments that are only valid for other ORMs, the generated Django models will not include those.\n\n### Specifying model relationships\n\nRelationship fields can be specified by giving the *reactant* `Field` special arguments.\n\n- `foreign_key` - identifies a foreign key\n- `many_key` - identifies a many-to-many\n- `one_key` - identifies a one-to-one\n\nThese behave differently for each framework. The [Get Started](#get-started) section shows an example of specifying a relationship.\n\n**Django REST**: Models inheriting from `DjangoORM` can use `foreign_key`, `many_key`, and `one_key` to establish relationships. The [Get Started](#get-started) section shows a `DjangoORM` example.\n\n**SQLAlchemy**: *reactant* aims not to introduce anything new except having knowledge of type hints and pydantic. It was hard to capture and generate SQLAlchemy relationships without introducing new *reactant* concepts due to SQLAlchemy\'s flexibility. Nevertheless, relationship code are still generated and you are given options as to what relationship you intend to build.\n\nCurrently, only the `foreign_key` parameter is available for establishing relationships. You may put the `Field` function with this parameter to any of your classes since the generated code will provide options for your intended relationship: One-to-Many, Many-to-One, One-to-One. Using `SQLAlchemyORM` for the reactant models in [Get Started](#get-started) section, the following code is generated:\n\n![sqla_rel_01](screenshots/sqla_01n2.png)\n\n**Peewee**: *reactant* models with `PeeweeORM` can use `foreign_key` parameter.\n\n## The generate function\n\nThe `generate` function can accept keyword arguments to control the behavior of generating code and files. Read below to learn what each supported framework accepts.\n\n### Django REST\n\nBy default, both function-based and class-based views are generated. To switch OFF either of the two, pass `class_based=False` or `function_based=False`. \n\n## Development\n\nThe project uses Poetry to package and manage dependencies.\n\n```cli\n(venv)$ poetry install\n```\n\nRun tests.\n```cli\npytest\n```\n\n## License\n\nMIT License. For more information and legal terms, see the LICENSE file.\n',
    'author': 'Neil Van',
    'author_email': 'nvq.alino@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
