"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addProxyMethodToApiResource = exports.RegionalRestApi = exports.GlobalRestApi = exports.RegionalLambdaRestApi = exports.GlobalLambdaRestApi = void 0;
// Imports
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const apiDefaults = require("./apigateway-defaults");
const cloudwatch_log_group_defaults_1 = require("./cloudwatch-log-group-defaults");
const utils_1 = require("./utils");
/**
 * Create and configures access logging for API Gateway resources.
 * @param scope - the construct to which the access logging capabilities should be attached to.
 * @param _api - an existing api.RestApi or api.LambdaRestApi.
 */
function configureCloudwatchRoleForApi(scope, _api) {
    var _a;
    // Setup the IAM Role for API Gateway CloudWatch access
    const restApiCloudwatchRole = new iam.Role(scope, 'LambdaRestApiCloudWatchRole', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        inlinePolicies: {
            LambdaRestApiCloudWatchRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:DescribeLogGroups',
                            'logs:DescribeLogStreams',
                            'logs:PutLogEvents',
                            'logs:GetLogEvents',
                            'logs:FilterLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
                    })]
            })
        }
    });
    // Create and configure AWS::ApiGateway::Account with CloudWatch Role for ApiGateway
    const CfnApi = _api.node.findChild('Resource');
    const cfnAccount = new api.CfnAccount(scope, 'LambdaRestApiAccount', {
        cloudWatchRoleArn: restApiCloudwatchRole.roleArn
    });
    cfnAccount.addDependsOn(CfnApi);
    // Suppress Cfn Nag warning for APIG
    const deployment = (_a = _api.latestDeployment) === null || _a === void 0 ? void 0 : _a.node.findChild('Resource');
    deployment.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W45',
                    reason: `ApiGateway has AccessLogging enabled in AWS::ApiGateway::Stage resource, but cfn_nag checkes for it in AWS::ApiGateway::Deployment resource`
                }]
        }
    };
    // Return the CW Role
    return restApiCloudwatchRole;
}
/**
 * Creates and configures an api.LambdaRestApi.
 * @param scope - the construct to which the LambdaRestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the LambdaRestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureLambdaRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API object
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, apiGatewayProps);
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    const cwRole = configureCloudwatchRoleForApi(scope, _api);
    let usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        const extraParams = { apiKey: _api.addApiKey('ApiKey') };
        usagePlanProps = Object.assign(usagePlanProps, extraParams);
    }
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', usagePlanProps);
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Creates and configures an api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the RestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, apiGatewayProps);
        _api = new api.RestApi(scope, 'RestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.RestApi(scope, 'RestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    const cwRole = configureCloudwatchRoleForApi(scope, _api);
    let usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        const extraParams = { apiKey: _api.addApiKey('ApiKey') };
        usagePlanProps = Object.assign(usagePlanProps, extraParams);
    }
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', usagePlanProps);
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Builds and returns a global api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = new logs.LogGroup(scope, 'ApiAccessLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    const defaultProps = apiDefaults.DefaultGlobalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalLambdaRestApi = GlobalLambdaRestApi;
/**
 * Builds and returns a regional api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = new logs.LogGroup(scope, 'ApiAccessLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    const defaultProps = apiDefaults.DefaultRegionalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalLambdaRestApi = RegionalLambdaRestApi;
/**
 * Builds and returns a standard api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalRestApi(scope, apiGatewayProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = new logs.LogGroup(scope, 'ApiAccessLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    const defaultProps = apiDefaults.DefaultGlobalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalRestApi = GlobalRestApi;
/**
 * Builds and returns a Regional api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalRestApi(scope, apiGatewayProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = new logs.LogGroup(scope, 'ApiAccessLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    const defaultProps = apiDefaults.DefaultRegionalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalRestApi = RegionalRestApi;
function addProxyMethodToApiResource(params) {
    let baseProps = {
        service: params.service,
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            credentialsRole: params.apiGatewayRole,
            requestParameters: {
                "integration.request.header.Content-Type": params.contentType ? params.contentType : "'application/json'"
            },
            requestTemplates: {
                "application/json": params.requestTemplate
            },
            integrationResponses: [
                {
                    statusCode: "200"
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        }
    };
    let extraProps;
    if (params.action) {
        extraProps = {
            action: params.action
        };
    }
    else if (params.path) {
        extraProps = {
            path: params.path
        };
    }
    else {
        throw Error('Either action or path is required');
    }
    // Setup the API Gateway AWS Integration
    baseProps = Object.assign(baseProps, extraProps);
    let apiGatewayIntegration;
    if (params.awsIntegrationProps) {
        const overridenProps = utils_1.overrideProps(baseProps, params.awsIntegrationProps);
        apiGatewayIntegration = new api.AwsIntegration(overridenProps);
    }
    else {
        apiGatewayIntegration = new api.AwsIntegration(baseProps);
    }
    const defaultMethodOptions = {
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ],
        requestValidator: params.requestValidator,
        requestModels: params.requestModel
    };
    let apiMethod;
    // Setup the API Gateway method
    if (params.methodOptions) {
        const overridenProps = utils_1.overrideProps(defaultMethodOptions, params.methodOptions);
        apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, overridenProps);
    }
    else {
        apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, defaultMethodOptions);
    }
    return apiMethod;
}
exports.addProxyMethodToApiResource = addProxyMethodToApiResource;
//# sourceMappingURL=data:application/json;base64,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