"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultCloudFrontDisributionForMediaStoreProps = exports.DefaultCloudFrontWebDistributionForS3Props = exports.DefaultCloudFrontWebDistributionForApiGatewayProps = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const cdk = require("@aws-cdk/core");
function DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, setHttpSecurityHeaders, edgeLambda) {
    const apiEndPointUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split("://", apiEndPoint.url));
    const apiEndPointDomainName = cdk.Fn.select(0, cdk.Fn.split("/", apiEndPointUrlWithoutProtocol));
    if (setHttpSecurityHeaders) {
        return {
            defaultBehavior: {
                origin: new origins.HttpOrigin(apiEndPointDomainName, {
                    originPath: `/${apiEndPoint.deploymentStage.stageName}`
                }),
                edgeLambdas: [
                    {
                        eventType: cloudfront.LambdaEdgeEventType.ORIGIN_RESPONSE,
                        functionVersion: edgeLambda
                    }
                ],
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
            },
            enableLogging: true,
            logBucket: loggingBucket,
        };
    }
    else {
        return {
            defaultBehavior: {
                origin: new origins.HttpOrigin(apiEndPointDomainName, {
                    originPath: `/${apiEndPoint.deploymentStage.stageName}`
                }),
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
            },
            enableLogging: true,
            logBucket: loggingBucket,
        };
    }
}
exports.DefaultCloudFrontWebDistributionForApiGatewayProps = DefaultCloudFrontWebDistributionForApiGatewayProps;
function DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, setHttpSecurityHeaders, edgeLambda) {
    if (setHttpSecurityHeaders) {
        return {
            defaultBehavior: {
                origin: new origins.S3Origin(sourceBucket),
                edgeLambdas: [
                    {
                        eventType: cloudfront.LambdaEdgeEventType.ORIGIN_RESPONSE,
                        functionVersion: edgeLambda
                    }
                ],
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
            },
            enableLogging: true,
            logBucket: loggingBucket,
            defaultRootObject: 'index.html'
        };
    }
    else {
        return {
            defaultBehavior: {
                origin: new origins.S3Origin(sourceBucket),
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
            },
            enableLogging: true,
            logBucket: loggingBucket,
            defaultRootObject: 'index.html'
        };
    }
}
exports.DefaultCloudFrontWebDistributionForS3Props = DefaultCloudFrontWebDistributionForS3Props;
function DefaultCloudFrontDisributionForMediaStoreProps(mediastoreContainer, loggingBucket, originRequestPolicy, setHttpSecurityHeaders, customHeaders, edgeLambda) {
    const mediaStoreContainerUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split('://', mediastoreContainer.attrEndpoint));
    const mediaStoreContainerDomainName = cdk.Fn.select(0, cdk.Fn.split('/', mediaStoreContainerUrlWithoutProtocol));
    const httpOrigin = customHeaders ?
        new origins.HttpOrigin(mediaStoreContainerDomainName, { customHeaders }) :
        new origins.HttpOrigin(mediaStoreContainerDomainName);
    if (setHttpSecurityHeaders) {
        return {
            defaultBehavior: {
                origin: httpOrigin,
                edgeLambdas: [
                    {
                        eventType: cloudfront.LambdaEdgeEventType.ORIGIN_RESPONSE,
                        functionVersion: edgeLambda
                    }
                ],
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                allowedMethods: cloudfront.AllowedMethods.ALLOW_GET_HEAD_OPTIONS,
                cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS,
                originRequestPolicy
            },
            enableLogging: true,
            logBucket: loggingBucket
        };
    }
    else {
        return {
            defaultBehavior: {
                origin: httpOrigin,
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                allowedMethods: cloudfront.AllowedMethods.ALLOW_GET_HEAD_OPTIONS,
                cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS,
                originRequestPolicy
            },
            enableLogging: true,
            logBucket: loggingBucket
        };
    }
}
exports.DefaultCloudFrontDisributionForMediaStoreProps = DefaultCloudFrontDisributionForMediaStoreProps;
//# sourceMappingURL=data:application/json;base64,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