"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.flagOverriddenDefaults = void 0;
const deepdiff = require("deep-diff");
const utils_1 = require("./utils");
/**
 * Emits a warning to the console when a prescriptive default value is overridden by the user.
 * @param {object} defaultProps the prescriptive defaults for the pattern.
 * @param {object} userProps the properties provided by the user, to be compared against the defaultProps.
 */
function flagOverriddenDefaults(defaultProps, userProps) {
    // Compare the properties and return any overrides
    const overrides = findOverrides(defaultProps, userProps);
    // Emit a warning for each override
    for (let i = 0; i < ((overrides !== undefined) ? overrides.length : 0); i++) {
        const e = Object.assign(overrides[i]);
        // Determine appropriate logging granularity
        const valuesAreReadable = (checkReadability(e.lhs) &&
            checkReadability(e.rhs));
        // Format the path for readability
        const path = formatOverridePath(e.path);
        // Output
        const details = (valuesAreReadable) ? ` Default value: '${e.lhs}'. You provided: '${e.rhs}'.` : '';
        utils_1.printWarning(`An override has been provided for the property: ${path}.` + details);
    }
}
exports.flagOverriddenDefaults = flagOverriddenDefaults;
/** The prefilter function returns true for any filtered path/key that should be excluded from the diff check.
 * Any Construct Props using cdk.Duration type is not properly handled by
 * 'deep-diff' library, whenever it encounters a Duration object, it throws the exception
 * 'Argument to Intrinsic must be a plain value object', so such props are excluded from the diff check.
 */
function _prefilter(_path, _key) {
    const prefilters = ['maxRecordAge', 'expiration', 'transitionAfter', 'destination', 'timeout'];
    if (prefilters.indexOf(_key) >= 0) {
        return true;
    }
    return false;
}
/**
 * Performs a diff check of the userProps against the defaultProps to detect overridden properties.
 * @param {object} defaultProps the prescriptive defaults for the pattern.
 * @param {object} userProps the properties provided by the user, to be compared against the defaultProps.
 * @return {Array} an array containing the overridden values.
 */
function findOverrides(defaultProps, userProps) {
    const diff = deepdiff.diff(defaultProps, userProps, _prefilter);
    // Filter the results
    return (diff !== undefined) ? diff === null || diff === void 0 ? void 0 : diff.filter((e) => {
        var _a, _b;
        return (e.kind === 'E' && // only return overrides
            !((_a = e.path) === null || _a === void 0 ? void 0 : _a.includes('node')) && // stop traversing once the object graph hits the node
            !((_b = e.path) === null || _b === void 0 ? void 0 : _b.includes('bind')) // stop traversing once the object graph hits internal functions
        );
    }) : [];
}
/**
 * Converts the path object from the deep-diff module to a user-readable, bracket notation format.
 * @param {string | string[]} path either a string value or an array of strings.
 * @return {string} the formatted override path.
 */
function formatOverridePath(path) {
    return (path !== undefined && path.length > 1) ? path.toString()
        .replace(/,/g, '][')
        .replace(/\]{1}/, '')
        .replace(/$/, ']') : path;
}
/**
 * Check the readability of an input value and, in the context of the override warning service, return true if it
 * meets the established criteria. This function is used to determine whether more-detailed log output can be given.
 * @param {any} value input to be evaluated against the given criteria.
 * @return {boolean} true if the value meets the given criteria.
 * @return {boolean} false if the value does not meet the given criteria.
 */
function checkReadability(value) {
    return (typeof (value) === 'string' && // strings only
        !value.includes('${Token[') && // no circular refs
        value !== '' // no empty strings
    );
}
//# sourceMappingURL=data:application/json;base64,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