"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildSagemakerNotebook = void 0;
const sagemaker = require("@aws-cdk/aws-sagemaker");
const ec2 = require("@aws-cdk/aws-ec2");
const kms_helper_1 = require("./kms-helper");
const sagemaker_defaults_1 = require("./sagemaker-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const vpc_helper_1 = require("./vpc-helper");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const vpc_defaults_1 = require("./vpc-defaults");
function addPermissions(_role) {
    // Grant permissions to NoteBookInstance for creating and training the model
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${core_1.Aws.PARTITION}:sagemaker:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:*`],
        actions: [
            "sagemaker:CreateTrainingJob",
            "sagemaker:DescribeTrainingJob",
            "sagemaker:CreateModel",
            "sagemaker:DescribeModel",
            "sagemaker:DeleteModel",
            "sagemaker:CreateEndpoint",
            "sagemaker:CreateEndpointConfig",
            "sagemaker:DescribeEndpoint",
            "sagemaker:DescribeEndpointConfig",
            "sagemaker:DeleteEndpoint",
            "sagemaker:DeleteEndpointConfig",
            "sagemaker:InvokeEndpoint"
        ],
    }));
    // Grant CloudWatch Logging permissions
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/sagemaker/*`],
        actions: [
            'logs:CreateLogGroup',
            'logs:CreateLogStream',
            'logs:DescribeLogStreams',
            'logs:GetLogEvents',
            'logs:PutLogEvents'
        ],
    }));
    // Grant GetRole permissions to the SageMaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: [
            'iam:GetRole'
        ],
    }));
    // Grant PassRole permissions to the SageMaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: [
            'iam:PassRole'
        ],
        conditions: {
            StringLike: { 'iam:PassedToService': 'sagemaker.amazonaws.com' }
        },
    }));
}
function buildSagemakerNotebook(scope, props) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j;
    // Setup the notebook properties
    let sagemakerNotebookProps;
    let vpcInstance;
    let securityGroup;
    let kmsKeyId;
    let subnetId;
    // Conditional Sagemaker Notebook creation
    if (!props.existingNotebookObj) {
        if ((((_a = props.sagemakerNotebookProps) === null || _a === void 0 ? void 0 : _a.subnetId) && ((_b = props.sagemakerNotebookProps) === null || _b === void 0 ? void 0 : _b.securityGroupIds) === undefined) ||
            (((_c = props.sagemakerNotebookProps) === null || _c === void 0 ? void 0 : _c.subnetId) === undefined && ((_d = props.sagemakerNotebookProps) === null || _d === void 0 ? void 0 : _d.securityGroupIds))) {
            throw new Error('Must define both sagemakerNotebookProps.subnetId and sagemakerNotebookProps.securityGroupIds');
        }
        addPermissions(props.role);
        if (((_e = props.sagemakerNotebookProps) === null || _e === void 0 ? void 0 : _e.kmsKeyId) === undefined) {
            kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
        }
        else {
            kmsKeyId = props.sagemakerNotebookProps.kmsKeyId;
        }
        if (props.deployInsideVpc === undefined || props.deployInsideVpc) {
            if (((_f = props.sagemakerNotebookProps) === null || _f === void 0 ? void 0 : _f.subnetId) === undefined && ((_g = props.sagemakerNotebookProps) === null || _g === void 0 ? void 0 : _g.securityGroupIds) === undefined) {
                vpcInstance = vpc_helper_1.buildVpc(scope, {
                    defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps()
                });
                securityGroup = new ec2.SecurityGroup(scope, "SecurityGroup", {
                    vpc: vpcInstance,
                    allowAllOutbound: false
                });
                securityGroup.addEgressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(443));
                // Add Cfn_Nag Suppression for WARN W5: Security Groups found with cidr open to world on egress
                const cfnSecurityGroup = securityGroup.node.findChild('Resource');
                cfnSecurityGroup.cfnOptions.metadata = {
                    cfn_nag: {
                        rules_to_suppress: [{
                                id: 'W5',
                                reason: 'Allow notebook users to access the Internet from the notebook'
                            }]
                    }
                };
                subnetId = vpcInstance.privateSubnets[0].subnetId;
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, subnetId, [securityGroup.securityGroupId]);
            }
            else {
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, (_h = props.sagemakerNotebookProps) === null || _h === void 0 ? void 0 : _h.subnetId, (_j = props.sagemakerNotebookProps) === null || _j === void 0 ? void 0 : _j.securityGroupIds);
            }
        }
        else {
            sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId);
        }
        if (props.sagemakerNotebookProps) {
            sagemakerNotebookProps = utils_1.overrideProps(sagemakerNotebookProps, props.sagemakerNotebookProps);
        }
        // Create the notebook
        const sagemakerInstance = new sagemaker.CfnNotebookInstance(scope, "SagemakerNotebook", sagemakerNotebookProps);
        if (vpcInstance) {
            return [sagemakerInstance, vpcInstance, securityGroup];
        }
        else {
            return [sagemakerInstance];
        }
    }
    else {
        // Return existing notebook object
        return [props.existingNotebookObj];
    }
}
exports.buildSagemakerNotebook = buildSagemakerNotebook;
//# sourceMappingURL=data:application/json;base64,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