"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildTopic = void 0;
// Imports
const sns = require("@aws-cdk/aws-sns");
const kms = require("@aws-cdk/aws-kms");
const sns_defaults_1 = require("./sns-defaults");
const kms_helper_1 = require("./kms-helper");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
function applySecureTopicPolicy(topic) {
    // Apply topic policy to enforce only the topic owner can publish and subscribe to this topic
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'TopicOwnerOnlyAccess',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AccountPrincipal(core_1.Stack.of(topic).account)],
        effect: aws_iam_1.Effect.ALLOW,
        conditions: {
            StringEquals: {
                "AWS:SourceOwner": core_1.Stack.of(topic).account
            }
        }
    }));
    // Apply Topic policy to enforce encryption of data in transit
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
function buildTopic(scope, props) {
    if (!props.existingTopicObj) {
        // Setup the topic properties
        let snsTopicProps;
        if (props.topicProps) {
            // If property overrides have been provided, incorporate them and deploy
            snsTopicProps = utils_1.overrideProps(sns_defaults_1.DefaultSnsTopicProps, props.topicProps);
        }
        else {
            // If no property overrides, deploy using the default configuration
            snsTopicProps = sns_defaults_1.DefaultSnsTopicProps;
        }
        // Set encryption properties
        if (props.enableEncryptionWithCustomerManagedKey === undefined || props.enableEncryptionWithCustomerManagedKey === false) {
            // Retrieve SNS managed key to encrypt the SNS Topic
            const awsManagedKey = kms.Alias.fromAliasName(scope, 'aws-managed-key', 'alias/aws/sns');
            snsTopicProps.masterKey = awsManagedKey;
        }
        else {
            // Use the imported Customer Managed KMS key
            if (props.encryptionKey) {
                snsTopicProps.masterKey = props.encryptionKey;
            }
            else {
                // Create a new Customer Managed KMS key
                snsTopicProps.masterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
            }
        }
        // Create the SNS Topic
        const topic = new sns.Topic(scope, 'SnsTopic', snsTopicProps);
        applySecureTopicPolicy(topic);
        return [topic, snsTopicProps.masterKey];
    }
    else {
        return [props.existingTopicObj];
    }
}
exports.buildTopic = buildTopic;
//# sourceMappingURL=data:application/json;base64,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