"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AddAwsServiceEndpoint = exports.ServiceEndpointTypes = exports.buildVpc = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const utils_1 = require("./utils");
function buildVpc(scope, props) {
    if (props === null || props === void 0 ? void 0 : props.existingVpc) {
        return props === null || props === void 0 ? void 0 : props.existingVpc;
    }
    let cumulativeProps = props === null || props === void 0 ? void 0 : props.defaultVpcProps;
    if (props === null || props === void 0 ? void 0 : props.userVpcProps) {
        cumulativeProps = utils_1.overrideProps(cumulativeProps, props === null || props === void 0 ? void 0 : props.userVpcProps);
    }
    if (props === null || props === void 0 ? void 0 : props.constructVpcProps) {
        cumulativeProps = utils_1.overrideProps(cumulativeProps, props === null || props === void 0 ? void 0 : props.constructVpcProps);
    }
    const vpc = new ec2.Vpc(scope, "Vpc", cumulativeProps);
    // Add VPC FlowLogs with the default setting of trafficType:ALL and destination: CloudWatch Logs
    vpc.addFlowLog("FlowLog");
    // Add Cfn Nag suppression for PUBLIC subnets to suppress WARN W33: EC2 Subnet should not have MapPublicIpOnLaunch set to true
    vpc.publicSubnets.forEach((subnet) => {
        const cfnSubnet = subnet.node.defaultChild;
        cfnSubnet.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W33',
                        reason: 'Allow Public Subnets to have MapPublicIpOnLaunch set to true'
                    }]
            }
        };
    });
    return vpc;
}
exports.buildVpc = buildVpc;
var ServiceEndpointTypes;
(function (ServiceEndpointTypes) {
    ServiceEndpointTypes["DYNAMODB"] = "DDB";
    ServiceEndpointTypes["SNS"] = "SNS";
    ServiceEndpointTypes["SQS"] = "SQS";
    ServiceEndpointTypes["S3"] = "S3";
    ServiceEndpointTypes["STEPFUNCTIONS"] = "STEPFUNCTIONS";
})(ServiceEndpointTypes = exports.ServiceEndpointTypes || (exports.ServiceEndpointTypes = {}));
var EndpointTypes;
(function (EndpointTypes) {
    EndpointTypes["GATEWAY"] = "Gateway";
    EndpointTypes["INTERFACE"] = "Interface";
})(EndpointTypes || (EndpointTypes = {}));
const endpointSettings = [
    {
        endpointName: ServiceEndpointTypes.DYNAMODB,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.DYNAMODB,
    },
    {
        endpointName: ServiceEndpointTypes.S3,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.S3,
    },
    {
        endpointName: ServiceEndpointTypes.SNS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SNS,
    },
    {
        endpointName: ServiceEndpointTypes.SQS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SQS,
    },
];
function AddAwsServiceEndpoint(scope, vpc, interfaceTag) {
    if (!vpc.node.children.some((child) => child.node.id === interfaceTag)) {
        const service = endpointSettings.find((endpoint) => endpoint.endpointName === interfaceTag);
        if (!service) {
            throw new Error("Unsupported Service sent to AddServiceEndpoint");
        }
        if (service.endpointType === EndpointTypes.GATEWAY) {
            vpc.addGatewayEndpoint(interfaceTag, {
                service: service.endpointGatewayService,
            });
        }
        if (service.endpointType === EndpointTypes.INTERFACE) {
            const endpointDefaultSecurityGroup = new ec2.SecurityGroup(scope, "ReplaceEndpointDefaultSecurityGroup", {
                vpc,
                allowAllOutbound: true,
            });
            // Allow https traffic from within the VPC
            endpointDefaultSecurityGroup.addIngressRule(ec2.Peer.ipv4(vpc.vpcCidrBlock), ec2.Port.tcp(443));
            const cfnSecurityGroup = endpointDefaultSecurityGroup.node.findChild("Resource");
            cfnSecurityGroup.cfnOptions.metadata = {
                cfn_nag: {
                    rules_to_suppress: [
                        {
                            id: "W5",
                            reason: "Egress of 0.0.0.0/0 is default and generally considered OK",
                        },
                        {
                            id: "W40",
                            reason: "Egress IPProtocol of -1 is default and generally considered OK",
                        },
                    ],
                },
            };
            vpc.addInterfaceEndpoint(interfaceTag, {
                service: service.endpointInterfaceService,
                securityGroups: [endpointDefaultSecurityGroup],
            });
        }
    }
    return;
}
exports.AddAwsServiceEndpoint = AddAwsServiceEndpoint;
//# sourceMappingURL=data:application/json;base64,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