"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const api = require("@aws-cdk/aws-apigateway");
const lambda = require("@aws-cdk/aws-lambda");
const defaults = require("../index");
const s3 = require("@aws-cdk/aws-s3");
const cloudfront_distribution_helper_1 = require("../lib/cloudfront-distribution-helper");
require("@aws-cdk/assert/jest");
const origins = require("@aws-cdk/aws-cloudfront-origins");
test('cloudfront distribution for ApiGateway with default params', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const func = new lambda.Function(stack, 'LambdaFunction', lambdaFunctionProps);
    const _api = new api.LambdaRestApi(stack, 'RestApi', {
        handler: func
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForApiGateway(stack, _api);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('cloudfront distribution for ApiGateway without security headers', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const func = new lambda.Function(stack, 'LambdaFunction', lambdaFunctionProps);
    const _api = new api.LambdaRestApi(stack, 'RestApi', {
        handler: func
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForApiGateway(stack, _api, {}, false);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('test cloudfront for Api Gateway with user provided logging bucket', () => {
    const stack = new core_1.Stack();
    const loggingBucket = new s3.Bucket(stack, 'MyCloudfrontLoggingBucket', defaults.DefaultS3Props());
    const inProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda-test`),
        runtime: lambda.Runtime.PYTHON_3_6,
        handler: 'index.handler'
    };
    const cfdProps = {
        loggingConfig: {
            bucket: loggingBucket
        }
    };
    const func = defaults.deployLambdaFunction(stack, inProps);
    const _api = new api.LambdaRestApi(stack, 'RestApi1', {
        handler: func
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForApiGateway(stack, _api, cfdProps);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: "origin-response",
                        LambdaFunctionARN: {
                            Ref: "SetHttpSecurityHeadersVersion660E2F72"
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "MyCloudfrontLoggingBucket9AA652E8",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: "https-only"
                    },
                    DomainName: {
                        "Fn::Select": [
                            0,
                            {
                                "Fn::Split": [
                                    "/",
                                    {
                                        "Fn::Select": [
                                            1,
                                            {
                                                "Fn::Split": [
                                                    "://",
                                                    {
                                                        "Fn::Join": [
                                                            "",
                                                            [
                                                                "https://",
                                                                {
                                                                    Ref: "RestApi1480AC499"
                                                                },
                                                                ".execute-api.",
                                                                {
                                                                    Ref: "AWS::Region"
                                                                },
                                                                ".",
                                                                {
                                                                    Ref: "AWS::URLSuffix"
                                                                },
                                                                "/",
                                                                {
                                                                    Ref: "RestApi1DeploymentStageprod4FFC9BB4"
                                                                },
                                                                "/"
                                                            ]
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    OriginPath: {
                        "Fn::Join": [
                            "",
                            [
                                "/",
                                {
                                    Ref: "RestApi1DeploymentStageprod4FFC9BB4"
                                }
                            ]
                        ]
                    }
                }
            ]
        }
    });
});
test('test cloudfront for Api Gateway override properties', () => {
    const stack = new core_1.Stack();
    const inProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda-test`),
        runtime: lambda.Runtime.PYTHON_3_6,
        handler: 'index.handler'
    };
    const func = defaults.deployLambdaFunction(stack, inProps);
    const _api = new api.LambdaRestApi(stack, 'RestApi1', {
        handler: func
    });
    const props = {
        defaultBehavior: {
            origin: new origins.HttpOrigin(_api.url, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY
            }),
            allowedMethods: cloudfront.AllowedMethods.ALLOW_ALL,
            cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS
        },
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForApiGateway(stack, _api, props);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "GET",
                    "HEAD",
                    "OPTIONS",
                    "PUT",
                    "PATCH",
                    "POST",
                    "DELETE"
                ],
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                CachedMethods: [
                    "GET",
                    "HEAD",
                    "OPTIONS"
                ],
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: "origin-response",
                        LambdaFunctionARN: {
                            Ref: "SetHttpSecurityHeadersVersion660E2F72"
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: "http-only"
                    },
                    DomainName: {
                        "Fn::Join": [
                            "",
                            [
                                "https://",
                                {
                                    Ref: "RestApi1480AC499"
                                },
                                ".execute-api.",
                                {
                                    Ref: "AWS::Region"
                                },
                                ".",
                                {
                                    Ref: "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    Ref: "RestApi1DeploymentStageprod4FFC9BB4"
                                },
                                "/"
                            ]
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12"
                }
            ]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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