"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const s3 = require("@aws-cdk/aws-s3");
const mediastore = require("@aws-cdk/aws-mediastore");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const cloudfront_distribution_helper_1 = require("../lib/cloudfront-distribution-helper");
test('CloudFront distribution for MediaStore with default params', () => {
    const stack = new core_1.Stack();
    const mediaStoreContainerProps = {
        containerName: 'TestContainer'
    };
    const mediaStoreContainer = new mediastore.CfnContainer(stack, 'MediaStoreContainer', mediaStoreContainerProps);
    cloudfront_distribution_helper_1.CloudFrontDistributionForMediaStore(stack, mediaStoreContainer);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('CloudFront distribution for MediaStore with user provided log bucket', () => {
    const stack = new core_1.Stack();
    const mediaStoreContainerProps = {
        containerName: 'TestContainer'
    };
    const mediaStoreContainer = new mediastore.CfnContainer(stack, 'MediaStoreContainer', mediaStoreContainerProps);
    const logBucket = new s3.Bucket(stack, 'LoggingBucket');
    const cfProps = {
        enableLogging: true,
        logBucket
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForMediaStore(stack, mediaStoreContainer, cfProps);
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-response',
                        LambdaFunctionARN: {
                            Ref: 'SetHttpSecurityHeadersVersion660E2F72'
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'CloudfrontOriginRequestPolicy299A10DB'
                },
                TargetOriginId: 'CloudFrontDistributionOrigin176EC3A12',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'LoggingBucket1E5A6F3B',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'MediaStoreContainer',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'CloudFrontDistributionOrigin176EC3A12'
                }
            ]
        }
    });
});
test('CloudFront distribution for MediaStore with user provided origin request policy', () => {
    const stack = new core_1.Stack();
    const mediaStoreContainerProps = {
        containerName: 'TestContainer'
    };
    const mediaStoreContainer = new mediastore.CfnContainer(stack, 'MediaStoreContainer', mediaStoreContainerProps);
    const originRequestPolicyProps = {
        headerBehavior: {
            behavior: 'all'
        },
        queryStringBehavior: {
            behavior: 'none'
        },
        cookieBehavior: {
            behavior: 'all'
        }
    };
    const originRequestPolicy = new cloudfront.OriginRequestPolicy(stack, 'MyCloudfrontOriginRequestPolicy', originRequestPolicyProps);
    const cfProps = {
        defaultBehavior: {
            originRequestPolicy
        }
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForMediaStore(stack, mediaStoreContainer, cfProps);
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-response',
                        LambdaFunctionARN: {
                            Ref: 'SetHttpSecurityHeadersVersion660E2F72'
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'MyCloudfrontOriginRequestPolicy632B7DED'
                },
                TargetOriginId: 'CloudFrontDistributionOrigin176EC3A12',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'CloudfrontLoggingBucket3C3EFAA7',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'MediaStoreContainer',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'CloudFrontDistributionOrigin176EC3A12'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            CookiesConfig: {
                CookieBehavior: 'all'
            },
            HeadersConfig: {
                HeaderBehavior: 'all'
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'none'
            },
            Name: 'MyCloudfrontOriginRequestPolicy'
        }
    });
});
test('CloudFront distribution for MediaStore with user provided custom headers with CloudFrontOriginAccessIdentity', () => {
    const stack = new core_1.Stack();
    const mediaStoreContainerProps = {
        containerName: 'TestContainer'
    };
    const mediaStoreContainer = new mediastore.CfnContainer(stack, 'MediaStoreContainer', mediaStoreContainerProps);
    const cloudfrontOriginAccessIdentity = cloudfront_distribution_helper_1.CloudFrontOriginAccessIdentity(stack);
    const cfProps = {
        customHeaders: {
            'User-Agent': cloudfrontOriginAccessIdentity.originAccessIdentityName
        }
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForMediaStore(stack, mediaStoreContainer, cfProps);
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-response',
                        LambdaFunctionARN: {
                            Ref: 'SetHttpSecurityHeadersVersion660E2F72'
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'CloudfrontOriginRequestPolicy299A10DB'
                },
                TargetOriginId: 'CloudFrontDistributionOrigin176EC3A12',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'CloudfrontLoggingBucket3C3EFAA7',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    OriginCustomHeaders: [
                        {
                            HeaderName: 'User-Agent',
                            HeaderValue: {
                                Ref: 'CloudFrontOriginAccessIdentity04EB66DA'
                            }
                        }
                    ],
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'MediaStoreContainer',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'CloudFrontDistributionOrigin176EC3A12'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::CloudFrontOriginAccessIdentity', {
        CloudFrontOriginAccessIdentityConfig: {
            Comment: {
                'Fn::Join': [
                    '',
                    [
                        'access-identity-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-',
                        {
                            Ref: 'AWS::StackName'
                        }
                    ]
                ]
            }
        }
    });
});
test('CloudFront distribution without HTTP security headers for MediaStore', () => {
    const stack = new core_1.Stack();
    const mediaStoreContainerProps = {
        containerName: 'TestContainer'
    };
    const mediaStoreContainer = new mediastore.CfnContainer(stack, 'MediaStoreContainer', mediaStoreContainerProps);
    cloudfront_distribution_helper_1.CloudFrontDistributionForMediaStore(stack, mediaStoreContainer, {}, false);
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                OriginRequestPolicyId: {
                    Ref: 'CloudfrontOriginRequestPolicy299A10DB'
                },
                TargetOriginId: 'CloudFrontDistributionOrigin176EC3A12',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'CloudfrontLoggingBucket3C3EFAA7',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'MediaStoreContainer',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'CloudFrontDistributionOrigin176EC3A12'
                }
            ]
        }
    });
});
test('CloudFront distribution for MediaStore override params', () => {
    const stack = new core_1.Stack();
    const mediaStoreContainerProps = {
        containerName: 'TestContainer'
    };
    const mediaStoreContainer = new mediastore.CfnContainer(stack, 'MediaStoreContainer', mediaStoreContainerProps);
    const cfProps = {
        defaultBehavior: {
            origin: new origins.HttpOrigin(mediaStoreContainer.attrEndpoint, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY
            }),
            viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.HTTPS_ONLY,
            allowedMethods: cloudfront.AllowedMethods.ALLOW_ALL,
            cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD
        }
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForMediaStore(stack, mediaStoreContainer, cfProps);
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS',
                    'PUT',
                    'PATCH',
                    'POST',
                    'DELETE'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD'
                ],
                Compress: true,
                OriginRequestPolicyId: {
                    Ref: 'CloudfrontOriginRequestPolicy299A10DB'
                },
                TargetOriginId: 'CloudFrontDistributionOrigin176EC3A12',
                ViewerProtocolPolicy: 'https-only'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'CloudfrontLoggingBucket3C3EFAA7',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'http-only'
                    },
                    DomainName: {
                        'Fn::GetAtt': [
                            'MediaStoreContainer',
                            'Endpoint'
                        ]
                    },
                    Id: 'CloudFrontDistributionOrigin176EC3A12'
                }
            ]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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