"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const kinesisfirehose = require("@aws-cdk/aws-kinesisfirehose");
const defaults = require("../index");
const utils_1 = require("../lib/utils");
require("@aws-cdk/assert/jest");
const kms = require("@aws-cdk/aws-kms");
test('snapshot test kinesisfirehose default params', () => {
    const stack = new core_1.Stack();
    const awsManagedKey = kms.Alias.fromAliasName(stack, 'aws-managed-key', 'alias/aws/s3');
    new kinesisfirehose.CfnDeliveryStream(stack, 'KinesisFirehose', defaults.DefaultCfnDeliveryStreamProps('bucket_arn', 'role_arn', 'log_group', 'log_stream', awsManagedKey));
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('test kinesisanalytics override buffer conditions', () => {
    const stack = new core_1.Stack();
    const inProps = {
        extendedS3DestinationConfiguration: {
            bufferingHints: {
                intervalInSeconds: 600,
                sizeInMBs: 10
            },
        }
    };
    const awsManagedKey = kms.Alias.fromAliasName(stack, 'aws-managed-key', 'alias/aws/s3');
    const defaultProps = defaults.DefaultCfnDeliveryStreamProps('bucket_arn', 'role_arn', 'log_group', 'log_stream', awsManagedKey);
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new kinesisfirehose.CfnDeliveryStream(stack, 'KinesisFirehose', outProps);
    expect(stack).toHaveResource("AWS::KinesisFirehose::DeliveryStream", {
        ExtendedS3DestinationConfiguration: {
            BucketARN: "bucket_arn",
            BufferingHints: {
                IntervalInSeconds: 600,
                SizeInMBs: 10
            },
            CloudWatchLoggingOptions: {
                Enabled: true,
                LogGroupName: "log_group",
                LogStreamName: "log_stream"
            },
            CompressionFormat: "GZIP",
            EncryptionConfiguration: {
                KMSEncryptionConfig: {
                    AWSKMSKeyARN: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                {
                                    Ref: "AWS::Partition"
                                },
                                ":kms:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":alias/aws/s3"
                            ]
                        ]
                    }
                }
            },
            RoleARN: "role_arn"
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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