"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const defaults = require("../index");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
test('test DynamoEventSourceProps with defaults', () => {
    const stack = new core_1.Stack();
    const props = defaults.DynamoEventSourceProps(stack);
    expect(props.onFailure).toBeInstanceOf(aws_lambda_event_sources_1.SqsDlq);
    expect(props.startingPosition).toEqual("TRIM_HORIZON");
    expect(props.bisectBatchOnError).toEqual(true);
    expect(props.retryAttempts).toEqual(500);
    expect(props.maxRecordAge).toEqual(core_1.Duration.hours(24));
});
test('test DynamoEventSourceProps with deployDeadLetterQueue: false', () => {
    const stack = new core_1.Stack();
    const props = defaults.DynamoEventSourceProps(stack, {
        deploySqsDlqQueue: false
    });
    expect(props).toEqual({
        startingPosition: "TRIM_HORIZON",
        bisectBatchOnError: true,
        maxRecordAge: core_1.Duration.hours(24),
        retryAttempts: 500
    });
});
test('test DynamoEventSourceProps override', () => {
    const stack = new core_1.Stack();
    const myProps = {
        startingPosition: lambda.StartingPosition.LATEST,
        batchSize: 5,
        retryAttempts: 3
    };
    const props = defaults.DynamoEventSourceProps(stack, {
        eventSourceProps: myProps,
        deploySqsDlqQueue: false
    });
    expect(props).toEqual({
        batchSize: 5,
        startingPosition: "LATEST",
        maxRecordAge: core_1.Duration.hours(24),
        bisectBatchOnError: true,
        retryAttempts: 3
    });
});
test('test S3EventSourceProps w/ default props', () => {
    const props = defaults.S3EventSourceProps();
    expect(props).toEqual({
        events: ["s3:ObjectCreated:*"]
    });
});
test('test S3EventSourceProps w/ user props', () => {
    const s3EventSourceProps = {
        events: [
            s3.EventType.OBJECT_CREATED_POST
        ]
    };
    const props = defaults.S3EventSourceProps(s3EventSourceProps);
    expect(props).toEqual({
        events: ["s3:ObjectCreated:Post"]
    });
});
test('test KinesisEventSourceProps with defaults', () => {
    const stack = new core_1.Stack();
    const props = defaults.KinesisEventSourceProps(stack);
    expect(props.onFailure).toBeInstanceOf(aws_lambda_event_sources_1.SqsDlq);
    expect(props.startingPosition).toEqual("TRIM_HORIZON");
    expect(props.bisectBatchOnError).toEqual(true);
    expect(props.retryAttempts).toEqual(500);
    expect(props.maxRecordAge).toEqual(core_1.Duration.hours(24));
});
test('test KinesisEventSourceProps with deployDeadLetterQueue: false', () => {
    const stack = new core_1.Stack();
    const props = defaults.KinesisEventSourceProps(stack, {
        deploySqsDlqQueue: false
    });
    expect(props).toEqual({
        startingPosition: "TRIM_HORIZON",
        bisectBatchOnError: true,
        maxRecordAge: core_1.Duration.hours(24),
        retryAttempts: 500
    });
});
test('test KinesisEventSourceProps override', () => {
    const stack = new core_1.Stack();
    const myProps = {
        startingPosition: lambda.StartingPosition.LATEST,
        batchSize: 5,
        retryAttempts: 3
    };
    const props = defaults.KinesisEventSourceProps(stack, {
        eventSourceProps: myProps,
        deploySqsDlqQueue: false
    });
    expect(props).toEqual({
        batchSize: 5,
        startingPosition: "LATEST",
        maxRecordAge: core_1.Duration.hours(24),
        bisectBatchOnError: true,
        retryAttempts: 3
    });
});
test('test sqsDlqQueueProps override', () => {
    const stack = new core_1.Stack();
    defaults.KinesisEventSourceProps(stack, {
        sqsDlqQueueProps: {
            queueName: 'hello-world',
            visibilityTimeout: core_1.Duration.seconds(50)
        }
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "hello-world",
        VisibilityTimeout: 50
    });
});
//# sourceMappingURL=data:application/json;base64,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