"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const sfn = require("@aws-cdk/aws-stepfunctions");
const aws_logs_1 = require("@aws-cdk/aws-logs");
// --------------------------------------------------------------
// Test minimal deployment with no properties
// --------------------------------------------------------------
test('Test minimal deployment with no properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    defaults.buildStateMachine(stack, {
        definition: startState
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test deployment w/ custom properties
// --------------------------------------------------------------
test('Test deployment w/ custom properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    defaults.buildStateMachine(stack, {
        definition: startState,
        stateMachineName: 'myStateMachine'
    });
    // Assertion
    expect(stack).toHaveResource("AWS::StepFunctions::StateMachine", {
        StateMachineName: "myStateMachine"
    });
});
// --------------------------------------------------------------
// Test deployment w/ logging enabled
// --------------------------------------------------------------
test('Test deployment w/ logging enabled', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Log group
    const logGroup = new aws_logs_1.LogGroup(stack, 'myLogGroup', defaults.DefaultLogGroupProps());
    // Build state machine
    defaults.buildStateMachine(stack, {
        definition: startState,
        logs: {
            destination: logGroup,
            level: sfn.LogLevel.FATAL
        }
    });
    // Assertion
    expect(stack).toHaveResource("AWS::StepFunctions::StateMachine", {
        LoggingConfiguration: {
            Destinations: [{
                    CloudWatchLogsLogGroup: {
                        LogGroupArn: {
                            "Fn::GetAtt": [
                                "myLogGroup46524CAB",
                                "Arn"
                            ]
                        }
                    }
                }],
            Level: 'FATAL'
        }
    });
});
// --------------------------------------------------------------
// Check default Cloudwatch perissions
// --------------------------------------------------------------
test('Test deployment w/ logging enabled', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    defaults.buildStateMachine(stack, {
        definition: startState
    });
    // Assertion
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "logs:CreateLogDelivery",
                        "logs:GetLogDelivery",
                        "logs:UpdateLogDelivery",
                        "logs:DeleteLogDelivery",
                        "logs:ListLogDeliveries"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "logs:PutResourcePolicy",
                        "logs:DescribeResourcePolicies",
                        "logs:DescribeLogGroups"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                {
                                    Ref: "AWS::Partition"
                                },
                                ":logs:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
// --------------------------------------------------------------
// Check CW Alarms
// --------------------------------------------------------------
test('Count State Machine CW Alarms', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    const [sm] = defaults.buildStateMachine(stack, {
        definition: startState
    });
    const cwList = defaults.buildStepFunctionCWAlarms(stack, sm);
    expect(cwList.length).toEqual(3);
});
//# sourceMappingURL=data:application/json;base64,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