"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const ec2 = require("@aws-cdk/aws-ec2");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const vpc_helper_1 = require("../lib/vpc-helper");
const vpc_defaults_1 = require("../lib/vpc-defaults");
// --------------------------------------------------------------
// Test minimal Public/Private deployment with no properties
// --------------------------------------------------------------
test('Test minimal deployment with no properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps()
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test minimal Isolated deployment with no properties
// --------------------------------------------------------------
test('Test minimal deployment with no properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultIsolatedVpcProps()
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toHaveResource("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toCountResources("AWS::EC2::Subnet", 2);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toCountResources("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test deployment w/ custom CIDR
// --------------------------------------------------------------
test('Test deployment w/ custom CIDR', () => {
    // Stack
    const stack = new core_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        userVpcProps: {
            cidr: '172.168.0.0/16'
        }
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test deployment w/ user provided custom properties
// --------------------------------------------------------------
test("Test deployment w/ user provided custom properties", () => {
    // Stack
    const stack = new core_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        userVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: '172.168.0.0/16'
        }
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        CidrBlock: "172.168.0.0/16",
        EnableDnsHostnames: false,
        EnableDnsSupport: false
    });
});
// --------------------------------------------------------------
// Test deployment w/ construct provided custom properties
// --------------------------------------------------------------
test("Test deployment w/ construct provided custom properties", () => {
    // Stack
    const stack = new core_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            cidr: "172.168.0.0/16",
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        CidrBlock: "172.168.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
// --------------------------------------------------------------
// Test deployment w/ construct and user provided custom properties
// --------------------------------------------------------------
test("Test deployment w/ construct and user provided custom properties", () => {
    // Stack
    const stack = new core_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        userVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: "10.0.0.0/16",
        },
        constructVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: "172.168.0.0/16",
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        CidrBlock: "172.168.0.0/16",
        EnableDnsHostnames: false,
        EnableDnsSupport: false,
    });
});
// --------------------------------------------------------------
// Test priority of default, user and construct properties
// --------------------------------------------------------------
test("Test deployment w/ construct and user provided custom properties", () => {
    // Stack
    const stack = new core_1.Stack();
    // Build VPC
    const v = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        userVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: "10.0.0.0/16",
        },
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            natGateways: 0,
            subnetConfiguration: [
                {
                    cidrMask: 18,
                    name: "isolated",
                    subnetType: ec2.SubnetType.ISOLATED,
                },
            ],
        },
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, v, defaults.ServiceEndpointTypes.SQS);
    // Expect 2 isolated subnets (usual error condition is 2 public/2 private)
    expect(stack).toCountResources("AWS::EC2::Subnet", 2);
    expect(stack).toCountResources("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test deployment w/ existing VPC provided
// --------------------------------------------------------------
test("Test deployment w/ existing VPC provided", () => {
    // Stack
    const stack = new core_1.Stack();
    // Build VPC
    const testExistingVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: "172.168.0.0/16",
        },
    });
    const newVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        existingVpc: testExistingVpc,
    });
    expect(newVpc).toBe(testExistingVpc);
});
// --------------------------------------------------------------
// Test adding Gateway Endpoint
// --------------------------------------------------------------
test("Test adding Gateway Endpoint", () => {
    // Stack
    const stack = new core_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.DYNAMODB);
    // Assertion
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Gateway",
    });
});
// --------------------------------------------------------------
// Test adding Interface Endpoint
// --------------------------------------------------------------
test("Test adding Interface Endpoint", () => {
    // Stack
    const stack = new core_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SNS);
    // Assertion
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
});
// --------------------------------------------------------------
// Test adding a second Endpoint of same service
// --------------------------------------------------------------
test("Test adding a second Endpoint of same service", () => {
    // Stack
    const stack = new core_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SNS);
    // Assertion
    expect(stack).toCountResources("AWS::EC2::VPCEndpoint", 1);
});
// --------------------------------------------------------------
// Test adding bad Endpoint
// --------------------------------------------------------------
test("Test adding bad Endpoint", () => {
    // Stack
    const stack = new core_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    const app = () => {
        vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, "string");
    };
    // Assertion
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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