import random
from typing import List, Dict, Union

from rich.box import MINIMAL
from rich.console import Console
from rich.table import Table

from malwarebazaar.config import Config
from malwarebazaar.models import Sample, YaraRule, Task

COLORS = [
    "navy_blue", "dark_blue", "blue3", "blue1", "dark_green", "deep_sky_blue4", "dodger_blue3",
    "dodger_blue2", "green4", "spring_green4", "turquoise4", "deep_sky_blue3", "dodger_blue1", "green3",
    "spring_green3", "dark_cyan", "light_sea_green", "deep_sky_blue2", "deep_sky_blue1", "spring_green2", "cyan3",
    "dark_turquoise", "turquoise2", "green1", "spring_green1", "medium_spring_green", "cyan2", "cyan1",
    "dark_red", "deep_pink4", "purple4", "purple3", "blue_violet", "orange4", "grey37", "gray37",
    "medium_purple4", "slate_blue3", "royal_blue1", "chartreuse4", "dark_sea_green4", "pale_turquoise4",
    "steel_blue", "steel_blue3", "cornflower_blue", "chartreuse3", "cadet_blue", "sky_blue3", "steel_blue1",
    "pale_green3", "sea_green3", "aquamarine3", "medium_turquoise", "chartreuse2", "sea_green2", "sea_green1",
    "aquamarine1", "dark_slate_gray2", "dark_magenta", "dark_violet", "purple", "light_pink4", "plum4",
    "medium_purple3", "slate_blue1", "yellow4", "wheat4", "grey53", "gray53", "light_slate_grey",
    "light_slate_gray", "medium_purple", "light_slate_blue", "dark_olive_green3", "dark_sea_green",
    "light_sky_blue3", "sky_blue2", "dark_sea_green3", "dark_slate_gray3", "sky_blue1", "chartreuse1",
    "light_green", "pale_green1", "dark_slate_gray1", "red3", "medium_violet_red", "magenta3", "dark_orange3",
    "indian_red", "hot_pink3", "medium_orchid3", "medium_orchid", "medium_purple2", "dark_goldenrod",
    "light_salmon3", "rosy_brown", "grey63", "gray63", "medium_purple1", "gold3", "dark_khaki", "navajo_white3",
    "grey69", "gray69", "light_steel_blue3", "light_steel_blue", "yellow3", "dark_sea_green2", "light_cyan3",
    "light_sky_blue1", "green_yellow", "dark_olive_green2", "dark_sea_green1", "pale_turquoise1", "deep_pink3",
    "magenta2", "hot_pink2", "orchid", "medium_orchid1", "orange3", "light_pink3", "pink3", "plum3", "violet",
    "light_goldenrod3", "tan", "misty_rose3", "thistle3", "plum2", "khaki3", "light_goldenrod2", "light_yellow3",
    "grey84", "gray84", "light_steel_blue1", "yellow2", "dark_olive_green1", "honeydew2", "light_cyan1", "red1",
    "deep_pink2", "deep_pink1", "magenta1", "orange_red1", "indian_red1", "hot_pink", "dark_orange", "salmon1",
    "light_coral", "pale_violet_red1", "orchid2", "orchid1", "orange1", "sandy_brown", "light_salmon1",
    "light_pink1", "pink1", "plum1", "gold1", "navajo_white1", "misty_rose1", "thistle1", "yellow1",
    "light_goldenrod1", "khaki1", "wheat1", "cornsilk1"
]


def format_none(s: str) -> str:
    return f"[italic bright_black]{s}[/italic bright_black]"


def format_tlp(s: str) -> str:
    l = s.lower()
    if "white" in l or "clear" in l:
        return f"[bold white on black]{s}[/bold white on black]"
    if "green" in l:
        return f"[bold green1 on black]{s}[/bold green1 on black]"
    if "amber" in l:
        return f"[bold orange1 on black]{s}[/bold orange1 on black]"
    if "red" in l:
        return f"[bold red1 on black]{s}[/bold red1 on black]"
    return s


def create_sample_table(s: Sample) -> Table:
    table = Table(show_header=False, box=MINIMAL)
    table.add_column()
    table.add_column(overflow="fold")
    table.add_row("Filename", s.file_name if s.file_name else format_none("None"))
    table.add_row("Filesize", str(s.file_size) + " bytes")
    table.add_row("Filetype", f"{s.file_type_mime}")
    if any([s.first_seen, s.last_seen, s.sightings]):
        table.add_row(
            "Sightings",
            f"First-Seen: {s.first_seen if s.first_seen else format_none('None')}\n"
            f"Last-Seen:  {s.last_seen if s.last_seen else format_none('None')}\n"
            f"Sightings:  {s.sightings if s.sightings else format_none('None')}"
        )
    table.add_row(
        "Hashes",
        f"MD5:\t    {s.md5_hash}\n"
        f"SHA1:\t    {s.sha1_hash}\n"
        f"SHA256:\t    {s.sha256_hash}\n"
        f"SHA3-384:   {s.sha3_384_hash}\n"
        f"Icon Dhash: {s.dhash_icon if s.dhash_icon else format_none('None')}"
    )
    if any([s.imphash, s.gimphash, s.telfhash]):
        table.add_row(
            "Import Hashes",
            f"Imphash:    {s.imphash if s.imphash else format_none('None')}\n"
            f"Gimphash:   {s.gimphash if s.gimphash else format_none('None')}\n"
            f"Telfhash:   {s.telfhash if s.telfhash else format_none('None')}"
        )
    table.add_row(
        "Fuzzy Hashes",
        f"Ssdeep:     {s.ssdeep}\n"
        f"Tlsh:       {s.tlsh}\n"
    )
    if s.signature:
        table.add_row("Signature", s.signature)
    if s.tags:
        table.add_row("Tags", ", ".join(s.tags))

    return table


def print_sample_table(s: Sample, console: Console = Console()):
    console.print(create_sample_table(s))


def process_vendor_info(vendor_info: Union[Dict, List[Dict]]) -> str:
    info = ""
    if isinstance(vendor_info, List):
        if len(vendor_info) == 0:
            return info
        vendor_info = vendor_info[0]

    if "malware_family" in vendor_info:
        mw = vendor_info["malware_family"]
        info += mw if mw else format_none("No family")
        info += "\n"
    elif "family_name" in vendor_info:
        mw = vendor_info["family_name"]
        info += mw if mw else format_none("No family")
        info += "\n"
    elif "detections" in vendor_info and isinstance(vendor_info["detections"], list):
        detections = vendor_info["detections"]
        if len(detections) == 0:
            info = format_none("Undetected") + "\n"
        else:
            mw = "\n".join(detections)
            info += mw + "\n"
    elif "detection" in vendor_info:
        d = vendor_info["detection"]
        if not d:
            info += format_none("Undetected") + "\n"
        elif "malicious" in d.lower():
            info += "[red1]malicious[/red1]"
        elif "suspicious" in d.lower():
            info += "[orange1]suspicious[/orange1]"
        else:
            info += f"[red1]{d}[/red1]"
    elif "threat_name" in vendor_info and vendor_info["threat_name"]:
        info += vendor_info["threat_name"] + "\n"

    if "verdict" in vendor_info or "status" in vendor_info:
        verdict = vendor_info.get("verdict", "") or vendor_info.get("status", "")
        verdict = verdict.lower()
        if "malicious" in verdict or "malware" in verdict:
            verdict = "[red1]malicious[/red1]"
        elif "suspicious" in verdict:
            verdict = "[orange1]suspicious[orange1]"
        else:
            verdict = format_none("Undetected")
        info += verdict
    elif "score" in vendor_info:
        score = vendor_info["score"]
        if score:
            if "." not in score:
                score = int(score, 10)
                if score > 6:
                    info += "[red1]malicious[/red1]"
                elif score > 1:
                    info += "[orange1]suspicious[/orange1]"
                else:
                    info += "[green1]clean[/green1]"

        else:
            info += format_none("No score")
    elif "detections" in vendor_info:
        d = vendor_info["detections"]
        if isinstance(d, list) and len(d) > 0:
            info += "[red1]malicious[/red1]"
    info += "\n"
    return info


def check_and_process_vendor_dict(d: Dict, key: str) -> str:
    if key == "UnpacMe":
        if len(d[key]) == 0:
            return ""
        use_idx = 0
        for idx, report in enumerate(d[key]):
            if len(report.get("detections", [])) > 0:
                use_idx = idx
                return process_vendor_info(d[key][use_idx])
        return process_vendor_info(d[key][use_idx])
    return process_vendor_info(d[key])


def create_vendor_info_table(s: Sample) -> Table:
    table = Table(show_header=False, box=MINIMAL)
    vi = s.vendor_intel
    table.add_column()
    table.add_column()
    table.add_column()
    table.add_column()

    keys = list(vi.keys())
    for idx in range(0, len(vi), 2):
        intel1 = check_and_process_vendor_dict(vi, keys[idx])
        if idx < len(vi) - 1:
            key2 = keys[idx + 1]
            intel2 = check_and_process_vendor_dict(vi, key2)
        else:
            key2 = None
            intel2 = None
        table.add_row(
            keys[idx],
            intel1,
            key2,
            intel2
        )
    return table


def print_vendor_info_table(s: Sample, c: Console = Console()):
    c.print(create_vendor_info_table(s))


def print_yararule_table(y: YaraRule, c: Console = Console()):
    table = Table(show_header=False, box=MINIMAL)
    table.add_column()
    table.add_column(overflow="fold")

    table.add_row(
        "Rule Metadata",
        f"Rule name:       {y.rule_name}\n"
        f"Date:            {y.date}\n"
        f"Author:          {y.author}\n"
        f"Description:     {y.description}\n"
        f"Malpedia Family: {y.malpedia_family if y.malpedia_family else format_none('None')}"
    )

    table.add_row(
        "YARAhub Metadata",
        f"Timestamp: {y.time_stamp}\n"
        f"UUID:      {y.yarahub_uuid}\n"
        f"Reference: {y.yarahub_reference_md5}\n"
        f"           {y.yarahub_reference_link if y.yarahub_reference_link else format_none('No reference link given')}"
        f"\nTLP:       Sharing  {format_tlp(y.yarahub_rule_sharing_tlp)}\n"
        f"           Matching {format_tlp(y.yarahub_rule_matching_tlp)}\n"
        f"License:   {y.yarahub_license}"
    )
    c.print(table)


def create_task_table(t: Task, *, include_sample: bool = True) -> Table:
    table = Table(show_header=False, box=MINIMAL)
    table.add_column()
    table.add_column(overflow="fold")
    table.add_row(
        "Task ID",
        t.task_id
    )
    table.add_row(
        "YARAify Parameters",
        f"ClamAV {':heavy_check_mark:' if t.clamav_scan else ':heavy_multiplication_x:'}"
        f" Unpack {':heavy_check_mark:' if t.unpack else ':heavy_multiplication_x:'}"
        f" Share {':heavy_check_mark:' if t.share_file else ':heavy_multiplication_x:'}"
    )
    if include_sample:
        table.add_row(
            "Sample",
            create_sample_table(t.metadata)
        )
    table.add_row(
        "Detections",
        f"Clam-AV:     {', '.join(t.clamav_results) if t.clamav_results else format_none('No Clam-AV results')}\n"
        + "\n".join([
            f"Name:        {r.rule_name}\n"
            f"Author:      {r.author}\n"
            f"Description: {r.description if r.description else format_none('No description provided.')}\n"
            f"TLP:         {format_tlp(r.tlp)}\n"
            for r in t.static_results
        ])
    )
    if t.unpacked_results:
        for idx, unpacked in enumerate(t.unpacked_results):
            table.add_row(
                f"Unpacked file {idx + 1}/{len(t.unpacked_results)}",
                f"Filename: {unpacked.unpacked_file_name}\n"
                f"MD5:      {unpacked.unpacked_md5}\n"
                f"SHA1:     {unpacked.unpacked_sha256}\n"
            )
            table.add_row(
                "Detections",
                "\n".join([
                    f"Name:        {r.rule_name}\n"
                    f"Author:      {r.author}\n"
                    f"Description: {r.description if r.description else format_none('No description provided.')}\n"
                    f"TLP:         {format_tlp(r.tlp)}\n"
                    for r in unpacked.unpacked_yara_matches
                ])
            )
    return table


def print_task_table(t: Task, c: Console = Console(), *, include_sample: bool = True):
    c.print(create_task_table(t, include_sample=include_sample))


def multiple_samples(samples: List[Sample], console: Console = Console()):
    for sample in samples:
        print_sample_table(sample, console)
        console.print()


def sample_csv_output(samples: List[Sample], console: Console = Console()):
    columns = Config.get_instance().csv_columns
    list_to_csv(columns, samples, console)


def yara_csv_output(rules: List[YaraRule], console: Console = Console()):
    columns = Config.get_instance().yaraify.csv_columns
    list_to_csv(columns, rules, console)


def list_to_csv(columns: Dict, l: List, console: Console = Console()):
    soft_wrap = console.soft_wrap
    console.soft_wrap = True
    console.print("\"", end="")
    console.print("\",\"".join(columns.keys()), end="")
    console.print("\"")
    for idx, item in enumerate(l):
        cols = []
        for key in columns.values():
            if not hasattr(item, key):
                col = "None"
            else:
                col = getattr(item, key)
            if not col:
                col = "None"
            elif isinstance(col, list):
                col = ",".join(col)
            col = f"\"{col}\""
            cols.append(col)
        console.print(",".join(cols))
    console.soft_wrap = soft_wrap


def get_random_color() -> str:
    return random.choice(COLORS)


def simple_sample_output(sample: Sample, tags: Dict, console: Console = Console(), include_fs: bool = False):
    if sample.signature:
        sig = f"[red1]{sample.signature}[/red1]"
    else:
        sig = format_none("Undetected")
    if sample.reporter and not sample.anonymous:
        reporter = f"[deep_sky_blue1]@{sample.reporter}[/deep_sky_blue1]"
    else:
        reporter = format_none("Anonymous")
    row = sample.sha256_hash + f" [{sig}]" + f" [{reporter}]" + " ("
    if include_fs:
        row = "[" + sample.first_seen.isoformat() + "] " + row
    if sample.tags:
        for idx, tag in enumerate(sample.tags):
            if tag not in tags:
                tags.update({
                    tag: get_random_color()
                })
            row += f"[{tags[tag]}]{tag}[/{tags[tag]}]"
            if idx < len(sample.tags) - 1:
                row += ", "
    row += ")"
    console.print(row)
