"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayRestApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const prepare_spec_1 = require("./prepare-spec-event-handler/prepare-spec");
const api_gateway_auth_1 = require("./spec/api-gateway-auth");
const api_gateway_integrations_1 = require("./spec/api-gateway-integrations");
const open_api_gateway_web_acl_1 = require("./waf/open-api-gateway-web-acl");
/**
 * A construct for creating an api gateway rest api based on the definition in the OpenAPI spec.
 */
class OpenApiGatewayRestApi extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { integrations, spec, specPath, operationLookup, defaultAuthorizer, corsOptions, ...options } = props;
        // Upload the spec to s3 as an asset
        const inputSpecAsset = new aws_s3_assets_1.Asset(this, "InputSpec", {
            path: specPath,
        });
        // We'll output the prepared spec in the same asset bucket
        const preparedSpecOutputKeyPrefix = `${inputSpecAsset.s3ObjectKey}-prepared`;
        const stack = aws_cdk_lib_1.Stack.of(this);
        const prepareSpecLambdaName = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")}PrepareSpec`;
        const prepareSpecRole = new aws_iam_1.Role(this, "PrepareSpecRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}:*`,
                            ],
                        }),
                    ],
                }),
                s3: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:getObject"],
                            resources: [
                                inputSpecAsset.bucket.arnForObjects(inputSpecAsset.s3ObjectKey),
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:putObject"],
                            resources: [
                                // The output file will include a hash of the prepared spec, which is not known until deploy time since
                                // tokens must be resolved
                                inputSpecAsset.bucket.arnForObjects(`${preparedSpecOutputKeyPrefix}/*`),
                            ],
                        }),
                    ],
                }),
            },
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(prepareSpecRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${prepareSpecLambdaName}:\*/g`,
                    },
                ],
            },
        ], true);
        // Create a custom resource for preparing the spec for deployment (adding integrations, authorizers, etc)
        const prepareSpec = new aws_lambda_1.Function(this, "PrepareSpec", {
            handler: "index.handler",
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../../lib/construct/prepare-spec-event-handler")),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            role: prepareSpecRole,
            functionName: prepareSpecLambdaName,
        });
        const providerFunctionName = `${prepareSpecLambdaName}-Provider`;
        const providerRole = new aws_iam_1.Role(this, "PrepareSpecProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "PrepareSpecProvider", {
            onEventHandler: prepareSpec,
            role: providerRole,
            providerFunctionName,
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
            {
                id: "AwsSolutions-L1",
                reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
            },
        ], true);
        const prepareSpecOptions = {
            defaultAuthorizerReference: api_gateway_auth_1.serializeAsAuthorizerReference(defaultAuthorizer),
            integrations: Object.fromEntries(Object.entries(integrations).map(([operationId, { authorizer, integration }]) => [
                operationId,
                {
                    integration: integration.render({
                        operationId,
                        scope: this,
                        ...operationLookup[operationId],
                    }),
                    methodAuthorizer: api_gateway_auth_1.serializeAsAuthorizerReference(authorizer),
                },
            ])),
            securitySchemes: api_gateway_auth_1.prepareSecuritySchemes(this, integrations, defaultAuthorizer),
            corsOptions: corsOptions && {
                allowHeaders: corsOptions.allowHeaders || aws_apigateway_1.Cors.DEFAULT_HEADERS,
                allowMethods: corsOptions.allowMethods || aws_apigateway_1.Cors.ALL_METHODS,
                allowOrigins: corsOptions.allowOrigins,
                statusCode: corsOptions.statusCode || 204,
            },
            operationLookup,
        };
        // Spec preparation will happen in a custom resource lambda so that references to lambda integrations etc can be
        // resolved. However, we also prepare inline to perform some additional validation at synth time.
        const preparedSpec = prepare_spec_1.prepareApiSpec(spec, prepareSpecOptions);
        const prepareApiSpecCustomResourceProperties = {
            inputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: inputSpecAsset.s3ObjectKey,
            },
            outputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: preparedSpecOutputKeyPrefix,
            },
            ...prepareSpecOptions,
        };
        const prepareSpecCustomResource = new aws_cdk_lib_1.CustomResource(this, "PrepareSpecCustomResource", {
            serviceToken: provider.serviceToken,
            properties: prepareApiSpecCustomResourceProperties,
        });
        // Create the api gateway resources from the spec, augmenting the spec with the properties specific to api gateway
        // such as integrations or auth types
        this.api = new aws_apigateway_1.SpecRestApi(this, id, {
            apiDefinition: aws_apigateway_1.ApiDefinition.fromBucket(inputSpecAsset.bucket, prepareSpecCustomResource.getAttString("outputSpecKey")),
            deployOptions: {
                accessLogDestination: new aws_apigateway_1.LogGroupLogDestination(new aws_logs_1.LogGroup(scope, `AccessLogs`)),
                accessLogFormat: aws_apigateway_1.AccessLogFormat.clf(),
                loggingLevel: aws_apigateway_1.MethodLoggingLevel.INFO,
            },
            ...options,
        });
        this.api.node.addDependency(prepareSpecCustomResource);
        // While the api will be updated when the output path from the custom resource changes, CDK still needs to know when
        // to redeploy the api. This is achieved by including a hash of the spec in the logical id (internalised in the
        // addToLogicalId method since this is how changes of individual resources/methods etc trigger redeployments in CDK)
        this.api.latestDeployment?.addToLogicalId(preparedSpec);
        // Grant API Gateway permission to invoke the integrations
        Object.keys(integrations).forEach((operationId) => {
            integrations[operationId].integration.grant({
                operationId,
                scope: this,
                api: this.api,
                ...operationLookup[operationId],
            });
        });
        // Grant API Gateway permission to invoke each custom authorizer lambda (if any)
        api_gateway_integrations_1.getAuthorizerFunctions(props).forEach(({ label, function: lambda }) => {
            new aws_lambda_1.CfnPermission(this, `LambdaPermission-${label}`, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: lambda.functionArn,
                sourceArn: stack.formatArn({
                    service: "execute-api",
                    resource: this.api.restApiId,
                    resourceName: "*/*",
                }),
            });
        });
        // Create and associate the web acl if not disabled
        if (!props.webAclOptions?.disable) {
            const acl = new open_api_gateway_web_acl_1.OpenApiGatewayWebAcl(this, `${id}-Acl`, {
                ...props.webAclOptions,
                apiDeploymentStageArn: this.api.deploymentStage.stageArn,
            });
            this.webAcl = acl.webAcl;
            this.ipSet = acl.ipSet;
            this.webAclAssociation = acl.webAclAssociation;
        }
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
            {
                id: "AwsSolutions-IAM4",
                reason: "Cloudwatch Role requires access to create/read groups at the root level.",
                appliesTo: [
                    {
                        regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs$/g`,
                    },
                ],
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
            {
                id: "AwsSolutions-APIG2",
                reason: "This construct implements fine grained validation via OpenApi.",
            },
        ], true);
    }
}
exports.OpenApiGatewayRestApi = OpenApiGatewayRestApi;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayRestApi[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayRestApi", version: "0.12.11" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3Blbi1hcGktZ2F0ZXdheS1yZXN0LWFwaS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9jb25zdHJ1Y3Qvb3Blbi1hcGktZ2F0ZXdheS1yZXN0LWFwaS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBO3NDQUNzQztBQUN0Qyw2QkFBNkI7QUFDN0IsMERBQXNEO0FBQ3RELDZDQUE4RDtBQUM5RCwrREFRb0M7QUFDcEMsaURBTTZCO0FBQzdCLHVEQUtnQztBQUNoQyxtREFBZ0Q7QUFDaEQsNkRBQWtEO0FBTWxELG1FQUF3RDtBQUN4RCxxQ0FBMEM7QUFDMUMsMkNBQXVDO0FBRXZDLDRFQUdtRDtBQUVuRCw4REFHaUM7QUFDakMsOEVBQXlFO0FBQ3pFLDZFQUFzRTtBQXdCdEU7O0dBRUc7QUFDSCxNQUFhLHFCQUFzQixTQUFRLHNCQUFTO0lBTWxELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBaUM7UUFDekUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLEVBQ0osWUFBWSxFQUNaLElBQUksRUFDSixRQUFRLEVBQ1IsZUFBZSxFQUNmLGlCQUFpQixFQUNqQixXQUFXLEVBQ1gsR0FBRyxPQUFPLEVBQ1gsR0FBRyxLQUFLLENBQUM7UUFFVixvQ0FBb0M7UUFDcEMsTUFBTSxjQUFjLEdBQUcsSUFBSSxxQkFBSyxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUU7WUFDbEQsSUFBSSxFQUFFLFFBQVE7U0FDZixDQUFDLENBQUM7UUFDSCwwREFBMEQ7UUFDMUQsTUFBTSwyQkFBMkIsR0FBRyxHQUFHLGNBQWMsQ0FBQyxXQUFXLFdBQVcsQ0FBQztRQUU3RSxNQUFNLEtBQUssR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU3QixNQUFNLHFCQUFxQixHQUFHLEdBQUcsZ0JBQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDO2FBQzFELEtBQUssQ0FBQyxHQUFHLENBQUM7YUFDVixJQUFJLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQztRQUMxQixNQUFNLGVBQWUsR0FBRyxJQUFJLGNBQUksQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDeEQsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMsc0JBQXNCLENBQUM7WUFDdkQsY0FBYyxFQUFFO2dCQUNkLElBQUksRUFBRSxJQUFJLHdCQUFjLENBQUM7b0JBQ3ZCLFVBQVUsRUFBRTt3QkFDVixJQUFJLHlCQUFlLENBQUM7NEJBQ2xCLE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7NEJBQ3BCLE9BQU8sRUFBRTtnQ0FDUCxxQkFBcUI7Z0NBQ3JCLHNCQUFzQjtnQ0FDdEIsbUJBQW1COzZCQUNwQjs0QkFDRCxTQUFTLEVBQUU7Z0NBQ1QsZ0JBQWdCLEtBQUssQ0FBQyxNQUFNLElBQUksS0FBSyxDQUFDLE9BQU8sMEJBQTBCLHFCQUFxQixFQUFFO2dDQUM5RixnQkFBZ0IsS0FBSyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsT0FBTywwQkFBMEIscUJBQXFCLElBQUk7NkJBQ2pHO3lCQUNGLENBQUM7cUJBQ0g7aUJBQ0YsQ0FBQztnQkFDRixFQUFFLEVBQUUsSUFBSSx3QkFBYyxDQUFDO29CQUNyQixVQUFVLEVBQUU7d0JBQ1YsSUFBSSx5QkFBZSxDQUFDOzRCQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxLQUFLOzRCQUNwQixPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7NEJBQ3pCLFNBQVMsRUFBRTtnQ0FDVCxjQUFjLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDOzZCQUNoRTt5QkFDRixDQUFDO3dCQUNGLElBQUkseUJBQWUsQ0FBQzs0QkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSzs0QkFDcEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDOzRCQUN6QixTQUFTLEVBQUU7Z0NBQ1QsdUdBQXVHO2dDQUN2RywwQkFBMEI7Z0NBQzFCLGNBQWMsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUNqQyxHQUFHLDJCQUEyQixJQUFJLENBQ25DOzZCQUNGO3lCQUNGLENBQUM7cUJBQ0g7aUJBQ0YsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgseUJBQWUsQ0FBQyx1QkFBdUIsQ0FDckMsZUFBZSxFQUNmO1lBQ0U7Z0JBQ0UsRUFBRSxFQUFFLG1CQUFtQjtnQkFDdkIsTUFBTSxFQUNKLHVJQUF1STtnQkFDekksU0FBUyxFQUFFO29CQUNUO3dCQUNFLEtBQUssRUFBRSw0QkFBNEIsZ0JBQU0sQ0FBQyxtQkFBbUIsQ0FDM0QsS0FBSyxDQUNOLElBQUksZ0JBQU0sQ0FBQyxvQkFBb0IsQ0FDOUIsS0FBSyxDQUNOLDBCQUEwQixxQkFBcUIsT0FBTztxQkFDeEQ7aUJBQ0Y7YUFDRjtTQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7UUFFRix5R0FBeUc7UUFDekcsTUFBTSxXQUFXLEdBQUcsSUFBSSxxQkFBYyxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDMUQsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLG9CQUFPLENBQUMsV0FBVztZQUM1QixJQUFJLEVBQUUsaUJBQUksQ0FBQyxTQUFTLENBQ2xCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLGdEQUFnRCxDQUFDLENBQ3ZFO1lBQ0QsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixJQUFJLEVBQUUsZUFBZTtZQUNyQixZQUFZLEVBQUUscUJBQXFCO1NBQ3BDLENBQUMsQ0FBQztRQUVILE1BQU0sb0JBQW9CLEdBQUcsR0FBRyxxQkFBcUIsV0FBVyxDQUFDO1FBQ2pFLE1BQU0sWUFBWSxHQUFHLElBQUksY0FBSSxDQUFDLElBQUksRUFBRSx5QkFBeUIsRUFBRTtZQUM3RCxTQUFTLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztZQUN2RCxjQUFjLEVBQUU7Z0JBQ2QsSUFBSSxFQUFFLElBQUksd0JBQWMsQ0FBQztvQkFDdkIsVUFBVSxFQUFFO3dCQUNWLElBQUkseUJBQWUsQ0FBQzs0QkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSzs0QkFDcEIsT0FBTyxFQUFFO2dDQUNQLHFCQUFxQjtnQ0FDckIsc0JBQXNCO2dDQUN0QixtQkFBbUI7NkJBQ3BCOzRCQUNELFNBQVMsRUFBRTtnQ0FDVCxnQkFBZ0IsS0FBSyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsT0FBTywwQkFBMEIsb0JBQW9CLEVBQUU7Z0NBQzdGLGdCQUFnQixLQUFLLENBQUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxPQUFPLDBCQUEwQixvQkFBb0IsSUFBSTs2QkFDaEc7eUJBQ0YsQ0FBQztxQkFDSDtpQkFDRixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLFFBQVEsR0FBRyxJQUFJLDJCQUFRLENBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFFO1lBQ3pELGNBQWMsRUFBRSxXQUFXO1lBQzNCLElBQUksRUFBRSxZQUFZO1lBQ2xCLG9CQUFvQjtTQUNyQixDQUFDLENBQUM7UUFFSCx5QkFBZSxDQUFDLHVCQUF1QixDQUNyQyxZQUFZLEVBQ1o7WUFDRTtnQkFDRSxFQUFFLEVBQUUsbUJBQW1CO2dCQUN2QixNQUFNLEVBQ0osdUlBQXVJO2FBQzFJO1NBQ0YsRUFDRCxJQUFJLENBQ0wsQ0FBQztRQUNGLHlCQUFlLENBQUMsdUJBQXVCLENBQ3JDLFFBQVEsRUFDUjtZQUNFO2dCQUNFLEVBQUUsRUFBRSxpQkFBaUI7Z0JBQ3JCLE1BQU0sRUFDSixtR0FBbUc7YUFDdEc7U0FDRixFQUNELElBQUksQ0FDTCxDQUFDO1FBRUYsTUFBTSxrQkFBa0IsR0FBMEI7WUFDaEQsMEJBQTBCLEVBQ3hCLGlEQUE4QixDQUFDLGlCQUFpQixDQUFDO1lBQ25ELFlBQVksRUFBRSxNQUFNLENBQUMsV0FBVyxDQUM5QixNQUFNLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDLEdBQUcsQ0FDOUIsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFLFVBQVUsRUFBRSxXQUFXLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQztnQkFDOUMsV0FBVztnQkFDWDtvQkFDRSxXQUFXLEVBQUUsV0FBVyxDQUFDLE1BQU0sQ0FBQzt3QkFDOUIsV0FBVzt3QkFDWCxLQUFLLEVBQUUsSUFBSTt3QkFDWCxHQUFHLGVBQWUsQ0FBQyxXQUFXLENBQUM7cUJBQ2hDLENBQUM7b0JBQ0YsZ0JBQWdCLEVBQUUsaURBQThCLENBQUMsVUFBVSxDQUFDO2lCQUM3RDthQUNGLENBQ0YsQ0FDRjtZQUNELGVBQWUsRUFBRSx5Q0FBc0IsQ0FDckMsSUFBSSxFQUNKLFlBQVksRUFDWixpQkFBaUIsQ0FDbEI7WUFDRCxXQUFXLEVBQUUsV0FBVyxJQUFJO2dCQUMxQixZQUFZLEVBQUUsV0FBVyxDQUFDLFlBQVksSUFBSSxxQkFBSSxDQUFDLGVBQWU7Z0JBQzlELFlBQVksRUFBRSxXQUFXLENBQUMsWUFBWSxJQUFJLHFCQUFJLENBQUMsV0FBVztnQkFDMUQsWUFBWSxFQUFFLFdBQVcsQ0FBQyxZQUFZO2dCQUN0QyxVQUFVLEVBQUUsV0FBVyxDQUFDLFVBQVUsSUFBSSxHQUFHO2FBQzFDO1lBQ0QsZUFBZTtTQUNoQixDQUFDO1FBRUYsZ0hBQWdIO1FBQ2hILGlHQUFpRztRQUNqRyxNQUFNLFlBQVksR0FBRyw2QkFBYyxDQUFDLElBQUksRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO1FBRTlELE1BQU0sc0NBQXNDLEdBQzFDO1lBQ0UsaUJBQWlCLEVBQUU7Z0JBQ2pCLE1BQU0sRUFBRSxjQUFjLENBQUMsTUFBTSxDQUFDLFVBQVU7Z0JBQ3hDLEdBQUcsRUFBRSxjQUFjLENBQUMsV0FBVzthQUNoQztZQUNELGtCQUFrQixFQUFFO2dCQUNsQixNQUFNLEVBQUUsY0FBYyxDQUFDLE1BQU0sQ0FBQyxVQUFVO2dCQUN4QyxHQUFHLEVBQUUsMkJBQTJCO2FBQ2pDO1lBQ0QsR0FBRyxrQkFBa0I7U0FDdEIsQ0FBQztRQUVKLE1BQU0seUJBQXlCLEdBQUcsSUFBSSw0QkFBYyxDQUNsRCxJQUFJLEVBQ0osMkJBQTJCLEVBQzNCO1lBQ0UsWUFBWSxFQUFFLFFBQVEsQ0FBQyxZQUFZO1lBQ25DLFVBQVUsRUFBRSxzQ0FBc0M7U0FDbkQsQ0FDRixDQUFDO1FBRUYsa0hBQWtIO1FBQ2xILHFDQUFxQztRQUNyQyxJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksNEJBQVcsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ25DLGFBQWEsRUFBRSw4QkFBYSxDQUFDLFVBQVUsQ0FDckMsY0FBYyxDQUFDLE1BQU0sRUFDckIseUJBQXlCLENBQUMsWUFBWSxDQUFDLGVBQWUsQ0FBQyxDQUN4RDtZQUNELGFBQWEsRUFBRTtnQkFDYixvQkFBb0IsRUFBRSxJQUFJLHVDQUFzQixDQUM5QyxJQUFJLG1CQUFRLENBQUMsS0FBSyxFQUFFLFlBQVksQ0FBQyxDQUNsQztnQkFDRCxlQUFlLEVBQUUsZ0NBQWUsQ0FBQyxHQUFHLEVBQUU7Z0JBQ3RDLFlBQVksRUFBRSxtQ0FBa0IsQ0FBQyxJQUFJO2FBQ3RDO1lBQ0QsR0FBRyxPQUFPO1NBQ1gsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFFdkQsb0hBQW9IO1FBQ3BILCtHQUErRztRQUMvRyxvSEFBb0g7UUFDcEgsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsRUFBRSxjQUFjLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFeEQsMERBQTBEO1FBQzFELE1BQU0sQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUU7WUFDaEQsWUFBWSxDQUFDLFdBQVcsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUM7Z0JBQzFDLFdBQVc7Z0JBQ1gsS0FBSyxFQUFFLElBQUk7Z0JBQ1gsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2dCQUNiLEdBQUcsZUFBZSxDQUFDLFdBQVcsQ0FBQzthQUNoQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILGdGQUFnRjtRQUNoRixpREFBc0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUUsTUFBTSxFQUFFLEVBQUUsRUFBRTtZQUNwRSxJQUFJLDBCQUFhLENBQUMsSUFBSSxFQUFFLG9CQUFvQixLQUFLLEVBQUUsRUFBRTtnQkFDbkQsTUFBTSxFQUFFLHVCQUF1QjtnQkFDL0IsU0FBUyxFQUFFLDBCQUEwQjtnQkFDckMsWUFBWSxFQUFFLE1BQU0sQ0FBQyxXQUFXO2dCQUNoQyxTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQztvQkFDekIsT0FBTyxFQUFFLGFBQWE7b0JBQ3RCLFFBQVEsRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLFNBQVM7b0JBQzVCLFlBQVksRUFBRSxLQUFLO2lCQUNwQixDQUFDO2FBQ0gsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxtREFBbUQ7UUFDbkQsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLEVBQUUsT0FBTyxFQUFFO1lBQ2pDLE1BQU0sR0FBRyxHQUFHLElBQUksK0NBQW9CLENBQUMsSUFBSSxFQUFFLEdBQUcsRUFBRSxNQUFNLEVBQUU7Z0JBQ3RELEdBQUcsS0FBSyxDQUFDLGFBQWE7Z0JBQ3RCLHFCQUFxQixFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDLFFBQVE7YUFDekQsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLE1BQU0sR0FBRyxHQUFHLENBQUMsTUFBTSxDQUFDO1lBQ3pCLElBQUksQ0FBQyxLQUFLLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQztZQUN2QixJQUFJLENBQUMsaUJBQWlCLEdBQUcsR0FBRyxDQUFDLGlCQUFpQixDQUFDO1NBQ2hEO1FBRUQseUJBQWUsQ0FBQyx1QkFBdUIsQ0FDckMsSUFBSSxFQUNKO1lBQ0U7Z0JBQ0UsRUFBRSxFQUFFLG1CQUFtQjtnQkFDdkIsTUFBTSxFQUNKLDBFQUEwRTtnQkFDNUUsU0FBUyxFQUFFO29CQUNUO3dCQUNFLEtBQUssRUFBRSxpQkFBaUIsZ0JBQU0sQ0FBQyxzQkFBc0IsQ0FDbkQsS0FBSyxDQUNOLHVFQUF1RTtxQkFDekU7aUJBQ0Y7YUFDRjtTQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7UUFFRix5QkFBZSxDQUFDLHVCQUF1QixDQUNyQyxJQUFJLEVBQ0o7WUFDRTtnQkFDRSxFQUFFLEVBQUUsb0JBQW9CO2dCQUN4QixNQUFNLEVBQ0osZ0VBQWdFO2FBQ25FO1NBQ0YsRUFDRCxJQUFJLENBQ0wsQ0FBQztJQUNKLENBQUM7O0FBcFRILHNEQXFUQyIsInNvdXJjZXNDb250ZW50IjpbIi8qISBDb3B5cmlnaHQgW0FtYXpvbi5jb21dKGh0dHA6Ly9hbWF6b24uY29tLyksIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMCAqL1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHsgUERLTmFnIH0gZnJvbSBcIkBhd3MtcHJvdG90eXBpbmctc2RrL3Bkay1uYWdcIjtcbmltcG9ydCB7IEN1c3RvbVJlc291cmNlLCBEdXJhdGlvbiwgU3RhY2sgfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7XG4gIEFjY2Vzc0xvZ0Zvcm1hdCxcbiAgQXBpRGVmaW5pdGlvbixcbiAgQ29ycyxcbiAgTG9nR3JvdXBMb2dEZXN0aW5hdGlvbixcbiAgTWV0aG9kTG9nZ2luZ0xldmVsLFxuICBSZXN0QXBpQmFzZVByb3BzLFxuICBTcGVjUmVzdEFwaSxcbn0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1hcGlnYXRld2F5XCI7XG5pbXBvcnQge1xuICBFZmZlY3QsXG4gIFBvbGljeURvY3VtZW50LFxuICBQb2xpY3lTdGF0ZW1lbnQsXG4gIFJvbGUsXG4gIFNlcnZpY2VQcmluY2lwYWwsXG59IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtaWFtXCI7XG5pbXBvcnQge1xuICBDZm5QZXJtaXNzaW9uLFxuICBDb2RlLFxuICBGdW5jdGlvbiBhcyBMYW1iZGFGdW5jdGlvbixcbiAgUnVudGltZSxcbn0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1sYW1iZGFcIjtcbmltcG9ydCB7IExvZ0dyb3VwIH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1sb2dzXCI7XG5pbXBvcnQgeyBBc3NldCB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtczMtYXNzZXRzXCI7XG5pbXBvcnQge1xuICBDZm5JUFNldCxcbiAgQ2ZuV2ViQUNMLFxuICBDZm5XZWJBQ0xBc3NvY2lhdGlvbixcbn0gZnJvbSBcImF3cy1jZGstbGliL2F3cy13YWZ2MlwiO1xuaW1wb3J0IHsgUHJvdmlkZXIgfSBmcm9tIFwiYXdzLWNkay1saWIvY3VzdG9tLXJlc291cmNlc1wiO1xuaW1wb3J0IHsgTmFnU3VwcHJlc3Npb25zIH0gZnJvbSBcImNkay1uYWdcIjtcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gXCJjb25zdHJ1Y3RzXCI7XG5pbXBvcnQgeyBQcmVwYXJlQXBpU3BlY0N1c3RvbVJlc291cmNlUHJvcGVydGllcyB9IGZyb20gXCIuL3ByZXBhcmUtc3BlYy1ldmVudC1oYW5kbGVyXCI7XG5pbXBvcnQge1xuICBwcmVwYXJlQXBpU3BlYyxcbiAgUHJlcGFyZUFwaVNwZWNPcHRpb25zLFxufSBmcm9tIFwiLi9wcmVwYXJlLXNwZWMtZXZlbnQtaGFuZGxlci9wcmVwYXJlLXNwZWNcIjtcbmltcG9ydCB7IE9wZW5BcGlPcHRpb25zIH0gZnJvbSBcIi4vc3BlY1wiO1xuaW1wb3J0IHtcbiAgcHJlcGFyZVNlY3VyaXR5U2NoZW1lcyxcbiAgc2VyaWFsaXplQXNBdXRob3JpemVyUmVmZXJlbmNlLFxufSBmcm9tIFwiLi9zcGVjL2FwaS1nYXRld2F5LWF1dGhcIjtcbmltcG9ydCB7IGdldEF1dGhvcml6ZXJGdW5jdGlvbnMgfSBmcm9tIFwiLi9zcGVjL2FwaS1nYXRld2F5LWludGVncmF0aW9uc1wiO1xuaW1wb3J0IHsgT3BlbkFwaUdhdGV3YXlXZWJBY2wgfSBmcm9tIFwiLi93YWYvb3Blbi1hcGktZ2F0ZXdheS13ZWItYWNsXCI7XG5pbXBvcnQgeyBPcGVuQXBpR2F0ZXdheVdlYkFjbE9wdGlvbnMgfSBmcm9tIFwiLi93YWYvdHlwZXNcIjtcblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciB0aGUgT3BlbkFwaUdhdGV3YXlSZXN0QXBpIGNvbnN0cnVjdFxuICovXG5leHBvcnQgaW50ZXJmYWNlIE9wZW5BcGlHYXRld2F5UmVzdEFwaVByb3BzXG4gIGV4dGVuZHMgUmVzdEFwaUJhc2VQcm9wcyxcbiAgICBPcGVuQXBpT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgcGFyc2VkIE9wZW5BUEkgc3BlY2lmaWNhdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgc3BlYzogYW55OyAvLyBUeXBlIGlzIE9wZW5BUElWMy5Eb2N1bWVudCAtIGhvd2V2ZXIgbm90IHRyYW5zcGlsZWFibGUgdmlhIGpzaWksIHNvIHdlIHVzZSBhbnkuXG4gIC8qKlxuICAgKiBQYXRoIHRvIHRoZSBKU09OIG9wZW4gYXBpIHNwZWNcbiAgICovXG4gIHJlYWRvbmx5IHNwZWNQYXRoOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBPcHRpb25zIGZvciB0aGUgQVdTIFdBRiB2MiBXZWJBQ0wgYXNzb2NpYXRlZCB3aXRoIHRoZSBhcGkuIEJ5IGRlZmF1bHQsIGEgV2ViIEFDTCB3aXRoIHRoZSBBV1MgZGVmYXVsdCBtYW5hZ2VkXG4gICAqIHJ1bGUgc2V0IHdpbGwgYmUgYXNzb2NpYXRlZCB3aXRoIHRoZSBBUEkuIFRoZXNlIG9wdGlvbnMgbWF5IGRpc2FibGUgb3Igb3ZlcnJpZGUgdGhlIGRlZmF1bHRzLlxuICAgKi9cbiAgcmVhZG9ubHkgd2ViQWNsT3B0aW9ucz86IE9wZW5BcGlHYXRld2F5V2ViQWNsT3B0aW9ucztcbn1cblxuLyoqXG4gKiBBIGNvbnN0cnVjdCBmb3IgY3JlYXRpbmcgYW4gYXBpIGdhdGV3YXkgcmVzdCBhcGkgYmFzZWQgb24gdGhlIGRlZmluaXRpb24gaW4gdGhlIE9wZW5BUEkgc3BlYy5cbiAqL1xuZXhwb3J0IGNsYXNzIE9wZW5BcGlHYXRld2F5UmVzdEFwaSBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHB1YmxpYyByZWFkb25seSBhcGk6IFNwZWNSZXN0QXBpO1xuICByZWFkb25seSB3ZWJBY2w/OiBDZm5XZWJBQ0w7XG4gIHJlYWRvbmx5IGlwU2V0PzogQ2ZuSVBTZXQ7XG4gIHJlYWRvbmx5IHdlYkFjbEFzc29jaWF0aW9uPzogQ2ZuV2ViQUNMQXNzb2NpYXRpb247XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IE9wZW5BcGlHYXRld2F5UmVzdEFwaVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHtcbiAgICAgIGludGVncmF0aW9ucyxcbiAgICAgIHNwZWMsXG4gICAgICBzcGVjUGF0aCxcbiAgICAgIG9wZXJhdGlvbkxvb2t1cCxcbiAgICAgIGRlZmF1bHRBdXRob3JpemVyLFxuICAgICAgY29yc09wdGlvbnMsXG4gICAgICAuLi5vcHRpb25zXG4gICAgfSA9IHByb3BzO1xuXG4gICAgLy8gVXBsb2FkIHRoZSBzcGVjIHRvIHMzIGFzIGFuIGFzc2V0XG4gICAgY29uc3QgaW5wdXRTcGVjQXNzZXQgPSBuZXcgQXNzZXQodGhpcywgXCJJbnB1dFNwZWNcIiwge1xuICAgICAgcGF0aDogc3BlY1BhdGgsXG4gICAgfSk7XG4gICAgLy8gV2UnbGwgb3V0cHV0IHRoZSBwcmVwYXJlZCBzcGVjIGluIHRoZSBzYW1lIGFzc2V0IGJ1Y2tldFxuICAgIGNvbnN0IHByZXBhcmVkU3BlY091dHB1dEtleVByZWZpeCA9IGAke2lucHV0U3BlY0Fzc2V0LnMzT2JqZWN0S2V5fS1wcmVwYXJlZGA7XG5cbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHRoaXMpO1xuXG4gICAgY29uc3QgcHJlcGFyZVNwZWNMYW1iZGFOYW1lID0gYCR7UERLTmFnLmdldFN0YWNrUHJlZml4KHN0YWNrKVxuICAgICAgLnNwbGl0KFwiL1wiKVxuICAgICAgLmpvaW4oXCItXCIpfVByZXBhcmVTcGVjYDtcbiAgICBjb25zdCBwcmVwYXJlU3BlY1JvbGUgPSBuZXcgUm9sZSh0aGlzLCBcIlByZXBhcmVTcGVjUm9sZVwiLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKFwibGFtYmRhLmFtYXpvbmF3cy5jb21cIiksXG4gICAgICBpbmxpbmVQb2xpY2llczoge1xuICAgICAgICBsb2dzOiBuZXcgUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAgIFwibG9nczpDcmVhdGVMb2dHcm91cFwiLFxuICAgICAgICAgICAgICAgIFwibG9nczpDcmVhdGVMb2dTdHJlYW1cIixcbiAgICAgICAgICAgICAgICBcImxvZ3M6UHV0TG9nRXZlbnRzXCIsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgIGBhcm46YXdzOmxvZ3M6JHtzdGFjay5yZWdpb259OiR7c3RhY2suYWNjb3VudH06bG9nLWdyb3VwOi9hd3MvbGFtYmRhLyR7cHJlcGFyZVNwZWNMYW1iZGFOYW1lfWAsXG4gICAgICAgICAgICAgICAgYGFybjphd3M6bG9nczoke3N0YWNrLnJlZ2lvbn06JHtzdGFjay5hY2NvdW50fTpsb2ctZ3JvdXA6L2F3cy9sYW1iZGEvJHtwcmVwYXJlU3BlY0xhbWJkYU5hbWV9OipgLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgfSksXG4gICAgICAgIHMzOiBuZXcgUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW1wiczM6Z2V0T2JqZWN0XCJdLFxuICAgICAgICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAgICAgICBpbnB1dFNwZWNBc3NldC5idWNrZXQuYXJuRm9yT2JqZWN0cyhpbnB1dFNwZWNBc3NldC5zM09iamVjdEtleSksXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW1wiczM6cHV0T2JqZWN0XCJdLFxuICAgICAgICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAgICAgICAvLyBUaGUgb3V0cHV0IGZpbGUgd2lsbCBpbmNsdWRlIGEgaGFzaCBvZiB0aGUgcHJlcGFyZWQgc3BlYywgd2hpY2ggaXMgbm90IGtub3duIHVudGlsIGRlcGxveSB0aW1lIHNpbmNlXG4gICAgICAgICAgICAgICAgLy8gdG9rZW5zIG11c3QgYmUgcmVzb2x2ZWRcbiAgICAgICAgICAgICAgICBpbnB1dFNwZWNBc3NldC5idWNrZXQuYXJuRm9yT2JqZWN0cyhcbiAgICAgICAgICAgICAgICAgIGAke3ByZXBhcmVkU3BlY091dHB1dEtleVByZWZpeH0vKmBcbiAgICAgICAgICAgICAgICApLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgcHJlcGFyZVNwZWNSb2xlLFxuICAgICAgW1xuICAgICAgICB7XG4gICAgICAgICAgaWQ6IFwiQXdzU29sdXRpb25zLUlBTTVcIixcbiAgICAgICAgICByZWFzb246XG4gICAgICAgICAgICBcIkNsb3Vkd2F0Y2ggcmVzb3VyY2VzIGhhdmUgYmVlbiBzY29wZWQgZG93biB0byB0aGUgTG9nR3JvdXAgbGV2ZWwsIGhvd2V2ZXIgKiBpcyBzdGlsbCBuZWVkZWQgYXMgc3RyZWFtIG5hbWVzIGFyZSBjcmVhdGVkIGp1c3QgaW4gdGltZS5cIixcbiAgICAgICAgICBhcHBsaWVzVG86IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgcmVnZXg6IGAvXlJlc291cmNlOjphcm46YXdzOmxvZ3M6JHtQREtOYWcuZ2V0U3RhY2tSZWdpb25SZWdleChcbiAgICAgICAgICAgICAgICBzdGFja1xuICAgICAgICAgICAgICApfToke1BES05hZy5nZXRTdGFja0FjY291bnRSZWdleChcbiAgICAgICAgICAgICAgICBzdGFja1xuICAgICAgICAgICAgICApfTpsb2ctZ3JvdXA6L2F3cy9sYW1iZGEvJHtwcmVwYXJlU3BlY0xhbWJkYU5hbWV9OlxcKi9nYCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICB0cnVlXG4gICAgKTtcblxuICAgIC8vIENyZWF0ZSBhIGN1c3RvbSByZXNvdXJjZSBmb3IgcHJlcGFyaW5nIHRoZSBzcGVjIGZvciBkZXBsb3ltZW50IChhZGRpbmcgaW50ZWdyYXRpb25zLCBhdXRob3JpemVycywgZXRjKVxuICAgIGNvbnN0IHByZXBhcmVTcGVjID0gbmV3IExhbWJkYUZ1bmN0aW9uKHRoaXMsIFwiUHJlcGFyZVNwZWNcIiwge1xuICAgICAgaGFuZGxlcjogXCJpbmRleC5oYW5kbGVyXCIsXG4gICAgICBydW50aW1lOiBSdW50aW1lLk5PREVKU18xNl9YLFxuICAgICAgY29kZTogQ29kZS5mcm9tQXNzZXQoXG4gICAgICAgIHBhdGguam9pbihfX2Rpcm5hbWUsIFwiLi4vLi4vbGliL2NvbnN0cnVjdC9wcmVwYXJlLXNwZWMtZXZlbnQtaGFuZGxlclwiKVxuICAgICAgKSxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLnNlY29uZHMoMzApLFxuICAgICAgcm9sZTogcHJlcGFyZVNwZWNSb2xlLFxuICAgICAgZnVuY3Rpb25OYW1lOiBwcmVwYXJlU3BlY0xhbWJkYU5hbWUsXG4gICAgfSk7XG5cbiAgICBjb25zdCBwcm92aWRlckZ1bmN0aW9uTmFtZSA9IGAke3ByZXBhcmVTcGVjTGFtYmRhTmFtZX0tUHJvdmlkZXJgO1xuICAgIGNvbnN0IHByb3ZpZGVyUm9sZSA9IG5ldyBSb2xlKHRoaXMsIFwiUHJlcGFyZVNwZWNQcm92aWRlclJvbGVcIiwge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbChcImxhbWJkYS5hbWF6b25hd3MuY29tXCIpLFxuICAgICAgaW5saW5lUG9saWNpZXM6IHtcbiAgICAgICAgbG9nczogbmV3IFBvbGljeURvY3VtZW50KHtcbiAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgZWZmZWN0OiBFZmZlY3QuQUxMT1csXG4gICAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgICBcImxvZ3M6Q3JlYXRlTG9nR3JvdXBcIixcbiAgICAgICAgICAgICAgICBcImxvZ3M6Q3JlYXRlTG9nU3RyZWFtXCIsXG4gICAgICAgICAgICAgICAgXCJsb2dzOlB1dExvZ0V2ZW50c1wiLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAgICAgICBgYXJuOmF3czpsb2dzOiR7c3RhY2sucmVnaW9ufToke3N0YWNrLmFjY291bnR9OmxvZy1ncm91cDovYXdzL2xhbWJkYS8ke3Byb3ZpZGVyRnVuY3Rpb25OYW1lfWAsXG4gICAgICAgICAgICAgICAgYGFybjphd3M6bG9nczoke3N0YWNrLnJlZ2lvbn06JHtzdGFjay5hY2NvdW50fTpsb2ctZ3JvdXA6L2F3cy9sYW1iZGEvJHtwcm92aWRlckZ1bmN0aW9uTmFtZX06KmAsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBwcm92aWRlciA9IG5ldyBQcm92aWRlcih0aGlzLCBcIlByZXBhcmVTcGVjUHJvdmlkZXJcIiwge1xuICAgICAgb25FdmVudEhhbmRsZXI6IHByZXBhcmVTcGVjLFxuICAgICAgcm9sZTogcHJvdmlkZXJSb2xlLFxuICAgICAgcHJvdmlkZXJGdW5jdGlvbk5hbWUsXG4gICAgfSk7XG5cbiAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoXG4gICAgICBwcm92aWRlclJvbGUsXG4gICAgICBbXG4gICAgICAgIHtcbiAgICAgICAgICBpZDogXCJBd3NTb2x1dGlvbnMtSUFNNVwiLFxuICAgICAgICAgIHJlYXNvbjpcbiAgICAgICAgICAgIFwiQ2xvdWR3YXRjaCByZXNvdXJjZXMgaGF2ZSBiZWVuIHNjb3BlZCBkb3duIHRvIHRoZSBMb2dHcm91cCBsZXZlbCwgaG93ZXZlciAqIGlzIHN0aWxsIG5lZWRlZCBhcyBzdHJlYW0gbmFtZXMgYXJlIGNyZWF0ZWQganVzdCBpbiB0aW1lLlwiLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIHRydWVcbiAgICApO1xuICAgIE5hZ1N1cHByZXNzaW9ucy5hZGRSZXNvdXJjZVN1cHByZXNzaW9ucyhcbiAgICAgIHByb3ZpZGVyLFxuICAgICAgW1xuICAgICAgICB7XG4gICAgICAgICAgaWQ6IFwiQXdzU29sdXRpb25zLUwxXCIsXG4gICAgICAgICAgcmVhc29uOlxuICAgICAgICAgICAgXCJMYXRlc3QgcnVudGltZSBjYW5ub3QgYmUgY29uZmlndXJlZC4gQ0RLIHdpbGwgbmVlZCB0byB1cGdyYWRlIHRoZSBQcm92aWRlciBjb25zdHJ1Y3QgYWNjb3JkaW5nbHkuXCIsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgdHJ1ZVxuICAgICk7XG5cbiAgICBjb25zdCBwcmVwYXJlU3BlY09wdGlvbnM6IFByZXBhcmVBcGlTcGVjT3B0aW9ucyA9IHtcbiAgICAgIGRlZmF1bHRBdXRob3JpemVyUmVmZXJlbmNlOlxuICAgICAgICBzZXJpYWxpemVBc0F1dGhvcml6ZXJSZWZlcmVuY2UoZGVmYXVsdEF1dGhvcml6ZXIpLFxuICAgICAgaW50ZWdyYXRpb25zOiBPYmplY3QuZnJvbUVudHJpZXMoXG4gICAgICAgIE9iamVjdC5lbnRyaWVzKGludGVncmF0aW9ucykubWFwKFxuICAgICAgICAgIChbb3BlcmF0aW9uSWQsIHsgYXV0aG9yaXplciwgaW50ZWdyYXRpb24gfV0pID0+IFtcbiAgICAgICAgICAgIG9wZXJhdGlvbklkLFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBpbnRlZ3JhdGlvbjogaW50ZWdyYXRpb24ucmVuZGVyKHtcbiAgICAgICAgICAgICAgICBvcGVyYXRpb25JZCxcbiAgICAgICAgICAgICAgICBzY29wZTogdGhpcyxcbiAgICAgICAgICAgICAgICAuLi5vcGVyYXRpb25Mb29rdXBbb3BlcmF0aW9uSWRdLFxuICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgICAgbWV0aG9kQXV0aG9yaXplcjogc2VyaWFsaXplQXNBdXRob3JpemVyUmVmZXJlbmNlKGF1dGhvcml6ZXIpLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdXG4gICAgICAgIClcbiAgICAgICksXG4gICAgICBzZWN1cml0eVNjaGVtZXM6IHByZXBhcmVTZWN1cml0eVNjaGVtZXMoXG4gICAgICAgIHRoaXMsXG4gICAgICAgIGludGVncmF0aW9ucyxcbiAgICAgICAgZGVmYXVsdEF1dGhvcml6ZXJcbiAgICAgICksXG4gICAgICBjb3JzT3B0aW9uczogY29yc09wdGlvbnMgJiYge1xuICAgICAgICBhbGxvd0hlYWRlcnM6IGNvcnNPcHRpb25zLmFsbG93SGVhZGVycyB8fCBDb3JzLkRFRkFVTFRfSEVBREVSUyxcbiAgICAgICAgYWxsb3dNZXRob2RzOiBjb3JzT3B0aW9ucy5hbGxvd01ldGhvZHMgfHwgQ29ycy5BTExfTUVUSE9EUyxcbiAgICAgICAgYWxsb3dPcmlnaW5zOiBjb3JzT3B0aW9ucy5hbGxvd09yaWdpbnMsXG4gICAgICAgIHN0YXR1c0NvZGU6IGNvcnNPcHRpb25zLnN0YXR1c0NvZGUgfHwgMjA0LFxuICAgICAgfSxcbiAgICAgIG9wZXJhdGlvbkxvb2t1cCxcbiAgICB9O1xuXG4gICAgLy8gU3BlYyBwcmVwYXJhdGlvbiB3aWxsIGhhcHBlbiBpbiBhIGN1c3RvbSByZXNvdXJjZSBsYW1iZGEgc28gdGhhdCByZWZlcmVuY2VzIHRvIGxhbWJkYSBpbnRlZ3JhdGlvbnMgZXRjIGNhbiBiZVxuICAgIC8vIHJlc29sdmVkLiBIb3dldmVyLCB3ZSBhbHNvIHByZXBhcmUgaW5saW5lIHRvIHBlcmZvcm0gc29tZSBhZGRpdGlvbmFsIHZhbGlkYXRpb24gYXQgc3ludGggdGltZS5cbiAgICBjb25zdCBwcmVwYXJlZFNwZWMgPSBwcmVwYXJlQXBpU3BlYyhzcGVjLCBwcmVwYXJlU3BlY09wdGlvbnMpO1xuXG4gICAgY29uc3QgcHJlcGFyZUFwaVNwZWNDdXN0b21SZXNvdXJjZVByb3BlcnRpZXM6IFByZXBhcmVBcGlTcGVjQ3VzdG9tUmVzb3VyY2VQcm9wZXJ0aWVzID1cbiAgICAgIHtcbiAgICAgICAgaW5wdXRTcGVjTG9jYXRpb246IHtcbiAgICAgICAgICBidWNrZXQ6IGlucHV0U3BlY0Fzc2V0LmJ1Y2tldC5idWNrZXROYW1lLFxuICAgICAgICAgIGtleTogaW5wdXRTcGVjQXNzZXQuczNPYmplY3RLZXksXG4gICAgICAgIH0sXG4gICAgICAgIG91dHB1dFNwZWNMb2NhdGlvbjoge1xuICAgICAgICAgIGJ1Y2tldDogaW5wdXRTcGVjQXNzZXQuYnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgICAga2V5OiBwcmVwYXJlZFNwZWNPdXRwdXRLZXlQcmVmaXgsXG4gICAgICAgIH0sXG4gICAgICAgIC4uLnByZXBhcmVTcGVjT3B0aW9ucyxcbiAgICAgIH07XG5cbiAgICBjb25zdCBwcmVwYXJlU3BlY0N1c3RvbVJlc291cmNlID0gbmV3IEN1c3RvbVJlc291cmNlKFxuICAgICAgdGhpcyxcbiAgICAgIFwiUHJlcGFyZVNwZWNDdXN0b21SZXNvdXJjZVwiLFxuICAgICAge1xuICAgICAgICBzZXJ2aWNlVG9rZW46IHByb3ZpZGVyLnNlcnZpY2VUb2tlbixcbiAgICAgICAgcHJvcGVydGllczogcHJlcGFyZUFwaVNwZWNDdXN0b21SZXNvdXJjZVByb3BlcnRpZXMsXG4gICAgICB9XG4gICAgKTtcblxuICAgIC8vIENyZWF0ZSB0aGUgYXBpIGdhdGV3YXkgcmVzb3VyY2VzIGZyb20gdGhlIHNwZWMsIGF1Z21lbnRpbmcgdGhlIHNwZWMgd2l0aCB0aGUgcHJvcGVydGllcyBzcGVjaWZpYyB0byBhcGkgZ2F0ZXdheVxuICAgIC8vIHN1Y2ggYXMgaW50ZWdyYXRpb25zIG9yIGF1dGggdHlwZXNcbiAgICB0aGlzLmFwaSA9IG5ldyBTcGVjUmVzdEFwaSh0aGlzLCBpZCwge1xuICAgICAgYXBpRGVmaW5pdGlvbjogQXBpRGVmaW5pdGlvbi5mcm9tQnVja2V0KFxuICAgICAgICBpbnB1dFNwZWNBc3NldC5idWNrZXQsXG4gICAgICAgIHByZXBhcmVTcGVjQ3VzdG9tUmVzb3VyY2UuZ2V0QXR0U3RyaW5nKFwib3V0cHV0U3BlY0tleVwiKVxuICAgICAgKSxcbiAgICAgIGRlcGxveU9wdGlvbnM6IHtcbiAgICAgICAgYWNjZXNzTG9nRGVzdGluYXRpb246IG5ldyBMb2dHcm91cExvZ0Rlc3RpbmF0aW9uKFxuICAgICAgICAgIG5ldyBMb2dHcm91cChzY29wZSwgYEFjY2Vzc0xvZ3NgKVxuICAgICAgICApLFxuICAgICAgICBhY2Nlc3NMb2dGb3JtYXQ6IEFjY2Vzc0xvZ0Zvcm1hdC5jbGYoKSxcbiAgICAgICAgbG9nZ2luZ0xldmVsOiBNZXRob2RMb2dnaW5nTGV2ZWwuSU5GTyxcbiAgICAgIH0sXG4gICAgICAuLi5vcHRpb25zLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hcGkubm9kZS5hZGREZXBlbmRlbmN5KHByZXBhcmVTcGVjQ3VzdG9tUmVzb3VyY2UpO1xuXG4gICAgLy8gV2hpbGUgdGhlIGFwaSB3aWxsIGJlIHVwZGF0ZWQgd2hlbiB0aGUgb3V0cHV0IHBhdGggZnJvbSB0aGUgY3VzdG9tIHJlc291cmNlIGNoYW5nZXMsIENESyBzdGlsbCBuZWVkcyB0byBrbm93IHdoZW5cbiAgICAvLyB0byByZWRlcGxveSB0aGUgYXBpLiBUaGlzIGlzIGFjaGlldmVkIGJ5IGluY2x1ZGluZyBhIGhhc2ggb2YgdGhlIHNwZWMgaW4gdGhlIGxvZ2ljYWwgaWQgKGludGVybmFsaXNlZCBpbiB0aGVcbiAgICAvLyBhZGRUb0xvZ2ljYWxJZCBtZXRob2Qgc2luY2UgdGhpcyBpcyBob3cgY2hhbmdlcyBvZiBpbmRpdmlkdWFsIHJlc291cmNlcy9tZXRob2RzIGV0YyB0cmlnZ2VyIHJlZGVwbG95bWVudHMgaW4gQ0RLKVxuICAgIHRoaXMuYXBpLmxhdGVzdERlcGxveW1lbnQ/LmFkZFRvTG9naWNhbElkKHByZXBhcmVkU3BlYyk7XG5cbiAgICAvLyBHcmFudCBBUEkgR2F0ZXdheSBwZXJtaXNzaW9uIHRvIGludm9rZSB0aGUgaW50ZWdyYXRpb25zXG4gICAgT2JqZWN0LmtleXMoaW50ZWdyYXRpb25zKS5mb3JFYWNoKChvcGVyYXRpb25JZCkgPT4ge1xuICAgICAgaW50ZWdyYXRpb25zW29wZXJhdGlvbklkXS5pbnRlZ3JhdGlvbi5ncmFudCh7XG4gICAgICAgIG9wZXJhdGlvbklkLFxuICAgICAgICBzY29wZTogdGhpcyxcbiAgICAgICAgYXBpOiB0aGlzLmFwaSxcbiAgICAgICAgLi4ub3BlcmF0aW9uTG9va3VwW29wZXJhdGlvbklkXSxcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgLy8gR3JhbnQgQVBJIEdhdGV3YXkgcGVybWlzc2lvbiB0byBpbnZva2UgZWFjaCBjdXN0b20gYXV0aG9yaXplciBsYW1iZGEgKGlmIGFueSlcbiAgICBnZXRBdXRob3JpemVyRnVuY3Rpb25zKHByb3BzKS5mb3JFYWNoKCh7IGxhYmVsLCBmdW5jdGlvbjogbGFtYmRhIH0pID0+IHtcbiAgICAgIG5ldyBDZm5QZXJtaXNzaW9uKHRoaXMsIGBMYW1iZGFQZXJtaXNzaW9uLSR7bGFiZWx9YCwge1xuICAgICAgICBhY3Rpb246IFwibGFtYmRhOkludm9rZUZ1bmN0aW9uXCIsXG4gICAgICAgIHByaW5jaXBhbDogXCJhcGlnYXRld2F5LmFtYXpvbmF3cy5jb21cIixcbiAgICAgICAgZnVuY3Rpb25OYW1lOiBsYW1iZGEuZnVuY3Rpb25Bcm4sXG4gICAgICAgIHNvdXJjZUFybjogc3RhY2suZm9ybWF0QXJuKHtcbiAgICAgICAgICBzZXJ2aWNlOiBcImV4ZWN1dGUtYXBpXCIsXG4gICAgICAgICAgcmVzb3VyY2U6IHRoaXMuYXBpLnJlc3RBcGlJZCxcbiAgICAgICAgICByZXNvdXJjZU5hbWU6IFwiKi8qXCIsXG4gICAgICAgIH0pLFxuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICAvLyBDcmVhdGUgYW5kIGFzc29jaWF0ZSB0aGUgd2ViIGFjbCBpZiBub3QgZGlzYWJsZWRcbiAgICBpZiAoIXByb3BzLndlYkFjbE9wdGlvbnM/LmRpc2FibGUpIHtcbiAgICAgIGNvbnN0IGFjbCA9IG5ldyBPcGVuQXBpR2F0ZXdheVdlYkFjbCh0aGlzLCBgJHtpZH0tQWNsYCwge1xuICAgICAgICAuLi5wcm9wcy53ZWJBY2xPcHRpb25zLFxuICAgICAgICBhcGlEZXBsb3ltZW50U3RhZ2VBcm46IHRoaXMuYXBpLmRlcGxveW1lbnRTdGFnZS5zdGFnZUFybixcbiAgICAgIH0pO1xuXG4gICAgICB0aGlzLndlYkFjbCA9IGFjbC53ZWJBY2w7XG4gICAgICB0aGlzLmlwU2V0ID0gYWNsLmlwU2V0O1xuICAgICAgdGhpcy53ZWJBY2xBc3NvY2lhdGlvbiA9IGFjbC53ZWJBY2xBc3NvY2lhdGlvbjtcbiAgICB9XG5cbiAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoXG4gICAgICB0aGlzLFxuICAgICAgW1xuICAgICAgICB7XG4gICAgICAgICAgaWQ6IFwiQXdzU29sdXRpb25zLUlBTTRcIixcbiAgICAgICAgICByZWFzb246XG4gICAgICAgICAgICBcIkNsb3Vkd2F0Y2ggUm9sZSByZXF1aXJlcyBhY2Nlc3MgdG8gY3JlYXRlL3JlYWQgZ3JvdXBzIGF0IHRoZSByb290IGxldmVsLlwiLFxuICAgICAgICAgIGFwcGxpZXNUbzogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICByZWdleDogYC9eUG9saWN5Ojphcm46JHtQREtOYWcuZ2V0U3RhY2tQYXJ0aXRpb25SZWdleChcbiAgICAgICAgICAgICAgICBzdGFja1xuICAgICAgICAgICAgICApfTppYW06OmF3czpwb2xpY3kvc2VydmljZS1yb2xlL0FtYXpvbkFQSUdhdGV3YXlQdXNoVG9DbG91ZFdhdGNoTG9ncyQvZ2AsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgdHJ1ZVxuICAgICk7XG5cbiAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoXG4gICAgICB0aGlzLFxuICAgICAgW1xuICAgICAgICB7XG4gICAgICAgICAgaWQ6IFwiQXdzU29sdXRpb25zLUFQSUcyXCIsXG4gICAgICAgICAgcmVhc29uOlxuICAgICAgICAgICAgXCJUaGlzIGNvbnN0cnVjdCBpbXBsZW1lbnRzIGZpbmUgZ3JhaW5lZCB2YWxpZGF0aW9uIHZpYSBPcGVuQXBpLlwiLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIHRydWVcbiAgICApO1xuICB9XG59XG4iXX0=