"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpApi = void 0;
const core_1 = require("@aws-cdk/core");
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const route_1 = require("./route");
const stage_1 = require("./stage");
/**
 * Create a new API Gateway HTTP API endpoint.
 * @resource AWS::ApiGatewayV2::Api
 */
class HttpApi extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const apiName = (_a = props === null || props === void 0 ? void 0 : props.apiName) !== null && _a !== void 0 ? _a : id;
        let corsConfiguration;
        if (props === null || props === void 0 ? void 0 : props.corsPreflight) {
            const cors = props.corsPreflight;
            if (cors.allowOrigins && cors.allowOrigins.includes('*') && cors.allowCredentials) {
                throw new Error("CORS preflight - allowCredentials is not supported when allowOrigin is '*'");
            }
            const { allowCredentials, allowHeaders, allowMethods, allowOrigins, exposeHeaders, maxAge, } = props.corsPreflight;
            corsConfiguration = {
                allowCredentials,
                allowHeaders,
                allowMethods,
                allowOrigins,
                exposeHeaders,
                maxAge: maxAge === null || maxAge === void 0 ? void 0 : maxAge.toSeconds(),
            };
        }
        const apiProps = {
            name: apiName,
            protocolType: 'HTTP',
            corsConfiguration,
        };
        const resource = new apigatewayv2_generated_1.CfnApi(this, 'Resource', apiProps);
        this.httpApiId = resource.ref;
        if (props === null || props === void 0 ? void 0 : props.defaultIntegration) {
            new route_1.HttpRoute(this, 'DefaultRoute', {
                httpApi: this,
                routeKey: route_1.HttpRouteKey.DEFAULT,
                integration: props.defaultIntegration,
            });
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === undefined || props.createDefaultStage === true) {
            this.defaultStage = new stage_1.HttpStage(this, 'DefaultStage', {
                httpApi: this,
                autoDeploy: true,
                domainMapping: props === null || props === void 0 ? void 0 : props.defaultDomainMapping,
            });
            // to ensure the domain is ready before creating the default stage
            if (props === null || props === void 0 ? void 0 : props.defaultDomainMapping) {
                this.defaultStage.node.addDependency(props.defaultDomainMapping.domainName);
            }
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === false && props.defaultDomainMapping) {
            throw new Error('defaultDomainMapping not supported with createDefaultStage disabled');
        }
    }
    /**
     * Import an existing HTTP API into this CDK app.
     */
    static fromApiId(scope, id, httpApiId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.httpApiId = httpApiId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Get the URL to the default stage of this API.
     * Returns `undefined` if `createDefaultStage` is unset.
     */
    get url() {
        return this.defaultStage ? this.defaultStage.url : undefined;
    }
    /**
     * Add a new stage.
     */
    addStage(id, options) {
        const stage = new stage_1.HttpStage(this, id, {
            httpApi: this,
            ...options,
        });
        return stage;
    }
    /**
     * Add multiple routes that uses the same configuration. The routes all go to the same path, but for different
     * methods.
     */
    addRoutes(options) {
        var _a;
        const methods = (_a = options.methods) !== null && _a !== void 0 ? _a : [route_1.HttpMethod.ANY];
        return methods.map((method) => new route_1.HttpRoute(this, `${method}${options.path}`, {
            httpApi: this,
            routeKey: route_1.HttpRouteKey.with(options.path, method),
            integration: options.integration,
        }));
    }
}
exports.HttpApi = HttpApi;
//# sourceMappingURL=data:application/json;base64,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