"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpRoute = exports.HttpRouteKey = exports.HttpMethod = void 0;
const core_1 = require("@aws-cdk/core");
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const integration_1 = require("./integration");
/**
 * Supported HTTP methods
 */
var HttpMethod;
(function (HttpMethod) {
    /** HTTP ANY */
    HttpMethod["ANY"] = "ANY";
    /** HTTP DELETE */
    HttpMethod["DELETE"] = "DELETE";
    /** HTTP GET */
    HttpMethod["GET"] = "GET";
    /** HTTP HEAD */
    HttpMethod["HEAD"] = "HEAD";
    /** HTTP OPTIONS */
    HttpMethod["OPTIONS"] = "OPTIONS";
    /** HTTP PATCH */
    HttpMethod["PATCH"] = "PATCH";
    /** HTTP POST */
    HttpMethod["POST"] = "POST";
    /** HTTP PUT */
    HttpMethod["PUT"] = "PUT";
})(HttpMethod = exports.HttpMethod || (exports.HttpMethod = {}));
/**
 * HTTP route in APIGateway is a combination of the HTTP method and the path component.
 * This class models that combination.
 */
class HttpRouteKey {
    constructor(key, path) {
        this.key = key;
        this.path = path;
    }
    /**
     * Create a route key with the combination of the path and the method.
     * @param method default is 'ANY'
     */
    static with(path, method) {
        if (path !== '/' && (!path.startsWith('/') || path.endsWith('/'))) {
            throw new Error('path must always start with a "/" and not end with a "/"');
        }
        return new HttpRouteKey(`${method !== null && method !== void 0 ? method : 'ANY'} ${path}`, path);
    }
}
exports.HttpRouteKey = HttpRouteKey;
/**
 * The catch-all route of the API, i.e., when no other routes match
 */
HttpRouteKey.DEFAULT = new HttpRouteKey('$default');
/**
 * Route class that creates the Route for API Gateway HTTP API
 * @resource AWS::ApiGatewayV2::Route
 */
class HttpRoute extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.httpApi = props.httpApi;
        this.path = props.routeKey.path;
        let integration;
        const config = props.integration.bind({
            route: this,
            scope: this,
        });
        integration = new integration_1.HttpIntegration(this, `${this.node.id}-Integration`, {
            httpApi: props.httpApi,
            integrationType: config.type,
            integrationUri: config.uri,
            method: config.method,
            payloadFormatVersion: config.payloadFormatVersion,
        });
        const routeProps = {
            apiId: props.httpApi.httpApiId,
            routeKey: props.routeKey.key,
            target: `integrations/${integration.integrationId}`,
        };
        const route = new apigatewayv2_generated_1.CfnRoute(this, 'Resource', routeProps);
        this.routeId = route.ref;
    }
}
exports.HttpRoute = HttpRoute;
//# sourceMappingURL=data:application/json;base64,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