"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpStage = void 0;
const core_1 = require("@aws-cdk/core");
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const api_mapping_1 = require("./api-mapping");
const DEFAULT_STAGE_NAME = '$default';
/**
 * Represents a stage where an instance of the API is deployed.
 * @resource AWS::ApiGatewayV2::Stage
 */
class HttpStage extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.stageName ? props.stageName : DEFAULT_STAGE_NAME,
        });
        new apigatewayv2_generated_1.CfnStage(this, 'Resource', {
            apiId: props.httpApi.httpApiId,
            stageName: this.physicalName,
            autoDeploy: props.autoDeploy,
        });
        this.stageName = this.physicalName;
        this.httpApi = props.httpApi;
        if (props.domainMapping) {
            new api_mapping_1.HttpApiMapping(this, `${props.domainMapping.domainName}${props.domainMapping.mappingKey}`, {
                api: props.httpApi,
                domainName: props.domainMapping.domainName,
                stage: this,
                apiMappingKey: props.domainMapping.mappingKey,
            });
            // ensure the dependency
            this.node.addDependency(props.domainMapping.domainName);
        }
    }
    /**
     * Import an existing stage into this CDK app.
     */
    static fromStageName(scope, id, stageName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.stageName = stageName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The URL to this stage.
     */
    get url() {
        const s = core_1.Stack.of(this);
        const urlPath = this.stageName === DEFAULT_STAGE_NAME ? '' : this.stageName;
        return `https://${this.httpApi.httpApiId}.execute-api.${s.region}.${s.urlSuffix}/${urlPath}`;
    }
}
exports.HttpStage = HttpStage;
//# sourceMappingURL=data:application/json;base64,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