"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
const domainName = 'example.com';
const certArn = 'arn:aws:acm:us-east-1:111111111111:certificate';
describe('ApiMapping', () => {
    test('default stage', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'Api');
        const dn = new lib_1.DomainName(stack, 'DomainName', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        new lib_1.HttpApiMapping(stack, 'Mapping', {
            api,
            domainName: dn,
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::ApiMapping', {
            ApiId: {
                Ref: 'ApiF70053CD',
            },
            DomainName: 'example.com',
            Stage: '$default',
        });
    });
    test('beta stage mapping', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'Api', {
            createDefaultStage: false,
        });
        const beta = api.addStage('beta', {
            stageName: 'beta',
        });
        const dn = new lib_1.DomainName(stack, 'DomainName', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        new lib_1.HttpApiMapping(stack, 'Mapping', {
            api,
            domainName: dn,
            stage: beta,
            apiMappingKey: 'beta',
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::ApiMapping', {
            ApiId: {
                Ref: 'ApiF70053CD',
            },
            DomainName: 'example.com',
            Stage: 'beta',
            ApiMappingKey: 'beta',
        });
    });
    test('apiMappingKey validation - empty string not allowed', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'Api');
        const dn = new lib_1.DomainName(stack, 'DomainName', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        expect(() => {
            new lib_1.HttpApiMapping(stack, 'Mapping', {
                api,
                domainName: dn,
                apiMappingKey: '',
            });
        }).toThrow(/empty string for api mapping key not allowed/);
    });
    test('apiMappingKey validation - single slash not allowed', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'Api');
        const dn = new lib_1.DomainName(stack, 'DomainName', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        expect(() => {
            new lib_1.HttpApiMapping(stack, 'Mapping', {
                api,
                domainName: dn,
                apiMappingKey: '/',
            });
        }).toThrow(/An ApiMapping key may contain only letters, numbers and one of/);
    });
    test('apiMappingKey validation - prefix slash not allowd', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'Api');
        const dn = new lib_1.DomainName(stack, 'DomainName', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        expect(() => {
            new lib_1.HttpApiMapping(stack, 'Mapping', {
                api,
                domainName: dn,
                apiMappingKey: '/foo',
            });
        }).toThrow(/An ApiMapping key may contain only letters, numbers and one of/);
    });
    test('apiMappingKey validation - slash in the middle not allowed', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'Api');
        const dn = new lib_1.DomainName(stack, 'DomainName', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        expect(() => {
            new lib_1.HttpApiMapping(stack, 'Mapping', {
                api,
                domainName: dn,
                apiMappingKey: 'foo/bar',
            });
        }).toThrow(/An ApiMapping key may contain only letters, numbers and one of/);
    });
    test('apiMappingKey validation - trailing slash not allowed', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'Api');
        const dn = new lib_1.DomainName(stack, 'DomainName', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        expect(() => {
            new lib_1.HttpApiMapping(stack, 'Mapping', {
                api,
                domainName: dn,
                apiMappingKey: 'foo/',
            });
        }).toThrow(/An ApiMapping key may contain only letters, numbers and one of/);
    });
    test('apiMappingKey validation - special character in the prefix not allowed', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'Api');
        const dn = new lib_1.DomainName(stack, 'DomainName', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        expect(() => {
            new lib_1.HttpApiMapping(stack, 'Mapping', {
                api,
                domainName: dn,
                apiMappingKey: '^foo',
            });
        }).toThrow(/An ApiMapping key may contain only letters, numbers and one of/);
    });
    test('apiMappingKey validation - multiple special character not allowed', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'Api');
        const dn = new lib_1.DomainName(stack, 'DomainName', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        expect(() => {
            new lib_1.HttpApiMapping(stack, 'Mapping', {
                api,
                domainName: dn,
                apiMappingKey: 'foo.*$',
            });
        }).toThrow(/An ApiMapping key may contain only letters, numbers and one of/);
    });
    test('import mapping', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'Api');
        const dn = new lib_1.DomainName(stack, 'DomainName', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        const mapping = new lib_1.HttpApiMapping(stack, 'Mapping', {
            api,
            domainName: dn,
        });
        const imported = lib_1.HttpApiMapping.fromHttpApiMappingAttributes(stack, 'ImportedMapping', {
            apiMappingId: mapping.apiMappingId,
        });
        expect(imported.apiMappingId).toEqual(mapping.apiMappingId);
    });
});
//# sourceMappingURL=data:application/json;base64,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