"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const assert_1 = require("@aws-cdk/assert");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
describe('HttpApi', () => {
    test('default', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'api');
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Api', {
            Name: 'api',
            ProtocolType: 'HTTP',
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Stage', {
            ApiId: stack.resolve(api.httpApiId),
            StageName: '$default',
            AutoDeploy: true,
        });
        expect(stack).not.toHaveResource('AWS::ApiGatewayV2::Route');
        expect(stack).not.toHaveResource('AWS::ApiGatewayV2::Integration');
        expect(api.url).toBeDefined();
    });
    test('import', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'api', { apiName: 'customName' });
        const imported = lib_1.HttpApi.fromApiId(stack, 'imported', api.httpApiId);
        expect(imported.httpApiId).toEqual(api.httpApiId);
    });
    test('unsetting createDefaultStage', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'api', {
            createDefaultStage: false,
        });
        expect(stack).not.toHaveResource('AWS::ApiGatewayV2::Stage');
        expect(api.url).toBeUndefined();
    });
    test('default integration', () => {
        const stack = new core_1.Stack();
        const httpApi = new lib_1.HttpApi(stack, 'api', {
            defaultIntegration: new lib_1.LambdaProxyIntegration({
                handler: new aws_lambda_1.Function(stack, 'fn', {
                    code: aws_lambda_1.Code.fromInline('foo'),
                    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                    handler: 'index.handler',
                }),
            }),
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Route', {
            ApiId: stack.resolve(httpApi.httpApiId),
            RouteKey: '$default',
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Integration', {
            ApiId: stack.resolve(httpApi.httpApiId),
        });
    });
    test('addRoutes() configures the correct routes', () => {
        const stack = new core_1.Stack();
        const httpApi = new lib_1.HttpApi(stack, 'api');
        httpApi.addRoutes({
            path: '/pets',
            methods: [lib_1.HttpMethod.GET, lib_1.HttpMethod.PATCH],
            integration: new lib_1.LambdaProxyIntegration({
                handler: new aws_lambda_1.Function(stack, 'fn', {
                    code: aws_lambda_1.Code.fromInline('foo'),
                    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                    handler: 'index.handler',
                }),
            }),
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Route', {
            ApiId: stack.resolve(httpApi.httpApiId),
            RouteKey: 'GET /pets',
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Route', {
            ApiId: stack.resolve(httpApi.httpApiId),
            RouteKey: 'PATCH /pets',
        });
    });
    test('addRoutes() creates the default method', () => {
        const stack = new core_1.Stack();
        const httpApi = new lib_1.HttpApi(stack, 'api');
        httpApi.addRoutes({
            path: '/pets',
            integration: new lib_1.LambdaProxyIntegration({
                handler: new aws_lambda_1.Function(stack, 'fn', {
                    code: aws_lambda_1.Code.fromInline('foo'),
                    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                    handler: 'index.handler',
                }),
            }),
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Route', {
            ApiId: stack.resolve(httpApi.httpApiId),
            RouteKey: 'ANY /pets',
        });
    });
    describe('cors', () => {
        test('cors is correctly configured.', () => {
            const stack = new core_1.Stack();
            new lib_1.HttpApi(stack, 'HttpApi', {
                corsPreflight: {
                    allowHeaders: ['Authorization'],
                    allowMethods: [lib_1.HttpMethod.GET, lib_1.HttpMethod.HEAD, lib_1.HttpMethod.OPTIONS, lib_1.HttpMethod.POST],
                    allowOrigins: ['*'],
                    maxAge: core_1.Duration.seconds(36400),
                },
            });
            expect(stack).toHaveResource('AWS::ApiGatewayV2::Api', {
                CorsConfiguration: {
                    AllowHeaders: ['Authorization'],
                    AllowMethods: ['GET', 'HEAD', 'OPTIONS', 'POST'],
                    AllowOrigins: ['*'],
                    MaxAge: 36400,
                },
            });
        });
        test('cors is absent when not specified.', () => {
            const stack = new core_1.Stack();
            new lib_1.HttpApi(stack, 'HttpApi');
            expect(stack).toHaveResource('AWS::ApiGatewayV2::Api', {
                CorsConfiguration: assert_1.ABSENT,
            });
        });
        test('errors when allowConfiguration is specified along with origin "*"', () => {
            const stack = new core_1.Stack();
            expect(() => new lib_1.HttpApi(stack, 'HttpApi', {
                corsPreflight: {
                    allowCredentials: true,
                    allowOrigins: ['*'],
                },
            })).toThrowError(/allowCredentials is not supported/);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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