"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../../lib");
describe('LambdaProxyIntegration', () => {
    test('default', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'HttpApi');
        const fooFn = fooFunction(stack, 'Fn');
        new lib_1.HttpRoute(stack, 'LambdaProxyRoute', {
            httpApi: api,
            integration: new lib_1.LambdaProxyIntegration({
                handler: fooFn,
            }),
            routeKey: lib_1.HttpRouteKey.with('/pets'),
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Integration', {
            IntegrationType: 'AWS_PROXY',
            IntegrationUri: stack.resolve(fooFn.functionArn),
            PayloadFormatVersion: '2.0',
        });
    });
    test('payloadFormatVersion selection', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'HttpApi');
        new lib_1.HttpRoute(stack, 'LambdaProxyRoute', {
            httpApi: api,
            integration: new lib_1.LambdaProxyIntegration({
                handler: fooFunction(stack, 'Fn'),
                payloadFormatVersion: lib_1.PayloadFormatVersion.VERSION_1_0,
            }),
            routeKey: lib_1.HttpRouteKey.with('/pets'),
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Integration', {
            PayloadFormatVersion: '1.0',
        });
    });
    test('no dependency cycles', () => {
        const app = new core_1.App();
        const lambdaStack = new core_1.Stack(app, 'lambdaStack');
        const fooFn = fooFunction(lambdaStack, 'Fn');
        const apigwStack = new core_1.Stack(app, 'apigwStack');
        new lib_1.HttpApi(apigwStack, 'httpApi', {
            defaultIntegration: new lib_1.LambdaProxyIntegration({
                handler: fooFn,
            }),
        });
        expect(() => app.synth()).not.toThrow();
    });
});
function fooFunction(stack, id) {
    return new aws_lambda_1.Function(stack, id, {
        code: aws_lambda_1.Code.fromInline('foo'),
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        handler: 'index.handler',
    });
}
//# sourceMappingURL=data:application/json;base64,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