"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = exports.ElasticsearchVersion = void 0;
const url_1 = require("url");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const elasticsearch_access_policy_1 = require("./elasticsearch-access-policy");
const elasticsearch_generated_1 = require("./elasticsearch.generated");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const perms = require("./perms");
/**
 * (experimental) Elasticsearch version.
 *
 * @experimental
 */
class ElasticsearchVersion {
    /**
     *
     * @param version Elasticsearch version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * (experimental) Custom Elasticsearch version.
     *
     * @param version custom version number.
     * @experimental
     */
    static of(version) { return new ElasticsearchVersion(version); }
}
exports.ElasticsearchVersion = ElasticsearchVersion;
/**
 * (experimental) AWS Elasticsearch 1.5.
 *
 * @experimental
 */
ElasticsearchVersion.V1_5 = ElasticsearchVersion.of('1.5');
/**
 * (experimental) AWS Elasticsearch 2.3.
 *
 * @experimental
 */
ElasticsearchVersion.V2_3 = ElasticsearchVersion.of('2.3');
/**
 * (experimental) AWS Elasticsearch 5.1.
 *
 * @experimental
 */
ElasticsearchVersion.V5_1 = ElasticsearchVersion.of('5.1');
/**
 * (experimental) AWS Elasticsearch 5.3.
 *
 * @experimental
 */
ElasticsearchVersion.V5_3 = ElasticsearchVersion.of('5.3');
/**
 * (experimental) AWS Elasticsearch 5.5.
 *
 * @experimental
 */
ElasticsearchVersion.V5_5 = ElasticsearchVersion.of('5.5');
/**
 * (experimental) AWS Elasticsearch 5.6.
 *
 * @experimental
 */
ElasticsearchVersion.V5_6 = ElasticsearchVersion.of('5.6');
/**
 * (experimental) AWS Elasticsearch 6.0.
 *
 * @experimental
 */
ElasticsearchVersion.V6_0 = ElasticsearchVersion.of('6.0');
/**
 * (experimental) AWS Elasticsearch 6.2.
 *
 * @experimental
 */
ElasticsearchVersion.V6_2 = ElasticsearchVersion.of('6.2');
/**
 * (experimental) AWS Elasticsearch 6.3.
 *
 * @experimental
 */
ElasticsearchVersion.V6_3 = ElasticsearchVersion.of('6.3');
/**
 * (experimental) AWS Elasticsearch 6.4.
 *
 * @experimental
 */
ElasticsearchVersion.V6_4 = ElasticsearchVersion.of('6.4');
/**
 * (experimental) AWS Elasticsearch 6.5.
 *
 * @experimental
 */
ElasticsearchVersion.V6_5 = ElasticsearchVersion.of('6.5');
/**
 * (experimental) AWS Elasticsearch 6.7.
 *
 * @experimental
 */
ElasticsearchVersion.V6_7 = ElasticsearchVersion.of('6.7');
/**
 * (experimental) AWS Elasticsearch 6.8.
 *
 * @experimental
 */
ElasticsearchVersion.V6_8 = ElasticsearchVersion.of('6.8');
/**
 * (experimental) AWS Elasticsearch 7.1.
 *
 * @experimental
 */
ElasticsearchVersion.V7_1 = ElasticsearchVersion.of('7.1');
/**
 * (experimental) AWS Elasticsearch 7.4.
 *
 * @experimental
 */
ElasticsearchVersion.V7_4 = ElasticsearchVersion.of('7.4');
/**
 * (experimental) AWS Elasticsearch 7.7.
 *
 * @experimental
 */
ElasticsearchVersion.V7_7 = ElasticsearchVersion.of('7.7');
/**
 * (experimental) AWS Elasticsearch 7.8.
 *
 * @experimental
 */
ElasticsearchVersion.V7_8 = ElasticsearchVersion.of('7.8');
/**
 * (experimental) AWS Elasticsearch 7.9.
 *
 * @experimental
 */
ElasticsearchVersion.V7_9 = ElasticsearchVersion.of('7.9');
/**
 * (experimental) The minimum TLS version required for traffic to the domain.
 *
 * @experimental
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * (experimental) Grant read permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant read/write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant read permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant read/write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant read permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Grant write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Grant read/write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Return the given named metric for this Domain.
     *
     * @experimental
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensions: {
                DomainName: this.domainName,
                ClientId: this.stack.account,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * (experimental) Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     * @experimental
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     * @experimental
     */
    metricClusterIndexWriteBlocked(props) {
        return this.metric('ClusterIndexWriteBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * (experimental) Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     * @experimental
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * (experimental) Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for search latency.
     *
     * @default p99 over 5 minutes
     * @experimental
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatency', { statistic: 'p99', ...props });
    }
    /**
     * (experimental) Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     * @experimental
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatency', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * (experimental) Provides an Elasticsearch domain.
 *
 * @experimental
 */
class Domain extends DomainBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29, _30, _31;
        super(scope, id, {
            physicalName: props.domainName,
        });
        const defaultInstanceType = 'r5.large.elasticsearch';
        const dedicatedMasterType = (_c = (_b = (_a = props.capacity) === null || _a === void 0 ? void 0 : _a.masterNodeInstanceType) === null || _b === void 0 ? void 0 : _b.toLowerCase()) !== null && _c !== void 0 ? _c : defaultInstanceType;
        const dedicatedMasterCount = (_e = (_d = props.capacity) === null || _d === void 0 ? void 0 : _d.masterNodes) !== null && _e !== void 0 ? _e : 0;
        const dedicatedMasterEnabled = dedicatedMasterCount > 0;
        const instanceType = (_h = (_g = (_f = props.capacity) === null || _f === void 0 ? void 0 : _f.dataNodeInstanceType) === null || _g === void 0 ? void 0 : _g.toLowerCase()) !== null && _h !== void 0 ? _h : defaultInstanceType;
        const instanceCount = (_k = (_j = props.capacity) === null || _j === void 0 ? void 0 : _j.dataNodes) !== null && _k !== void 0 ? _k : 1;
        const availabilityZoneCount = (_m = (_l = props.zoneAwareness) === null || _l === void 0 ? void 0 : _l.availabilityZoneCount) !== null && _m !== void 0 ? _m : 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = (_p = (_o = props.zoneAwareness) === null || _o === void 0 ? void 0 : _o.enabled) !== null && _p !== void 0 ? _p : ((_q = props.zoneAwareness) === null || _q === void 0 ? void 0 : _q.availabilityZoneCount) != null;
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (props.vpcOptions != null && zoneAwarenessEnabled &&
            new Set((_r = props.vpcOptions) === null || _r === void 0 ? void 0 : _r.subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        ;
        if ([dedicatedMasterType, instanceType].some(t => !t.endsWith('.elasticsearch'))) {
            throw new Error('Master and data node instance types must end with ".elasticsearch".');
        }
        const elasticsearchVersion = props.version.version;
        const elasticsearchVersionNum = parseVersion(props.version);
        if (elasticsearchVersionNum <= 7.7 &&
            ![
                1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                7.7,
            ].includes(elasticsearchVersionNum)) {
            throw new Error(`Unknown Elasticsearch version: ${elasticsearchVersion}`);
        }
        const unsignedBasicAuthEnabled = (_s = props.useUnsignedBasicAuth) !== null && _s !== void 0 ? _s : false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (((_t = props.encryptionAtRest) === null || _t === void 0 ? void 0 : _t.enabled) == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.Anyone()],
            resources: [cdk.Lazy.string({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = (_u = props.fineGrainedAccessControl) === null || _u === void 0 ? void 0 : _u.masterUserArn;
        const masterUserNameProps = (_v = props.fineGrainedAccessControl) === null || _v === void 0 ? void 0 : _v.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps !== null && masterUserNameProps !== void 0 ? masterUserNameProps : 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn !== null && masterUserArn !== void 0 ? masterUserArn : masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = (_w = props.fineGrainedAccessControl) === null || _w === void 0 ? void 0 : _w.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp !== null && masterUserPasswordProp !== void 0 ? masterUserPasswordProp : createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = (_y = (_x = props.encryptionAtRest) === null || _x === void 0 ? void 0 : _x.enabled) !== null && _y !== void 0 ? _y : (((_z = props.encryptionAtRest) === null || _z === void 0 ? void 0 : _z.kmsKey) != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = (_0 = props.nodeToNodeEncryption) !== null && _0 !== void 0 ? _0 : unsignedBasicAuthEnabled;
        const volumeSize = (_2 = (_1 = props.ebs) === null || _1 === void 0 ? void 0 : _1.volumeSize) !== null && _2 !== void 0 ? _2 : 10;
        const volumeType = (_4 = (_3 = props.ebs) === null || _3 === void 0 ? void 0 : _3.volumeType) !== null && _4 !== void 0 ? _4 : ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = (_6 = (_5 = props.ebs) === null || _5 === void 0 ? void 0 : _5.enabled) !== null && _6 !== void 0 ? _6 : true;
        const enforceHttps = (_7 = props.enforceHttps) !== null && _7 !== void 0 ? _7 : unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryInstanceType(...instanceTypes) {
            return instanceTypes.some(t => dedicatedMasterType.startsWith(t))
                && instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch version, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-features-by-version.html
        if (elasticsearchVersionNum < 5.1) {
            if (((_8 = props.logging) === null || _8 === void 0 ? void 0 : _8.slowIndexLogEnabled) || ((_9 = props.logging) === null || _9 === void 0 ? void 0 : _9.appLogEnabled)
                || ((_10 = props.logging) === null || _10 === void 0 ? void 0 : _10.slowSearchLogEnabled)) {
                throw new Error('Error and slow logs publishing requires Elasticsearch version 5.1 or later.');
            }
            if ((_11 = props.encryptionAtRest) === null || _11 === void 0 ? void 0 : _11.enabled) {
                throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later.');
            }
            if (props.cognitoKibanaAuth != null) {
                throw new Error('Cognito authentication for Kibana requires Elasticsearch version 5.1 or later.');
            }
            if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.0) {
            if (props.nodeToNodeEncryption) {
                throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.7) {
            if (unsignedBasicAuthEnabled) {
                throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later.');
            }
            if (advancedSecurityEnabled) {
                throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later.');
            }
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html
        if (isInstanceType('i3') && ebsEnabled) {
            throw new Error('I3 instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && elasticsearchVersionNum > 2.3) {
            throw new Error('The t2.micro.elasticsearch instance type supports only Elasticsearch 1.5 and 2.3.');
        }
        // Only R3 and I3 support instance storage, per
        // https://aws.amazon.com/elasticsearch-service/pricing/
        if (!ebsEnabled && !isEveryInstanceType('r3', 'i3')) {
            throw new Error('EBS volumes are required when using instance types other than r3 or i3.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        // Validate fine grained access control enabled for audit logs, per
        // https://aws.amazon.com/about-aws/whats-new/2020/09/elasticsearch-audit-logs-now-available-on-amazon-elasticsearch-service/
        if (((_12 = props.logging) === null || _12 === void 0 ? void 0 : _12.auditLogEnabled) && !advancedSecurityEnabled) {
            throw new Error('Fine-grained access control is required when audit logs publishing is enabled.');
        }
        let cfnVpcOptions;
        if (props.vpcOptions) {
            cfnVpcOptions = {
                securityGroupIds: props.vpcOptions.securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: props.vpcOptions.subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if ((_13 = props.logging) === null || _13 === void 0 ? void 0 : _13.slowSearchLogEnabled) {
            this.slowSearchLogGroup = (_14 = props.logging.slowSearchLogGroup) !== null && _14 !== void 0 ? _14 : new logs.LogGroup(this, 'SlowSearchLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if ((_15 = props.logging) === null || _15 === void 0 ? void 0 : _15.slowIndexLogEnabled) {
            this.slowIndexLogGroup = (_16 = props.logging.slowIndexLogGroup) !== null && _16 !== void 0 ? _16 : new logs.LogGroup(this, 'SlowIndexLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if ((_17 = props.logging) === null || _17 === void 0 ? void 0 : _17.appLogEnabled) {
            this.appLogGroup = (_18 = props.logging.appLogGroup) !== null && _18 !== void 0 ? _18 : new logs.LogGroup(this, 'AppLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.appLogGroup);
        }
        ;
        if ((_19 = props.logging) === null || _19 === void 0 ? void 0 : _19.auditLogEnabled) {
            this.auditLogGroup = (_20 = props.logging.auditLogGroup) !== null && _20 !== void 0 ? _20 : new logs.LogGroup(this, 'AuditLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.auditLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, `ESLogGroupPolicy${this.node.addr}`, {
                // create a cloudwatch logs resource policy name that is unique to this domain instance
                policyName: `ESLogPolicy${this.node.addr}`,
                policyStatements: [logPolicyStatement],
            });
        }
        const logPublishing = {};
        if (this.appLogGroup) {
            logPublishing.ES_APPLICATION_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.appLogGroup.logGroupArn,
            };
        }
        if (this.slowSearchLogGroup) {
            logPublishing.SEARCH_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowSearchLogGroup.logGroupArn,
            };
        }
        if (this.slowIndexLogGroup) {
            logPublishing.INDEX_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowIndexLogGroup.logGroupArn,
            };
        }
        if (this.auditLogGroup) {
            logPublishing.AUDIT_LOGS = {
                enabled: this.auditLogGroup != null,
                cloudWatchLogsLogGroupArn: (_21 = this.auditLogGroup) === null || _21 === void 0 ? void 0 : _21.logGroupArn,
            };
        }
        // Create the domain
        this.domain = new elasticsearch_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            elasticsearchVersion,
            elasticsearchClusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? (_22 = props.ebs) === null || _22 === void 0 ? void 0 : _22.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? (_24 = (_23 = props.encryptionAtRest) === null || _23 === void 0 ? void 0 : _23.kmsKey) === null || _24 === void 0 ? void 0 : _24.keyId : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: logPublishing,
            cognitoOptions: {
                enabled: props.cognitoKibanaAuth != null,
                identityPoolId: (_25 = props.cognitoKibanaAuth) === null || _25 === void 0 ? void 0 : _25.identityPoolId,
                roleArn: (_26 = props.cognitoKibanaAuth) === null || _26 === void 0 ? void 0 : _26.role.roleArn,
                userPoolId: (_27 = props.cognitoKibanaAuth) === null || _27 === void 0 ? void 0 : _27.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: (_28 = props.tlsSecurityPolicy) !== null && _28 !== void 0 ? _28 : TLSSecurityPolicy.TLS_1_0,
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: (_29 = this.masterUserPassword) === null || _29 === void 0 ? void 0 : _29.toString(),
                    },
                }
                : undefined,
        });
        if (props.enableVersionUpgrade) {
            this.domain.cfnOptions.updatePolicy = {
                ...this.domain.cfnOptions.updatePolicy,
                enableVersionUpgrade: props.enableVersionUpgrade,
            };
        }
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        const accessPolicyStatements = unsignedBasicAuthEnabled
            ? ((_30 = props.accessPolicies) !== null && _30 !== void 0 ? _30 : []).concat(unsignedAccessPolicy)
            : props.accessPolicies;
        if (accessPolicyStatements != null) {
            const accessPolicy = new elasticsearch_access_policy_1.ElasticsearchAccessPolicy(this, 'ESAccessPolicy', {
                domainName: this.domainName,
                domainArn: this.domainArn,
                accessPolicies: accessPolicyStatements,
            });
            if ((_31 = props.encryptionAtRest) === null || _31 === void 0 ? void 0 : _31.kmsKey) {
                // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/encryption-at-rest.html
                // these permissions are documented as required during domain creation.
                // while not strictly documented for updates as well, it stands to reason that an update
                // operation might require these in case the cluster uses a kms key.
                // empircal evidence shows this is indeed required: https://github.com/aws/aws-cdk/issues/11412
                accessPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                    actions: ['kms:List*', 'kms:Describe*', 'kms:CreateGrant'],
                    resources: [props.encryptionAtRest.kmsKey.keyArn],
                    effect: iam.Effect.ALLOW,
                }));
            }
            accessPolicy.node.addDependency(this.domain);
        }
    }
    /**
     * (experimental) Creates a Domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     * @experimental
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * (experimental) Creates a Domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     * @experimental
     */
    static fromDomainAttributes(scope, id, attrs) {
        const { domainArn, domainEndpoint } = attrs;
        const domainName = extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainEndpoint = domainEndpoint;
            }
        };
    }
}
exports.Domain = Domain;
/**
 * Given an Elasticsearch domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Elasticsearch domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an Elasticsearch version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The Elasticsearch version object
 */
function parseVersion(version) {
    const versionStr = version.version;
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return parseFloat(versionStr);
        }
        else {
            return parseFloat(versionStr.substring(0, secondDot));
        }
    }
    catch (error) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZG9tYWluLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZG9tYWluLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDZCQUEwQjtBQUUxQiw0REFBMkU7QUFDM0Usd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUV4QywwQ0FBMEM7QUFDMUMsOERBQThEO0FBQzlELHFDQUFxQztBQUdyQywrRUFBMEU7QUFDMUUsdUVBQXNEO0FBQ3RELDJFQUFxRTtBQUNyRSxpQ0FBaUM7Ozs7OztBQUtqQyxNQUFhLG9CQUFvQjtJQTZEL0I7OztPQUdHO0lBQ0gsWUFBb0MsT0FBZTtRQUFmLFlBQU8sR0FBUCxPQUFPLENBQVE7SUFBSSxDQUFDOzs7Ozs7O0lBTmpELE1BQU0sQ0FBQyxFQUFFLENBQUMsT0FBZSxJQUFJLE9BQU8sSUFBSSxvQkFBb0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7O0FBM0RqRixvREFrRUM7Ozs7OztBQWhFd0IseUJBQUksR0FBRyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUd0Qyx5QkFBSSxHQUFHLG9CQUFvQixDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQzs7Ozs7O0FBR3RDLHlCQUFJLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFHdEMseUJBQUksR0FBRyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUd0Qyx5QkFBSSxHQUFHLG9CQUFvQixDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQzs7Ozs7O0FBR3RDLHlCQUFJLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFHdEMseUJBQUksR0FBRyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUd0Qyx5QkFBSSxHQUFHLG9CQUFvQixDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQzs7Ozs7O0FBR3RDLHlCQUFJLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFHdEMseUJBQUksR0FBRyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUd0Qyx5QkFBSSxHQUFHLG9CQUFvQixDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQzs7Ozs7O0FBR3RDLHlCQUFJLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFHdEMseUJBQUksR0FBRyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUd0Qyx5QkFBSSxHQUFHLG9CQUFvQixDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQzs7Ozs7O0FBR3RDLHlCQUFJLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFHdEMseUJBQUksR0FBRyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUd0Qyx5QkFBSSxHQUFHLG9CQUFvQixDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQzs7Ozs7O0FBR3RDLHlCQUFJLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUE4US9ELElBQVksaUJBS1g7QUFMRCxXQUFZLGlCQUFpQjtJQUUzQiwyREFBc0MsQ0FBQTtJQUV0QywyREFBc0MsQ0FBQTtBQUN4QyxDQUFDLEVBTFcsaUJBQWlCLEdBQWpCLHlCQUFpQixLQUFqQix5QkFBaUIsUUFLNUI7QUErWUQ7O0dBRUc7QUFDSCxNQUFlLFVBQVcsU0FBUSxHQUFHLENBQUMsUUFBUTs7Ozs7OztJQVc1QyxTQUFTLENBQUMsUUFBd0I7UUFDaEMsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUNmLFFBQVEsRUFDUixLQUFLLENBQUMsZUFBZSxFQUNyQixJQUFJLENBQUMsU0FBUyxFQUNkLEdBQUcsSUFBSSxDQUFDLFNBQVMsSUFBSSxDQUN0QixDQUFDO0lBQ0osQ0FBQzs7Ozs7OztJQVFELFVBQVUsQ0FBQyxRQUF3QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxLQUFLLENBQ2YsUUFBUSxFQUNSLEtBQUssQ0FBQyxnQkFBZ0IsRUFDdEIsSUFBSSxDQUFDLFNBQVMsRUFDZCxHQUFHLElBQUksQ0FBQyxTQUFTLElBQUksQ0FDdEIsQ0FBQztJQUNKLENBQUM7Ozs7Ozs7SUFRRCxjQUFjLENBQUMsUUFBd0I7UUFDckMsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUNmLFFBQVEsRUFDUixLQUFLLENBQUMscUJBQXFCLEVBQzNCLElBQUksQ0FBQyxTQUFTLEVBQ2QsR0FBRyxJQUFJLENBQUMsU0FBUyxJQUFJLENBQ3RCLENBQUM7SUFDSixDQUFDOzs7Ozs7OztJQVNELGNBQWMsQ0FBQyxLQUFhLEVBQUUsUUFBd0I7UUFDcEQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUNmLFFBQVEsRUFDUixLQUFLLENBQUMsZUFBZSxFQUNyQixHQUFHLElBQUksQ0FBQyxTQUFTLElBQUksS0FBSyxFQUFFLEVBQzVCLEdBQUcsSUFBSSxDQUFDLFNBQVMsSUFBSSxLQUFLLElBQUksQ0FDL0IsQ0FBQztJQUNKLENBQUM7Ozs7Ozs7O0lBU0QsZUFBZSxDQUFDLEtBQWEsRUFBRSxRQUF3QjtRQUNyRCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQ2YsUUFBUSxFQUNSLEtBQUssQ0FBQyxnQkFBZ0IsRUFDdEIsR0FBRyxJQUFJLENBQUMsU0FBUyxJQUFJLEtBQUssRUFBRSxFQUM1QixHQUFHLElBQUksQ0FBQyxTQUFTLElBQUksS0FBSyxJQUFJLENBQy9CLENBQUM7SUFDSixDQUFDOzs7Ozs7OztJQVNELG1CQUFtQixDQUFDLEtBQWEsRUFBRSxRQUF3QjtRQUN6RCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQ2YsUUFBUSxFQUNSLEtBQUssQ0FBQyxxQkFBcUIsRUFDM0IsR0FBRyxJQUFJLENBQUMsU0FBUyxJQUFJLEtBQUssRUFBRSxFQUM1QixHQUFHLElBQUksQ0FBQyxTQUFTLElBQUksS0FBSyxJQUFJLENBQy9CLENBQUM7SUFDSixDQUFDOzs7Ozs7OztJQVNELGFBQWEsQ0FBQyxJQUFZLEVBQUUsUUFBd0I7UUFDbEQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUNmLFFBQVEsRUFDUixLQUFLLENBQUMsZUFBZSxFQUNyQixHQUFHLElBQUksQ0FBQyxTQUFTLElBQUksSUFBSSxFQUFFLENBQzVCLENBQUM7SUFDSixDQUFDOzs7Ozs7OztJQVNELGNBQWMsQ0FBQyxJQUFZLEVBQUUsUUFBd0I7UUFDbkQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUNmLFFBQVEsRUFDUixLQUFLLENBQUMsZ0JBQWdCLEVBQ3RCLEdBQUcsSUFBSSxDQUFDLFNBQVMsSUFBSSxJQUFJLEVBQUUsQ0FDNUIsQ0FBQztJQUNKLENBQUM7Ozs7Ozs7O0lBU0Qsa0JBQWtCLENBQUMsSUFBWSxFQUFFLFFBQXdCO1FBQ3ZELE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FDZixRQUFRLEVBQ1IsS0FBSyxDQUFDLHFCQUFxQixFQUMzQixHQUFHLElBQUksQ0FBQyxTQUFTLElBQUksSUFBSSxFQUFFLENBQzVCLENBQUM7SUFDSixDQUFDOzs7Ozs7SUFLTSxNQUFNLENBQUMsVUFBa0IsRUFBRSxLQUFxQjtRQUNyRCxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixTQUFTLEVBQUUsUUFBUTtZQUNuQixVQUFVO1lBQ1YsVUFBVSxFQUFFO2dCQUNWLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtnQkFDM0IsUUFBUSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTzthQUM3QjtZQUNELEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDcEIsQ0FBQzs7Ozs7OztJQU9NLHNCQUFzQixDQUFDLEtBQXFCO1FBQ2pELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxtQkFBbUIsRUFBRTtZQUN0QyxTQUFTLEVBQUUsMEJBQVMsQ0FBQyxPQUFPO1lBQzVCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7Ozs7Ozs7SUFPTSx5QkFBeUIsQ0FBQyxLQUFxQjtRQUNwRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsc0JBQXNCLEVBQUU7WUFDekMsU0FBUyxFQUFFLDBCQUFTLENBQUMsT0FBTztZQUM1QixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7O0lBT00sc0JBQXNCLENBQUMsS0FBcUI7UUFDakQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGtCQUFrQixFQUFFO1lBQ3JDLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7OztJQU9NLDhCQUE4QixDQUFDLEtBQXFCO1FBQ3pELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQywwQkFBMEIsRUFBRTtZQUM3QyxTQUFTLEVBQUUsMEJBQVMsQ0FBQyxPQUFPO1lBQzVCLE1BQU0sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDL0IsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7OztJQU9NLFdBQVcsQ0FBQyxLQUFxQjtRQUN0QyxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxFQUFFO1lBQzFCLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztZQUM3QixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7O0lBT00sOEJBQThCLENBQUMsS0FBcUI7UUFDekQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLDBCQUEwQixFQUFFO1lBQzdDLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7OztJQU9NLG9CQUFvQixDQUFDLEtBQXFCO1FBQy9DLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsRUFBRTtZQUNuQyxTQUFTLEVBQUUsMEJBQVMsQ0FBQyxPQUFPO1lBQzVCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7Ozs7Ozs7SUFPTSx1QkFBdUIsQ0FBQyxLQUFxQjtRQUNsRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsbUJBQW1CLEVBQUU7WUFDdEMsU0FBUyxFQUFFLDBCQUFTLENBQUMsT0FBTztZQUM1QixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7O0lBT00sMEJBQTBCLENBQUMsS0FBcUI7UUFDckQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLHNCQUFzQixFQUFFO1lBQ3pDLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7OztJQU9NLDZCQUE2QixDQUFDLEtBQXFCO1FBQ3hELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyx5QkFBeUIsRUFBRTtZQUM1QyxTQUFTLEVBQUUsMEJBQVMsQ0FBQyxPQUFPO1lBQzVCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7Ozs7Ozs7SUFPTSxpQkFBaUIsQ0FBQyxLQUFxQjtRQUM1QyxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsYUFBYSxFQUFFO1lBQ2hDLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7OztJQU9NLHdCQUF3QixDQUFDLEtBQXFCO1FBQ25ELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxvQkFBb0IsRUFBRTtZQUN2QyxTQUFTLEVBQUUsMEJBQVMsQ0FBQyxPQUFPO1lBQzVCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7Ozs7Ozs7SUFPTSx5QkFBeUIsQ0FBQyxLQUFxQjtRQUNwRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMscUJBQXFCLEVBQUU7WUFDeEMsU0FBUyxFQUFFLDBCQUFTLENBQUMsT0FBTztZQUM1QixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7O0lBT00sbUJBQW1CLENBQUMsS0FBcUI7UUFDOUMsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGVBQWUsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQ3RFLENBQUM7Ozs7Ozs7SUFPTSxxQkFBcUIsQ0FBQyxLQUFxQjtRQUNoRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsaUJBQWlCLEVBQUUsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLEdBQUcsS0FBSyxFQUFFLENBQUMsQ0FBQztJQUN4RSxDQUFDO0lBRU8sS0FBSyxDQUNYLE9BQXVCLEVBQ3ZCLGFBQXVCLEVBQ3ZCLFdBQW1CLEVBQ25CLEdBQUcsaUJBQTJCO1FBRTlCLE1BQU0sWUFBWSxHQUFHLENBQUMsV0FBVyxFQUFFLEdBQUcsaUJBQWlCLENBQUMsQ0FBQztRQUV6RCxNQUFNLEtBQUssR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztZQUNyQyxPQUFPO1lBQ1AsT0FBTyxFQUFFLGFBQWE7WUFDdEIsWUFBWTtZQUNaLEtBQUssRUFBRSxJQUFJO1NBQ1osQ0FBQyxDQUFDO1FBRUgsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0NBRUY7Ozs7OztBQXNCRCxNQUFhLE1BQU8sU0FBUSxVQUFVOzs7O0lBdUZwQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWtCOztRQUMxRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLFlBQVksRUFBRSxLQUFLLENBQUMsVUFBVTtTQUMvQixDQUFDLENBQUM7UUFFSCxNQUFNLG1CQUFtQixHQUFHLHdCQUF3QixDQUFDO1FBRXJELE1BQU0sbUJBQW1CLHFCQUN2QixLQUFLLENBQUMsUUFBUSwwQ0FBRSxzQkFBc0IsMENBQUUsV0FBVyxxQ0FDbkQsbUJBQW1CLENBQUM7UUFDdEIsTUFBTSxvQkFBb0IsZUFBRyxLQUFLLENBQUMsUUFBUSwwQ0FBRSxXQUFXLG1DQUFJLENBQUMsQ0FBQztRQUM5RCxNQUFNLHNCQUFzQixHQUFHLG9CQUFvQixHQUFHLENBQUMsQ0FBQztRQUV4RCxNQUFNLFlBQVkscUJBQ2hCLEtBQUssQ0FBQyxRQUFRLDBDQUFFLG9CQUFvQiwwQ0FBRSxXQUFXLHFDQUNqRCxtQkFBbUIsQ0FBQztRQUN0QixNQUFNLGFBQWEsZUFBRyxLQUFLLENBQUMsUUFBUSwwQ0FBRSxTQUFTLG1DQUFJLENBQUMsQ0FBQztRQUVyRCxNQUFNLHFCQUFxQixlQUN6QixLQUFLLENBQUMsYUFBYSwwQ0FBRSxxQkFBcUIsbUNBQUksQ0FBQyxDQUFDO1FBRWxELElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMscUJBQXFCLENBQUMsRUFBRTtZQUMzQyxNQUFNLElBQUksS0FBSyxDQUFDLDRFQUE0RSxDQUFDLENBQUM7U0FDL0Y7UUFFRCxNQUFNLG9CQUFvQixlQUN4QixLQUFLLENBQUMsYUFBYSwwQ0FBRSxPQUFPLG1DQUM1QixPQUFBLEtBQUssQ0FBQyxhQUFhLDBDQUFFLHFCQUFxQixLQUFJLElBQUksQ0FBQztRQUVyRCxzRkFBc0Y7UUFDdEYsSUFBSSxLQUFLLENBQUMsVUFBVSxJQUFJLElBQUksSUFBSSxvQkFBb0I7WUFDbEQsSUFBSSxHQUFHLE9BQUMsS0FBSyxDQUFDLFVBQVUsMENBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsTUFBTSxDQUFDLGdCQUFnQixFQUFFLENBQUMsSUFBSSxHQUFHLHFCQUFxQixFQUFFO1lBQzFHLE1BQU0sSUFBSSxLQUFLLENBQUMsbUZBQW1GLENBQUMsQ0FBQztTQUN0RztRQUFBLENBQUM7UUFFRixJQUFJLENBQUMsbUJBQW1CLEVBQUUsWUFBWSxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDLENBQUMsRUFBRTtZQUNoRixNQUFNLElBQUksS0FBSyxDQUFDLHFFQUFxRSxDQUFDLENBQUM7U0FDeEY7UUFFRCxNQUFNLG9CQUFvQixHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDO1FBQ25ELE1BQU0sdUJBQXVCLEdBQUcsWUFBWSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUU1RCxJQUNFLHVCQUF1QixJQUFJLEdBQUc7WUFDOUIsQ0FBQztnQkFDQyxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHO2dCQUNqQyxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRztnQkFDdEMsR0FBRzthQUNKLENBQUMsUUFBUSxDQUFDLHVCQUF1QixDQUFDLEVBQ25DO1lBQ0EsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQ0FBa0Msb0JBQW9CLEVBQUUsQ0FBQyxDQUFDO1NBQzNFO1FBRUQsTUFBTSx3QkFBd0IsU0FBRyxLQUFLLENBQUMsb0JBQW9CLG1DQUFJLEtBQUssQ0FBQztRQUVyRSxJQUFJLHdCQUF3QixFQUFFO1lBQzVCLElBQUksS0FBSyxDQUFDLFlBQVksSUFBSSxLQUFLLEVBQUU7Z0JBQy9CLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELENBQUMsQ0FBQzthQUN6RTtZQUNELElBQUksS0FBSyxDQUFDLG9CQUFvQixJQUFJLEtBQUssRUFBRTtnQkFDdkMsTUFBTSxJQUFJLEtBQUssQ0FBQyx3RUFBd0UsQ0FBQyxDQUFDO2FBQzNGO1lBQ0QsSUFBSSxPQUFBLEtBQUssQ0FBQyxnQkFBZ0IsMENBQUUsT0FBTyxLQUFJLEtBQUssRUFBRTtnQkFDNUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxtRUFBbUUsQ0FBQyxDQUFDO2FBQ3RGO1NBQ0Y7UUFFRCxNQUFNLG9CQUFvQixHQUFHLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNuRCxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRSxDQUFDLFlBQVksQ0FBQztZQUN2QixVQUFVLEVBQUUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxNQUFNLEVBQUUsQ0FBQztZQUM5QixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxTQUFTLElBQUksRUFBRSxDQUFDLENBQUM7U0FDdkUsQ0FBQyxDQUFDO1FBRUgsTUFBTSxhQUFhLFNBQUcsS0FBSyxDQUFDLHdCQUF3QiwwQ0FBRSxhQUFhLENBQUM7UUFDcEUsTUFBTSxtQkFBbUIsU0FBRyxLQUFLLENBQUMsd0JBQXdCLDBDQUFFLGNBQWMsQ0FBQztRQUMzRSx5RkFBeUY7UUFDekYsTUFBTSxjQUFjLEdBQUcsd0JBQXdCO1lBQzdDLENBQUMsQ0FBQyxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsbUJBQW1CLGFBQW5CLG1CQUFtQixjQUFuQixtQkFBbUIsR0FBSSxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQ3hFLENBQUMsQ0FBQyxtQkFBbUIsQ0FBQztRQUV4QixJQUFJLGFBQWEsSUFBSSxJQUFJLElBQUksY0FBYyxJQUFJLElBQUksRUFBRTtZQUNuRCxNQUFNLElBQUksS0FBSyxDQUFDLGtIQUFrSCxDQUFDLENBQUM7U0FDckk7UUFFRCxNQUFNLHVCQUF1QixHQUFHLENBQUMsYUFBYSxhQUFiLGFBQWEsY0FBYixhQUFhLEdBQUksY0FBYyxDQUFDLElBQUksSUFBSSxDQUFDO1FBQzFFLE1BQU0sMkJBQTJCLEdBQUcsY0FBYyxJQUFJLElBQUksQ0FBQztRQUMzRCxNQUFNLHNCQUFzQixTQUFHLEtBQUssQ0FBQyx3QkFBd0IsMENBQUUsa0JBQWtCLENBQUM7UUFDbEYsTUFBTSx3QkFBd0IsR0FBRyxHQUFvQixFQUFFO1lBQ3JELE9BQU8sSUFBSSxjQUFjLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7Z0JBQ25ELG9CQUFvQixFQUFFO29CQUNwQixvQkFBb0IsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDO3dCQUNuQyxRQUFRLEVBQUUsY0FBYztxQkFDekIsQ0FBQztvQkFDRixpQkFBaUIsRUFBRSxVQUFVO29CQUM3QixpQkFBaUIsRUFBRSxhQUFhO2lCQUNqQzthQUNGLENBQUM7aUJBQ0MsbUJBQW1CLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDckMsQ0FBQyxDQUFDO1FBQ0YsSUFBSSxDQUFDLGtCQUFrQixHQUFHLDJCQUEyQixDQUFDLENBQUM7WUFDckQsQ0FBQyxzQkFBc0IsYUFBdEIsc0JBQXNCLGNBQXRCLHNCQUFzQixHQUFJLHdCQUF3QixFQUFFLENBQUM7WUFDdEQsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUVkLE1BQU0sdUJBQXVCLGVBQzNCLEtBQUssQ0FBQyxnQkFBZ0IsMENBQUUsT0FBTyxtQ0FBSSxDQUFDLE9BQUEsS0FBSyxDQUFDLGdCQUFnQiwwQ0FBRSxNQUFNLEtBQUksSUFBSSxJQUFJLHdCQUF3QixDQUFDLENBQUM7UUFDMUcsTUFBTSwyQkFBMkIsU0FBRyxLQUFLLENBQUMsb0JBQW9CLG1DQUFJLHdCQUF3QixDQUFDO1FBQzNGLE1BQU0sVUFBVSxlQUFHLEtBQUssQ0FBQyxHQUFHLDBDQUFFLFVBQVUsbUNBQUksRUFBRSxDQUFDO1FBQy9DLE1BQU0sVUFBVSxlQUFHLEtBQUssQ0FBQyxHQUFHLDBDQUFFLFVBQVUsbUNBQUksR0FBRyxDQUFDLG1CQUFtQixDQUFDLG1CQUFtQixDQUFDO1FBQ3hGLE1BQU0sVUFBVSxlQUFHLEtBQUssQ0FBQyxHQUFHLDBDQUFFLE9BQU8sbUNBQUksSUFBSSxDQUFDO1FBQzlDLE1BQU0sWUFBWSxTQUFHLEtBQUssQ0FBQyxZQUFZLG1DQUFJLHdCQUF3QixDQUFDO1FBRXBFLFNBQVMsY0FBYyxDQUFDLENBQVM7WUFDL0IsT0FBTyxtQkFBbUIsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLElBQUksWUFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN6RSxDQUFDO1FBQUEsQ0FBQztRQUVGLFNBQVMsa0JBQWtCLENBQUMsR0FBRyxhQUF1QjtZQUNwRCxPQUFPLGFBQWEsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDNUMsQ0FBQztRQUFBLENBQUM7UUFFRixTQUFTLG1CQUFtQixDQUFDLEdBQUcsYUFBdUI7WUFDckQsT0FBTyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDO21CQUM1RCxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzNELENBQUM7UUFBQSxDQUFDO1FBRUYsb0VBQW9FO1FBQ3BFLHVHQUF1RztRQUN2RyxJQUFJLHVCQUF1QixHQUFHLEdBQUcsRUFBRTtZQUNqQyxJQUNFLE9BQUEsS0FBSyxDQUFDLE9BQU8sMENBQUUsbUJBQW1CLFlBQy9CLEtBQUssQ0FBQyxPQUFPLDBDQUFFLGFBQWEsQ0FBQTsyQkFDNUIsS0FBSyxDQUFDLE9BQU8sNENBQUUsb0JBQW9CLENBQUEsRUFDdEM7Z0JBQ0EsTUFBTSxJQUFJLEtBQUssQ0FBQyw2RUFBNkUsQ0FBQyxDQUFDO2FBQ2hHO1lBQ0QsV0FBSSxLQUFLLENBQUMsZ0JBQWdCLDRDQUFFLE9BQU8sRUFBRTtnQkFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyx5RUFBeUUsQ0FBQyxDQUFDO2FBQzVGO1lBQ0QsSUFBSSxLQUFLLENBQUMsaUJBQWlCLElBQUksSUFBSSxFQUFFO2dCQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLGdGQUFnRixDQUFDLENBQUM7YUFDbkc7WUFDRCxJQUFJLGtCQUFrQixDQUFDLElBQUksRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxFQUFFO2dCQUM5QyxNQUFNLElBQUksS0FBSyxDQUFDLCtFQUErRSxDQUFDLENBQUM7YUFDbEc7U0FDRjtRQUVELElBQUksdUJBQXVCLEdBQUcsR0FBRyxFQUFFO1lBQ2pDLElBQUksS0FBSyxDQUFDLG9CQUFvQixFQUFFO2dCQUM5QixNQUFNLElBQUksS0FBSyxDQUFDLHNFQUFzRSxDQUFDLENBQUM7YUFDekY7U0FDRjtRQUVELElBQUksdUJBQXVCLEdBQUcsR0FBRyxFQUFFO1lBQ2pDLElBQUksd0JBQXdCLEVBQUU7Z0JBQzVCLE1BQU0sSUFBSSxLQUFLLENBQUMsd0VBQXdFLENBQUMsQ0FBQzthQUMzRjtZQUNELElBQUksdUJBQXVCLEVBQUU7Z0JBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsMEVBQTBFLENBQUMsQ0FBQzthQUM3RjtTQUNGO1FBRUQsbURBQW1EO1FBQ25ELDRHQUE0RztRQUM1RyxJQUFJLGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxVQUFVLEVBQUU7WUFDdEMsTUFBTSxJQUFJLEtBQUssQ0FBQyx1REFBdUQsQ0FBQyxDQUFDO1NBQzFFO1FBRUQsSUFBSSxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLHVCQUF1QixFQUFFO1lBQ25FLE1BQU0sSUFBSSxLQUFLLENBQUMsMEVBQTBFLENBQUMsQ0FBQztTQUM3RjtRQUVELElBQUksY0FBYyxDQUFDLFVBQVUsQ0FBQyxJQUFJLHVCQUF1QixHQUFHLEdBQUcsRUFBRTtZQUMvRCxNQUFNLElBQUksS0FBSyxDQUFDLG1GQUFtRixDQUFDLENBQUM7U0FDdEc7UUFFRCwrQ0FBK0M7UUFDL0Msd0RBQXdEO1FBQ3hELElBQUksQ0FBQyxVQUFVLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLEVBQUU7WUFDbkQsTUFBTSxJQUFJLEtBQUssQ0FBQyx5RUFBeUUsQ0FBQyxDQUFDO1NBQzVGO1FBRUQsb0ZBQW9GO1FBQ3BGLHNCQUFzQjtRQUN0QixJQUFJLHVCQUF1QixFQUFFO1lBQzNCLElBQUksQ0FBQywyQkFBMkIsRUFBRTtnQkFDaEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxrRkFBa0YsQ0FBQyxDQUFDO2FBQ3JHO1lBQ0QsSUFBSSxDQUFDLHVCQUF1QixFQUFFO2dCQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLDZFQUE2RSxDQUFDLENBQUM7YUFDaEc7WUFDRCxJQUFJLENBQUMsWUFBWSxFQUFFO2dCQUNqQixNQUFNLElBQUksS0FBSyxDQUFDLHdFQUF3RSxDQUFDLENBQUM7YUFDM0Y7U0FDRjtRQUVELG1FQUFtRTtRQUNuRSw2SEFBNkg7UUFDN0gsSUFBSSxRQUFBLEtBQUssQ0FBQyxPQUFPLDRDQUFFLGVBQWUsS0FBSSxDQUFDLHVCQUF1QixFQUFFO1lBQzlELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0ZBQWdGLENBQUMsQ0FBQztTQUNuRztRQUVELElBQUksYUFBdUQsQ0FBQztRQUM1RCxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDcEIsYUFBYSxHQUFHO2dCQUNkLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQztnQkFDakYsU0FBUyxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQzthQUNyRSxDQUFDO1NBQ0g7UUFFRCxnQkFBZ0I7UUFDaEIsTUFBTSxTQUFTLEdBQXFCLEVBQUUsQ0FBQztRQUV2QyxXQUFJLEtBQUssQ0FBQyxPQUFPLDRDQUFFLG9CQUFvQixFQUFFO1lBQ3ZDLElBQUksQ0FBQyxrQkFBa0IsVUFBRyxLQUFLLENBQUMsT0FBTyxDQUFDLGtCQUFrQixxQ0FDeEQsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtnQkFDeEMsU0FBUyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUzthQUN4QyxDQUFDLENBQUM7WUFFTCxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1NBQ3pDO1FBQUEsQ0FBQztRQUVGLFdBQUksS0FBSyxDQUFDLE9BQU8sNENBQUUsbUJBQW1CLEVBQUU7WUFDdEMsSUFBSSxDQUFDLGlCQUFpQixVQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsaUJBQWlCLHFDQUN0RCxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLGVBQWUsRUFBRTtnQkFDdkMsU0FBUyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUzthQUN4QyxDQUFDLENBQUM7WUFFTCxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1NBQ3hDO1FBQUEsQ0FBQztRQUVGLFdBQUksS0FBSyxDQUFDLE9BQU8sNENBQUUsYUFBYSxFQUFFO1lBQ2hDLElBQUksQ0FBQyxXQUFXLFVBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxXQUFXLHFDQUMxQyxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtnQkFDakMsU0FBUyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUzthQUN4QyxDQUFDLENBQUM7WUFFTCxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUNsQztRQUFBLENBQUM7UUFFRixXQUFJLEtBQUssQ0FBQyxPQUFPLDRDQUFFLGVBQWUsRUFBRTtZQUNsQyxJQUFJLENBQUMsYUFBYSxVQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsYUFBYSxxQ0FDNUMsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUU7Z0JBQ25DLFNBQVMsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVM7YUFDeEMsQ0FBQyxDQUFDO1lBRVAsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDcEM7UUFBQSxDQUFDO1FBRUYsSUFBSSxzQkFBc0IsR0FBa0MsSUFBSSxDQUFDO1FBQ2pFLElBQUksU0FBUyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDeEIsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ2pELE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7Z0JBQ3hCLE9BQU8sRUFBRSxDQUFDLG1CQUFtQixFQUFFLHNCQUFzQixDQUFDO2dCQUN0RCxTQUFTLEVBQUUsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQztnQkFDaEQsVUFBVSxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsa0JBQWtCLENBQUMsQ0FBQzthQUMzRCxDQUFDLENBQUM7WUFFSCx1R0FBdUc7WUFDdkcsNkNBQTZDO1lBQzdDLHNCQUFzQixHQUFHLElBQUksa0RBQXNCLENBQUMsSUFBSSxFQUFFLG1CQUFtQixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxFQUFFO2dCQUM3Rix1RkFBdUY7Z0JBQ3ZGLFVBQVUsRUFBRSxjQUFjLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFO2dCQUMxQyxnQkFBZ0IsRUFBRSxDQUFDLGtCQUFrQixDQUFDO2FBQ3ZDLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSxhQUFhLEdBQXdCLEVBQUUsQ0FBQztRQUU5QyxJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUU7WUFDcEIsYUFBYSxDQUFDLG1CQUFtQixHQUFHO2dCQUNsQyxPQUFPLEVBQUUsSUFBSTtnQkFDYix5QkFBeUIsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLFdBQVc7YUFDeEQsQ0FBQztTQUNIO1FBRUQsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDM0IsYUFBYSxDQUFDLGdCQUFnQixHQUFHO2dCQUMvQixPQUFPLEVBQUUsSUFBSTtnQkFDYix5QkFBeUIsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsV0FBVzthQUMvRCxDQUFDO1NBQ0g7UUFFRCxJQUFJLElBQUksQ0FBQyxpQkFBaUIsRUFBRTtZQUMxQixhQUFhLENBQUMsZUFBZSxHQUFHO2dCQUM5QixPQUFPLEVBQUUsSUFBSTtnQkFDYix5QkFBeUIsRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsV0FBVzthQUM5RCxDQUFDO1NBQ0g7UUFFRCxJQUFJLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDdEIsYUFBYSxDQUFDLFVBQVUsR0FBRztnQkFDekIsT0FBTyxFQUFFLElBQUksQ0FBQyxhQUFhLElBQUksSUFBSTtnQkFDbkMseUJBQXlCLFNBQUUsSUFBSSxDQUFDLGFBQWEsNENBQUUsV0FBVzthQUMzRCxDQUFDO1NBQ0g7UUFFRCxvQkFBb0I7UUFDcEIsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLG1DQUFTLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUM1QyxVQUFVLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDN0Isb0JBQW9CO1lBQ3BCLDBCQUEwQixFQUFFO2dCQUMxQixzQkFBc0I7Z0JBQ3RCLG9CQUFvQixFQUFFLHNCQUFzQjtvQkFDMUMsQ0FBQyxDQUFDLG9CQUFvQjtvQkFDdEIsQ0FBQyxDQUFDLFNBQVM7Z0JBQ2IsbUJBQW1CLEVBQUUsc0JBQXNCO29CQUN6QyxDQUFDLENBQUMsbUJBQW1CO29CQUNyQixDQUFDLENBQUMsU0FBUztnQkFDYixhQUFhO2dCQUNiLFlBQVk7Z0JBQ1osb0JBQW9CO2dCQUNwQixtQkFBbUIsRUFBRSxvQkFBb0I7b0JBQ3ZDLENBQUMsQ0FBQyxFQUFFLHFCQUFxQixFQUFFO29CQUMzQixDQUFDLENBQUMsU0FBUzthQUNkO1lBQ0QsVUFBVSxFQUFFO2dCQUNWLFVBQVU7Z0JBQ1YsVUFBVSxFQUFFLFVBQVUsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUMvQyxVQUFVLEVBQUUsVUFBVSxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQy9DLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQyxRQUFDLEtBQUssQ0FBQyxHQUFHLDRDQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsU0FBUzthQUMvQztZQUNELHVCQUF1QixFQUFFO2dCQUN2QixPQUFPLEVBQUUsdUJBQXVCO2dCQUNoQyxRQUFRLEVBQUUsdUJBQXVCO29CQUMvQixDQUFDLGVBQUMsS0FBSyxDQUFDLGdCQUFnQiw0Q0FBRSxNQUFNLDRDQUFFLEtBQUssQ0FDdkMsQ0FBQyxDQUFDLFNBQVM7YUFDZDtZQUNELDJCQUEyQixFQUFFLEVBQUUsT0FBTyxFQUFFLDJCQUEyQixFQUFFO1lBQ3JFLG9CQUFvQixFQUFFLGFBQWE7WUFDbkMsY0FBYyxFQUFFO2dCQUNkLE9BQU8sRUFBRSxLQUFLLENBQUMsaUJBQWlCLElBQUksSUFBSTtnQkFDeEMsY0FBYyxTQUFFLEtBQUssQ0FBQyxpQkFBaUIsNENBQUUsY0FBYztnQkFDdkQsT0FBTyxTQUFFLEtBQUssQ0FBQyxpQkFBaUIsNENBQUUsSUFBSSxDQUFDLE9BQU87Z0JBQzlDLFVBQVUsU0FBRSxLQUFLLENBQUMsaUJBQWlCLDRDQUFFLFVBQVU7YUFDaEQ7WUFDRCxVQUFVLEVBQUUsYUFBYTtZQUN6QixlQUFlLEVBQUUsS0FBSyxDQUFDLDBCQUEwQjtnQkFDL0MsQ0FBQyxDQUFDLEVBQUUsMEJBQTBCLEVBQUUsS0FBSyxDQUFDLDBCQUEwQixFQUFFO2dCQUNsRSxDQUFDLENBQUMsU0FBUztZQUNiLHFCQUFxQixFQUFFO2dCQUNyQixZQUFZO2dCQUNaLGlCQUFpQixTQUFFLEtBQUssQ0FBQyxpQkFBaUIscUNBQUksaUJBQWlCLENBQUMsT0FBTzthQUN4RTtZQUNELHVCQUF1QixFQUFFLHVCQUF1QjtnQkFDOUMsQ0FBQyxDQUFDO29CQUNBLE9BQU8sRUFBRSxJQUFJO29CQUNiLDJCQUEyQjtvQkFDM0IsaUJBQWlCLEVBQUU7d0JBQ2pCLGFBQWEsRUFBRSxhQUFhO3dCQUM1QixjQUFjLEVBQUUsY0FBYzt3QkFDOUIsa0JBQWtCLFNBQUUsSUFBSSxDQUFDLGtCQUFrQiw0Q0FBRSxRQUFRLEVBQUU7cUJBQ3hEO2lCQUNGO2dCQUNELENBQUMsQ0FBQyxTQUFTO1NBQ2QsQ0FBQyxDQUFDO1FBRUgsSUFBSSxLQUFLLENBQUMsb0JBQW9CLEVBQUU7WUFDOUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsWUFBWSxHQUFHO2dCQUNwQyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLFlBQVk7Z0JBQ3RDLG9CQUFvQixFQUFFLEtBQUssQ0FBQyxvQkFBb0I7YUFDakQsQ0FBQztTQUNIO1FBRUQsSUFBSSxzQkFBc0IsRUFBRTtZQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1NBQUU7UUFFdkYsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFO1lBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMseUJBQXlCLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQUU7UUFFN0YsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVqRSxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7UUFFdEUsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLEVBQUU7WUFDakUsT0FBTyxFQUFFLElBQUk7WUFDYixRQUFRLEVBQUUsUUFBUTtZQUNsQixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7U0FDaEMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxzQkFBc0IsR0FBc0Msd0JBQXdCO1lBQ3hGLENBQUMsQ0FBQyxRQUFDLEtBQUssQ0FBQyxjQUFjLHFDQUFJLEVBQUUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQztZQUMzRCxDQUFDLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztRQUV6QixJQUFJLHNCQUFzQixJQUFJLElBQUksRUFBRTtZQUNsQyxNQUFNLFlBQVksR0FBRyxJQUFJLHVEQUF5QixDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtnQkFDekUsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVO2dCQUMzQixTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVM7Z0JBQ3pCLGNBQWMsRUFBRSxzQkFBc0I7YUFDdkMsQ0FBQyxDQUFDO1lBRUgsV0FBSSxLQUFLLENBQUMsZ0JBQWdCLDRDQUFFLE1BQU0sRUFBRTtnQkFFbEMsa0dBQWtHO2dCQUVsRyx1RUFBdUU7Z0JBQ3ZFLHdGQUF3RjtnQkFDeEYsb0VBQW9FO2dCQUNwRSwrRkFBK0Y7Z0JBQy9GLFlBQVksQ0FBQyxjQUFjLENBQUMsb0JBQW9CLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUN2RSxPQUFPLEVBQUUsQ0FBQyxXQUFXLEVBQUUsZUFBZSxFQUFFLGlCQUFpQixDQUFDO29CQUMxRCxTQUFTLEVBQUUsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQztvQkFDakQsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztpQkFDekIsQ0FBQyxDQUFDLENBQUM7YUFDTDtZQUVELFlBQVksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUM5QztJQUNILENBQUM7Ozs7Ozs7OztJQXBlTSxNQUFNLENBQUMsa0JBQWtCLENBQzlCLEtBQWdCLEVBQ2hCLEVBQVUsRUFDVixjQUFzQjtRQUV0QixNQUFNLEtBQUssR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNsQyxNQUFNLFVBQVUsR0FBRyx1QkFBdUIsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUMzRCxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1lBQ2hDLE9BQU8sRUFBRSxJQUFJO1lBQ2IsUUFBUSxFQUFFLFFBQVE7WUFDbEIsWUFBWSxFQUFFLFVBQVU7U0FDekIsQ0FBQyxDQUFDO1FBRUgsT0FBTyxNQUFNLENBQUMsb0JBQW9CLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUM1QyxTQUFTO1lBQ1QsY0FBYztTQUNmLENBQUMsQ0FBQztJQUNMLENBQUM7Ozs7Ozs7OztJQVNNLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1QjtRQUN0RixNQUFNLEVBQUUsU0FBUyxFQUFFLGNBQWMsRUFBRSxHQUFHLEtBQUssQ0FBQztRQUM1QyxNQUFNLFVBQVUsR0FBRyx1QkFBdUIsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUUzRCxPQUFPLElBQUksS0FBTSxTQUFRLFVBQVU7WUFLakM7Z0JBQWdCLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7Z0JBSmpCLGNBQVMsR0FBRyxTQUFTLENBQUM7Z0JBQ3RCLGVBQVUsR0FBRyxVQUFVLENBQUM7Z0JBQ3hCLG1CQUFjLEdBQUcsY0FBYyxDQUFDO1lBRWQsQ0FBQztTQUNwQyxDQUFDO0lBQ0osQ0FBQztDQWdjRjtBQTdlRCx3QkE2ZUM7QUFFRDs7Ozs7Ozs7Ozs7OztHQWFHO0FBQ0gsU0FBUyx1QkFBdUIsQ0FBQyxjQUFzQjtJQUNyRCxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsSUFBSSxTQUFHLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDN0MsTUFBTSxNQUFNLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN0QyxNQUFNLE1BQU0sR0FBRyxHQUFHLEdBQUcsTUFBTSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNwRCxPQUFPLE1BQU0sQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFDakMsQ0FBQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFTLFlBQVksQ0FBQyxPQUE2QjtJQUNqRCxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsT0FBTyxDQUFDO0lBQ25DLE1BQU0sUUFBUSxHQUFHLFVBQVUsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFekMsSUFBSSxRQUFRLEdBQUcsQ0FBQyxFQUFFO1FBQ2hCLE1BQU0sSUFBSSxLQUFLLENBQUMsa0NBQWtDLFVBQVUscUVBQXFFLENBQUMsQ0FBQztLQUNwSTtJQUVELE1BQU0sU0FBUyxHQUFHLFVBQVUsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLFFBQVEsR0FBRyxDQUFDLENBQUMsQ0FBQztJQUV4RCxJQUFJO1FBQ0YsSUFBSSxTQUFTLElBQUksQ0FBQyxDQUFDLEVBQUU7WUFDbkIsT0FBTyxVQUFVLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDL0I7YUFBTTtZQUNMLE9BQU8sVUFBVSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLFNBQVMsQ0FBQyxDQUFDLENBQUM7U0FDdkQ7S0FDRjtJQUFDLE9BQU8sS0FBSyxFQUFFO1FBQ2QsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQ0FBa0MsVUFBVSxxRUFBcUUsQ0FBQyxDQUFDO0tBQ3BJO0FBQ0gsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IFVSTCB9IGZyb20gJ3VybCc7XG5cbmltcG9ydCB7IE1ldHJpYywgTWV0cmljT3B0aW9ucywgU3RhdGlzdGljIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3Vkd2F0Y2gnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMga21zIGZyb20gJ0Bhd3MtY2RrL2F3cy1rbXMnO1xuaW1wb3J0ICogYXMgbG9ncyBmcm9tICdAYXdzLWNkay9hd3MtbG9ncyc7XG5pbXBvcnQgKiBhcyBzZWNyZXRzbWFuYWdlciBmcm9tICdAYXdzLWNkay9hd3Mtc2VjcmV0c21hbmFnZXInO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5cbmltcG9ydCB7IEVsYXN0aWNzZWFyY2hBY2Nlc3NQb2xpY3kgfSBmcm9tICcuL2VsYXN0aWNzZWFyY2gtYWNjZXNzLXBvbGljeSc7XG5pbXBvcnQgeyBDZm5Eb21haW4gfSBmcm9tICcuL2VsYXN0aWNzZWFyY2guZ2VuZXJhdGVkJztcbmltcG9ydCB7IExvZ0dyb3VwUmVzb3VyY2VQb2xpY3kgfSBmcm9tICcuL2xvZy1ncm91cC1yZXNvdXJjZS1wb2xpY3knO1xuaW1wb3J0ICogYXMgcGVybXMgZnJvbSAnLi9wZXJtcyc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgRWxhc3RpY3NlYXJjaFZlcnNpb24ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjFfNSA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCcxLjUnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjJfMyA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCcyLjMnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjVfMSA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc1LjEnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjVfMyA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc1LjMnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjVfNSA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc1LjUnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjVfNiA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc1LjYnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjZfMCA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc2LjAnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjZfMiA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc2LjInKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjZfMyA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc2LjMnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjZfNCA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc2LjQnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjZfNSA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc2LjUnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjZfNyA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc2LjcnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjZfOCA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc2LjgnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjdfMSA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc3LjEnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjdfNCA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc3LjQnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjdfNyA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc3LjcnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjdfOCA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc3LjgnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjdfOSA9IEVsYXN0aWNzZWFyY2hWZXJzaW9uLm9mKCc3LjknKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgb2YodmVyc2lvbjogc3RyaW5nKSB7IHJldHVybiBuZXcgRWxhc3RpY3NlYXJjaFZlcnNpb24odmVyc2lvbik7IH1cblxuICAvKipcbiAgICpcbiAgICogQHBhcmFtIHZlcnNpb24gRWxhc3RpY3NlYXJjaCB2ZXJzaW9uIG51bWJlclxuICAgKi9cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgdmVyc2lvbjogc3RyaW5nKSB7IH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQ2FwYWNpdHlDb25maWcge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtYXN0ZXJOb2Rlcz86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtYXN0ZXJOb2RlSW5zdGFuY2VUeXBlPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGF0YU5vZGVzPzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRhdGFOb2RlSW5zdGFuY2VUeXBlPzogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFpvbmVBd2FyZW5lc3NDb25maWcge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5hYmxlZD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhdmFpbGFiaWxpdHlab25lQ291bnQ/OiBudW1iZXI7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEVic09wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbmFibGVkPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW9wcz86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2b2x1bWVTaXplPzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZvbHVtZVR5cGU/OiBlYzIuRWJzRGV2aWNlVm9sdW1lVHlwZTtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgTG9nZ2luZ09wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzbG93U2VhcmNoTG9nRW5hYmxlZD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzbG93U2VhcmNoTG9nR3JvdXA/OiBsb2dzLklMb2dHcm91cDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNsb3dJbmRleExvZ0VuYWJsZWQ/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNsb3dJbmRleExvZ0dyb3VwPzogbG9ncy5JTG9nR3JvdXA7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhcHBMb2dFbmFibGVkPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFwcExvZ0dyb3VwPzogbG9ncy5JTG9nR3JvdXA7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYXVkaXRMb2dFbmFibGVkPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYXVkaXRMb2dHcm91cD86IGxvZ3MuSUxvZ0dyb3VwO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEVuY3J5cHRpb25BdFJlc3RPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5hYmxlZD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBrbXNLZXk/OiBrbXMuSUtleTtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBDb2duaXRvT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpZGVudGl0eVBvb2xJZDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByb2xlOiBpYW0uSVJvbGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdXNlclBvb2xJZDogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFZwY09wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBzOiBlYzIuSVNlY3VyaXR5R3JvdXBbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc3VibmV0czogZWMyLklTdWJuZXRbXTtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gVExTU2VjdXJpdHlQb2xpY3kge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgVExTXzFfMCA9ICdQb2xpY3ktTWluLVRMUy0xLTAtMjAxOS0wNycsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBUTFNfMV8yID0gJ1BvbGljeS1NaW4tVExTLTEtMi0yMDE5LTA3J1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEFkdmFuY2VkU2VjdXJpdHlPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtYXN0ZXJVc2VyQXJuPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1hc3RlclVzZXJOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtYXN0ZXJVc2VyUGFzc3dvcmQ/OiBjZGsuU2VjcmV0VmFsdWU7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBEb21haW5Qcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYWNjZXNzUG9saWNpZXM/OiBpYW0uUG9saWN5U3RhdGVtZW50W107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYWR2YW5jZWRPcHRpb25zPzogeyBba2V5OiBzdHJpbmddOiAoc3RyaW5nKSB9O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvZ25pdG9LaWJhbmFBdXRoPzogQ29nbml0b09wdGlvbnM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZG9tYWluTmFtZT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVicz86IEVic09wdGlvbnM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNhcGFjaXR5PzogQ2FwYWNpdHlDb25maWc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB6b25lQXdhcmVuZXNzPzogWm9uZUF3YXJlbmVzc0NvbmZpZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdmVyc2lvbjogRWxhc3RpY3NlYXJjaFZlcnNpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5jcnlwdGlvbkF0UmVzdD86IEVuY3J5cHRpb25BdFJlc3RPcHRpb25zO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbG9nZ2luZz86IExvZ2dpbmdPcHRpb25zO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG5vZGVUb05vZGVFbmNyeXB0aW9uPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGF1dG9tYXRlZFNuYXBzaG90U3RhcnRIb3VyPzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGNPcHRpb25zPzogVnBjT3B0aW9ucztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbmZvcmNlSHR0cHM/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdGxzU2VjdXJpdHlQb2xpY3k/OiBUTFNTZWN1cml0eVBvbGljeTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBmaW5lR3JhaW5lZEFjY2Vzc0NvbnRyb2w/OiBBZHZhbmNlZFNlY3VyaXR5T3B0aW9ucztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHVzZVVuc2lnbmVkQmFzaWNBdXRoPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbmFibGVWZXJzaW9uVXBncmFkZT86IGJvb2xlYW47XG5cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIElEb21haW4gZXh0ZW5kcyBjZGsuSVJlc291cmNlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZG9tYWluQXJuOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkb21haW5OYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkb21haW5FbmRwb2ludDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgZ3JhbnRSZWFkKGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBncmFudFdyaXRlKGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGdyYW50UmVhZFdyaXRlKGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGdyYW50SW5kZXhSZWFkKGluZGV4OiBzdHJpbmcsIGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBncmFudEluZGV4V3JpdGUoaW5kZXg6IHN0cmluZywgaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgZ3JhbnRJbmRleFJlYWRXcml0ZShpbmRleDogc3RyaW5nLCBpZGVudGl0eTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGdyYW50UGF0aFJlYWQocGF0aDogc3RyaW5nLCBpZGVudGl0eTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBncmFudFBhdGhXcml0ZShwYXRoOiBzdHJpbmcsIGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgZ3JhbnRQYXRoUmVhZFdyaXRlKHBhdGg6IHN0cmluZywgaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgbWV0cmljKG1ldHJpY05hbWU6IHN0cmluZywgcHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgbWV0cmljQ2x1c3RlclN0YXR1c1JlZChwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBtZXRyaWNDbHVzdGVyU3RhdHVzWWVsbG93KHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBtZXRyaWNGcmVlU3RvcmFnZVNwYWNlKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBtZXRyaWNDbHVzdGVySW5kZXhXcml0ZUJsb2NrZWQocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgbWV0cmljTm9kZXMocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBtZXRyaWNBdXRvbWF0ZWRTbmFwc2hvdEZhaWx1cmUocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBtZXRyaWNDUFVVdGlsaXphdGlvbihwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBtZXRyaWNKVk1NZW1vcnlQcmVzc3VyZShwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBtZXRyaWNNYXN0ZXJDUFVVdGlsaXphdGlvbihwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgbWV0cmljTWFzdGVySlZNTWVtb3J5UHJlc3N1cmUocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIG1ldHJpY0tNU0tleUVycm9yKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBtZXRyaWNLTVNLZXlJbmFjY2Vzc2libGUocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBtZXRyaWNTZWFyY2hhYmxlRG9jdW1lbnRzKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIG1ldHJpY1NlYXJjaExhdGVuY3kocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBtZXRyaWNJbmRleGluZ0xhdGVuY3kocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xufVxuXG5cbi8qKlxuICogQSBuZXcgb3IgaW1wb3J0ZWQgZG9tYWluLlxuICovXG5hYnN0cmFjdCBjbGFzcyBEb21haW5CYXNlIGV4dGVuZHMgY2RrLlJlc291cmNlIGltcGxlbWVudHMgSURvbWFpbiB7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBkb21haW5Bcm46IHN0cmluZztcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGRvbWFpbkVuZHBvaW50OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBncmFudFJlYWQoaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChcbiAgICAgIGlkZW50aXR5LFxuICAgICAgcGVybXMuRVNfUkVBRF9BQ1RJT05TLFxuICAgICAgdGhpcy5kb21haW5Bcm4sXG4gICAgICBgJHt0aGlzLmRvbWFpbkFybn0vKmAsXG4gICAgKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGdyYW50V3JpdGUoaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChcbiAgICAgIGlkZW50aXR5LFxuICAgICAgcGVybXMuRVNfV1JJVEVfQUNUSU9OUyxcbiAgICAgIHRoaXMuZG9tYWluQXJuLFxuICAgICAgYCR7dGhpcy5kb21haW5Bcm59LypgLFxuICAgICk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGdyYW50UmVhZFdyaXRlKGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoXG4gICAgICBpZGVudGl0eSxcbiAgICAgIHBlcm1zLkVTX1JFQURfV1JJVEVfQUNUSU9OUyxcbiAgICAgIHRoaXMuZG9tYWluQXJuLFxuICAgICAgYCR7dGhpcy5kb21haW5Bcm59LypgLFxuICAgICk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGdyYW50SW5kZXhSZWFkKGluZGV4OiBzdHJpbmcsIGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoXG4gICAgICBpZGVudGl0eSxcbiAgICAgIHBlcm1zLkVTX1JFQURfQUNUSU9OUyxcbiAgICAgIGAke3RoaXMuZG9tYWluQXJufS8ke2luZGV4fWAsXG4gICAgICBgJHt0aGlzLmRvbWFpbkFybn0vJHtpbmRleH0vKmAsXG4gICAgKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGdyYW50SW5kZXhXcml0ZShpbmRleDogc3RyaW5nLCBpZGVudGl0eTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIHJldHVybiB0aGlzLmdyYW50KFxuICAgICAgaWRlbnRpdHksXG4gICAgICBwZXJtcy5FU19XUklURV9BQ1RJT05TLFxuICAgICAgYCR7dGhpcy5kb21haW5Bcm59LyR7aW5kZXh9YCxcbiAgICAgIGAke3RoaXMuZG9tYWluQXJufS8ke2luZGV4fS8qYCxcbiAgICApO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBncmFudEluZGV4UmVhZFdyaXRlKGluZGV4OiBzdHJpbmcsIGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoXG4gICAgICBpZGVudGl0eSxcbiAgICAgIHBlcm1zLkVTX1JFQURfV1JJVEVfQUNUSU9OUyxcbiAgICAgIGAke3RoaXMuZG9tYWluQXJufS8ke2luZGV4fWAsXG4gICAgICBgJHt0aGlzLmRvbWFpbkFybn0vJHtpbmRleH0vKmAsXG4gICAgKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBncmFudFBhdGhSZWFkKHBhdGg6IHN0cmluZywgaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChcbiAgICAgIGlkZW50aXR5LFxuICAgICAgcGVybXMuRVNfUkVBRF9BQ1RJT05TLFxuICAgICAgYCR7dGhpcy5kb21haW5Bcm59LyR7cGF0aH1gLFxuICAgICk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGdyYW50UGF0aFdyaXRlKHBhdGg6IHN0cmluZywgaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChcbiAgICAgIGlkZW50aXR5LFxuICAgICAgcGVybXMuRVNfV1JJVEVfQUNUSU9OUyxcbiAgICAgIGAke3RoaXMuZG9tYWluQXJufS8ke3BhdGh9YCxcbiAgICApO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGdyYW50UGF0aFJlYWRXcml0ZShwYXRoOiBzdHJpbmcsIGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoXG4gICAgICBpZGVudGl0eSxcbiAgICAgIHBlcm1zLkVTX1JFQURfV1JJVEVfQUNUSU9OUyxcbiAgICAgIGAke3RoaXMuZG9tYWluQXJufS8ke3BhdGh9YCxcbiAgICApO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgbWV0cmljKG1ldHJpY05hbWU6IHN0cmluZywgcHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gbmV3IE1ldHJpYyh7XG4gICAgICBuYW1lc3BhY2U6ICdBV1MvRVMnLFxuICAgICAgbWV0cmljTmFtZSxcbiAgICAgIGRpbWVuc2lvbnM6IHtcbiAgICAgICAgRG9tYWluTmFtZTogdGhpcy5kb21haW5OYW1lLFxuICAgICAgICBDbGllbnRJZDogdGhpcy5zdGFjay5hY2NvdW50LFxuICAgICAgfSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pLmF0dGFjaFRvKHRoaXMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgbWV0cmljQ2x1c3RlclN0YXR1c1JlZChwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnQ2x1c3RlclN0YXR1cy5yZWQnLCB7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBtZXRyaWNDbHVzdGVyU3RhdHVzWWVsbG93KHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdDbHVzdGVyU3RhdHVzLnllbGxvdycsIHtcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLk1BWElNVU0sXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBtZXRyaWNGcmVlU3RvcmFnZVNwYWNlKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdGcmVlU3RvcmFnZVNwYWNlJywge1xuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuTUlOSU1VTSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG1ldHJpY0NsdXN0ZXJJbmRleFdyaXRlQmxvY2tlZChwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnQ2x1c3RlckluZGV4V3JpdGVCbG9ja2VkJywge1xuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuTUFYSU1VTSxcbiAgICAgIHBlcmlvZDogY2RrLkR1cmF0aW9uLm1pbnV0ZXMoMSksXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG1ldHJpY05vZGVzKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdOb2RlcycsIHtcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLk1JTklNVU0sXG4gICAgICBwZXJpb2Q6IGNkay5EdXJhdGlvbi5ob3VycygxKSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBtZXRyaWNBdXRvbWF0ZWRTbmFwc2hvdEZhaWx1cmUocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ0F1dG9tYXRlZFNuYXBzaG90RmFpbHVyZScsIHtcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLk1BWElNVU0sXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgbWV0cmljQ1BVVXRpbGl6YXRpb24ocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ0NQVVV0aWxpemF0aW9uJywge1xuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuTUFYSU1VTSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgbWV0cmljSlZNTWVtb3J5UHJlc3N1cmUocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ0pWTU1lbW9yeVByZXNzdXJlJywge1xuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuTUFYSU1VTSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgbWV0cmljTWFzdGVyQ1BVVXRpbGl6YXRpb24ocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ01hc3RlckNQVVV0aWxpemF0aW9uJywge1xuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuTUFYSU1VTSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG1ldHJpY01hc3RlckpWTU1lbW9yeVByZXNzdXJlKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdNYXN0ZXJKVk1NZW1vcnlQcmVzc3VyZScsIHtcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLk1BWElNVU0sXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBtZXRyaWNLTVNLZXlFcnJvcihwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnS01TS2V5RXJyb3InLCB7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgbWV0cmljS01TS2V5SW5hY2Nlc3NpYmxlKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdLTVNLZXlJbmFjY2Vzc2libGUnLCB7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG1ldHJpY1NlYXJjaGFibGVEb2N1bWVudHMocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ1NlYXJjaGFibGVEb2N1bWVudHMnLCB7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBtZXRyaWNTZWFyY2hMYXRlbmN5KHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdTZWFyY2hMYXRlbmN5JywgeyBzdGF0aXN0aWM6ICdwOTknLCAuLi5wcm9wcyB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgbWV0cmljSW5kZXhpbmdMYXRlbmN5KHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdJbmRleGluZ0xhdGVuY3knLCB7IHN0YXRpc3RpYzogJ3A5OScsIC4uLnByb3BzIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBncmFudChcbiAgICBncmFudGVlOiBpYW0uSUdyYW50YWJsZSxcbiAgICBkb21haW5BY3Rpb25zOiBzdHJpbmdbXSxcbiAgICByZXNvdXJjZUFybjogc3RyaW5nLFxuICAgIC4uLm90aGVyUmVzb3VyY2VBcm5zOiBzdHJpbmdbXVxuICApOiBpYW0uR3JhbnQge1xuICAgIGNvbnN0IHJlc291cmNlQXJucyA9IFtyZXNvdXJjZUFybiwgLi4ub3RoZXJSZXNvdXJjZUFybnNdO1xuXG4gICAgY29uc3QgZ3JhbnQgPSBpYW0uR3JhbnQuYWRkVG9QcmluY2lwYWwoe1xuICAgICAgZ3JhbnRlZSxcbiAgICAgIGFjdGlvbnM6IGRvbWFpbkFjdGlvbnMsXG4gICAgICByZXNvdXJjZUFybnMsXG4gICAgICBzY29wZTogdGhpcyxcbiAgICB9KTtcblxuICAgIHJldHVybiBncmFudDtcbiAgfVxuXG59XG5cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIERvbWFpbkF0dHJpYnV0ZXMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZG9tYWluQXJuOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRvbWFpbkVuZHBvaW50OiBzdHJpbmc7XG59XG5cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBEb21haW4gZXh0ZW5kcyBEb21haW5CYXNlIGltcGxlbWVudHMgSURvbWFpbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21Eb21haW5FbmRwb2ludChcbiAgICBzY29wZTogQ29uc3RydWN0LFxuICAgIGlkOiBzdHJpbmcsXG4gICAgZG9tYWluRW5kcG9pbnQ6IHN0cmluZyxcbiAgKTogSURvbWFpbiB7XG4gICAgY29uc3Qgc3RhY2sgPSBjZGsuU3RhY2sub2Yoc2NvcGUpO1xuICAgIGNvbnN0IGRvbWFpbk5hbWUgPSBleHRyYWN0TmFtZUZyb21FbmRwb2ludChkb21haW5FbmRwb2ludCk7XG4gICAgY29uc3QgZG9tYWluQXJuID0gc3RhY2suZm9ybWF0QXJuKHtcbiAgICAgIHNlcnZpY2U6ICdlcycsXG4gICAgICByZXNvdXJjZTogJ2RvbWFpbicsXG4gICAgICByZXNvdXJjZU5hbWU6IGRvbWFpbk5hbWUsXG4gICAgfSk7XG5cbiAgICByZXR1cm4gRG9tYWluLmZyb21Eb21haW5BdHRyaWJ1dGVzKHNjb3BlLCBpZCwge1xuICAgICAgZG9tYWluQXJuLFxuICAgICAgZG9tYWluRW5kcG9pbnQsXG4gICAgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbURvbWFpbkF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IERvbWFpbkF0dHJpYnV0ZXMpOiBJRG9tYWluIHtcbiAgICBjb25zdCB7IGRvbWFpbkFybiwgZG9tYWluRW5kcG9pbnQgfSA9IGF0dHJzO1xuICAgIGNvbnN0IGRvbWFpbk5hbWUgPSBleHRyYWN0TmFtZUZyb21FbmRwb2ludChkb21haW5FbmRwb2ludCk7XG5cbiAgICByZXR1cm4gbmV3IGNsYXNzIGV4dGVuZHMgRG9tYWluQmFzZSB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluQXJuID0gZG9tYWluQXJuO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGRvbWFpbk5hbWUgPSBkb21haW5OYW1lO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGRvbWFpbkVuZHBvaW50ID0gZG9tYWluRW5kcG9pbnQ7XG5cbiAgICAgIGNvbnN0cnVjdG9yKCkgeyBzdXBlcihzY29wZSwgaWQpOyB9XG4gICAgfTtcbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSBkb21haW5Bcm46IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGRvbWFpbkVuZHBvaW50OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgc2xvd1NlYXJjaExvZ0dyb3VwPzogbG9ncy5JTG9nR3JvdXA7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBzbG93SW5kZXhMb2dHcm91cD86IGxvZ3MuSUxvZ0dyb3VwO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGFwcExvZ0dyb3VwPzogbG9ncy5JTG9nR3JvdXA7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgYXVkaXRMb2dHcm91cD86IGxvZ3MuSUxvZ0dyb3VwO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgbWFzdGVyVXNlclBhc3N3b3JkPzogY2RrLlNlY3JldFZhbHVlO1xuXG5cbiAgcHJpdmF0ZSByZWFkb25seSBkb21haW46IENmbkRvbWFpbjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRG9tYWluUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuZG9tYWluTmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGRlZmF1bHRJbnN0YW5jZVR5cGUgPSAncjUubGFyZ2UuZWxhc3RpY3NlYXJjaCc7XG5cbiAgICBjb25zdCBkZWRpY2F0ZWRNYXN0ZXJUeXBlID1cbiAgICAgIHByb3BzLmNhcGFjaXR5Py5tYXN0ZXJOb2RlSW5zdGFuY2VUeXBlPy50b0xvd2VyQ2FzZSgpID8/XG4gICAgICBkZWZhdWx0SW5zdGFuY2VUeXBlO1xuICAgIGNvbnN0IGRlZGljYXRlZE1hc3RlckNvdW50ID0gcHJvcHMuY2FwYWNpdHk/Lm1hc3Rlck5vZGVzID8/IDA7XG4gICAgY29uc3QgZGVkaWNhdGVkTWFzdGVyRW5hYmxlZCA9IGRlZGljYXRlZE1hc3RlckNvdW50ID4gMDtcblxuICAgIGNvbnN0IGluc3RhbmNlVHlwZSA9XG4gICAgICBwcm9wcy5jYXBhY2l0eT8uZGF0YU5vZGVJbnN0YW5jZVR5cGU/LnRvTG93ZXJDYXNlKCkgPz9cbiAgICAgIGRlZmF1bHRJbnN0YW5jZVR5cGU7XG4gICAgY29uc3QgaW5zdGFuY2VDb3VudCA9IHByb3BzLmNhcGFjaXR5Py5kYXRhTm9kZXMgPz8gMTtcblxuICAgIGNvbnN0IGF2YWlsYWJpbGl0eVpvbmVDb3VudCA9XG4gICAgICBwcm9wcy56b25lQXdhcmVuZXNzPy5hdmFpbGFiaWxpdHlab25lQ291bnQgPz8gMjtcblxuICAgIGlmICghWzIsIDNdLmluY2x1ZGVzKGF2YWlsYWJpbGl0eVpvbmVDb3VudCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignSW52YWxpZCB6b25lIGF3YXJlbmVzcyBjb25maWd1cmF0aW9uOyBhdmFpbGFiaWxpdHlab25lQ291bnQgbXVzdCBiZSAyIG9yIDMnKTtcbiAgICB9XG5cbiAgICBjb25zdCB6b25lQXdhcmVuZXNzRW5hYmxlZCA9XG4gICAgICBwcm9wcy56b25lQXdhcmVuZXNzPy5lbmFibGVkID8/XG4gICAgICBwcm9wcy56b25lQXdhcmVuZXNzPy5hdmFpbGFiaWxpdHlab25lQ291bnQgIT0gbnVsbDtcblxuICAgIC8vIElmIFZQQyBvcHRpb25zIGFyZSBzdXBwbGllZCBlbnN1cmUgdGhhdCB0aGUgbnVtYmVyIG9mIHN1Ym5ldHMgbWF0Y2hlcyB0aGUgbnVtYmVyIEFaXG4gICAgaWYgKHByb3BzLnZwY09wdGlvbnMgIT0gbnVsbCAmJiB6b25lQXdhcmVuZXNzRW5hYmxlZCAmJlxuICAgICAgbmV3IFNldChwcm9wcy52cGNPcHRpb25zPy5zdWJuZXRzLm1hcCgoc3VibmV0KSA9PiBzdWJuZXQuYXZhaWxhYmlsaXR5Wm9uZSkpLnNpemUgPCBhdmFpbGFiaWxpdHlab25lQ291bnQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignV2hlbiBwcm92aWRpbmcgdnBjIG9wdGlvbnMgeW91IG5lZWQgdG8gcHJvdmlkZSBhIHN1Ym5ldCBmb3IgZWFjaCBBWiB5b3UgYXJlIHVzaW5nJyk7XG4gICAgfTtcblxuICAgIGlmIChbZGVkaWNhdGVkTWFzdGVyVHlwZSwgaW5zdGFuY2VUeXBlXS5zb21lKHQgPT4gIXQuZW5kc1dpdGgoJy5lbGFzdGljc2VhcmNoJykpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ01hc3RlciBhbmQgZGF0YSBub2RlIGluc3RhbmNlIHR5cGVzIG11c3QgZW5kIHdpdGggXCIuZWxhc3RpY3NlYXJjaFwiLicpO1xuICAgIH1cblxuICAgIGNvbnN0IGVsYXN0aWNzZWFyY2hWZXJzaW9uID0gcHJvcHMudmVyc2lvbi52ZXJzaW9uO1xuICAgIGNvbnN0IGVsYXN0aWNzZWFyY2hWZXJzaW9uTnVtID0gcGFyc2VWZXJzaW9uKHByb3BzLnZlcnNpb24pO1xuXG4gICAgaWYgKFxuICAgICAgZWxhc3RpY3NlYXJjaFZlcnNpb25OdW0gPD0gNy43ICYmXG4gICAgICAhW1xuICAgICAgICAxLjUsIDIuMywgNS4xLCA1LjMsIDUuNSwgNS42LCA2LjAsXG4gICAgICAgIDYuMiwgNi4zLCA2LjQsIDYuNSwgNi43LCA2LjgsIDcuMSwgNy40LFxuICAgICAgICA3LjcsXG4gICAgICBdLmluY2x1ZGVzKGVsYXN0aWNzZWFyY2hWZXJzaW9uTnVtKVxuICAgICkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmtub3duIEVsYXN0aWNzZWFyY2ggdmVyc2lvbjogJHtlbGFzdGljc2VhcmNoVmVyc2lvbn1gKTtcbiAgICB9XG5cbiAgICBjb25zdCB1bnNpZ25lZEJhc2ljQXV0aEVuYWJsZWQgPSBwcm9wcy51c2VVbnNpZ25lZEJhc2ljQXV0aCA/PyBmYWxzZTtcblxuICAgIGlmICh1bnNpZ25lZEJhc2ljQXV0aEVuYWJsZWQpIHtcbiAgICAgIGlmIChwcm9wcy5lbmZvcmNlSHR0cHMgPT0gZmFsc2UpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgY2Fubm90IGRpc2FibGUgSFRUUFMgYW5kIHVzZSB1bnNpZ25lZCBiYXNpYyBhdXRoJyk7XG4gICAgICB9XG4gICAgICBpZiAocHJvcHMubm9kZVRvTm9kZUVuY3J5cHRpb24gPT0gZmFsc2UpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgY2Fubm90IGRpc2FibGUgbm9kZSB0byBub2RlIGVuY3J5cHRpb24gYW5kIHVzZSB1bnNpZ25lZCBiYXNpYyBhdXRoJyk7XG4gICAgICB9XG4gICAgICBpZiAocHJvcHMuZW5jcnlwdGlvbkF0UmVzdD8uZW5hYmxlZCA9PSBmYWxzZSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBjYW5ub3QgZGlzYWJsZSBlbmNyeXB0aW9uIGF0IHJlc3QgYW5kIHVzZSB1bnNpZ25lZCBiYXNpYyBhdXRoJyk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgdW5zaWduZWRBY2Nlc3NQb2xpY3kgPSBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBhY3Rpb25zOiBbJ2VzOkVTSHR0cConXSxcbiAgICAgIHByaW5jaXBhbHM6IFtuZXcgaWFtLkFueW9uZSgpXSxcbiAgICAgIHJlc291cmNlczogW2Nkay5MYXp5LnN0cmluZyh7IHByb2R1Y2U6ICgpID0+IGAke3RoaXMuZG9tYWluQXJufS8qYCB9KV0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBtYXN0ZXJVc2VyQXJuID0gcHJvcHMuZmluZUdyYWluZWRBY2Nlc3NDb250cm9sPy5tYXN0ZXJVc2VyQXJuO1xuICAgIGNvbnN0IG1hc3RlclVzZXJOYW1lUHJvcHMgPSBwcm9wcy5maW5lR3JhaW5lZEFjY2Vzc0NvbnRyb2w/Lm1hc3RlclVzZXJOYW1lO1xuICAgIC8vIElmIGJhc2ljIGF1dGggaXMgZW5hYmxlZCBzZXQgdGhlIHVzZXIgbmFtZSB0byBhZG1pbiBpZiBubyBvdGhlciB1c2VyIGluZm8gaXMgc3VwcGxpZWQuXG4gICAgY29uc3QgbWFzdGVyVXNlck5hbWUgPSB1bnNpZ25lZEJhc2ljQXV0aEVuYWJsZWRcbiAgICAgID8gKG1hc3RlclVzZXJBcm4gPT0gbnVsbCA/IChtYXN0ZXJVc2VyTmFtZVByb3BzID8/ICdhZG1pbicpIDogdW5kZWZpbmVkKVxuICAgICAgOiBtYXN0ZXJVc2VyTmFtZVByb3BzO1xuXG4gICAgaWYgKG1hc3RlclVzZXJBcm4gIT0gbnVsbCAmJiBtYXN0ZXJVc2VyTmFtZSAhPSBudWxsKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ludmFsaWQgZmluZSBncmFpbmVkIGFjY2VzcyBjb250cm9sIHNldHRpbmdzLiBPbmx5IHByb3ZpZGUgb25lIG9mIG1hc3RlciB1c2VyIEFSTiBvciBtYXN0ZXIgdXNlciBuYW1lLiBOb3QgYm90aC4nKTtcbiAgICB9XG5cbiAgICBjb25zdCBhZHZhbmNlZFNlY3VyaXR5RW5hYmxlZCA9IChtYXN0ZXJVc2VyQXJuID8/IG1hc3RlclVzZXJOYW1lKSAhPSBudWxsO1xuICAgIGNvbnN0IGludGVybmFsVXNlckRhdGFiYXNlRW5hYmxlZCA9IG1hc3RlclVzZXJOYW1lICE9IG51bGw7XG4gICAgY29uc3QgbWFzdGVyVXNlclBhc3N3b3JkUHJvcCA9IHByb3BzLmZpbmVHcmFpbmVkQWNjZXNzQ29udHJvbD8ubWFzdGVyVXNlclBhc3N3b3JkO1xuICAgIGNvbnN0IGNyZWF0ZU1hc3RlclVzZXJQYXNzd29yZCA9ICgpOiBjZGsuU2VjcmV0VmFsdWUgPT4ge1xuICAgICAgcmV0dXJuIG5ldyBzZWNyZXRzbWFuYWdlci5TZWNyZXQodGhpcywgJ01hc3RlclVzZXInLCB7XG4gICAgICAgIGdlbmVyYXRlU2VjcmV0U3RyaW5nOiB7XG4gICAgICAgICAgc2VjcmV0U3RyaW5nVGVtcGxhdGU6IEpTT04uc3RyaW5naWZ5KHtcbiAgICAgICAgICAgIHVzZXJuYW1lOiBtYXN0ZXJVc2VyTmFtZSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgICBnZW5lcmF0ZVN0cmluZ0tleTogJ3Bhc3N3b3JkJyxcbiAgICAgICAgICBleGNsdWRlQ2hhcmFjdGVyczogXCJ7fSdcXFxcKltdKClgXCIsXG4gICAgICAgIH0sXG4gICAgICB9KVxuICAgICAgICAuc2VjcmV0VmFsdWVGcm9tSnNvbigncGFzc3dvcmQnKTtcbiAgICB9O1xuICAgIHRoaXMubWFzdGVyVXNlclBhc3N3b3JkID0gaW50ZXJuYWxVc2VyRGF0YWJhc2VFbmFibGVkID9cbiAgICAgIChtYXN0ZXJVc2VyUGFzc3dvcmRQcm9wID8/IGNyZWF0ZU1hc3RlclVzZXJQYXNzd29yZCgpKVxuICAgICAgOiB1bmRlZmluZWQ7XG5cbiAgICBjb25zdCBlbmNyeXB0aW9uQXRSZXN0RW5hYmxlZCA9XG4gICAgICBwcm9wcy5lbmNyeXB0aW9uQXRSZXN0Py5lbmFibGVkID8/IChwcm9wcy5lbmNyeXB0aW9uQXRSZXN0Py5rbXNLZXkgIT0gbnVsbCB8fCB1bnNpZ25lZEJhc2ljQXV0aEVuYWJsZWQpO1xuICAgIGNvbnN0IG5vZGVUb05vZGVFbmNyeXB0aW9uRW5hYmxlZCA9IHByb3BzLm5vZGVUb05vZGVFbmNyeXB0aW9uID8/IHVuc2lnbmVkQmFzaWNBdXRoRW5hYmxlZDtcbiAgICBjb25zdCB2b2x1bWVTaXplID0gcHJvcHMuZWJzPy52b2x1bWVTaXplID8/IDEwO1xuICAgIGNvbnN0IHZvbHVtZVR5cGUgPSBwcm9wcy5lYnM/LnZvbHVtZVR5cGUgPz8gZWMyLkVic0RldmljZVZvbHVtZVR5cGUuR0VORVJBTF9QVVJQT1NFX1NTRDtcbiAgICBjb25zdCBlYnNFbmFibGVkID0gcHJvcHMuZWJzPy5lbmFibGVkID8/IHRydWU7XG4gICAgY29uc3QgZW5mb3JjZUh0dHBzID0gcHJvcHMuZW5mb3JjZUh0dHBzID8/IHVuc2lnbmVkQmFzaWNBdXRoRW5hYmxlZDtcblxuICAgIGZ1bmN0aW9uIGlzSW5zdGFuY2VUeXBlKHQ6IHN0cmluZyk6IEJvb2xlYW4ge1xuICAgICAgcmV0dXJuIGRlZGljYXRlZE1hc3RlclR5cGUuc3RhcnRzV2l0aCh0KSB8fCBpbnN0YW5jZVR5cGUuc3RhcnRzV2l0aCh0KTtcbiAgICB9O1xuXG4gICAgZnVuY3Rpb24gaXNTb21lSW5zdGFuY2VUeXBlKC4uLmluc3RhbmNlVHlwZXM6IHN0cmluZ1tdKTogQm9vbGVhbiB7XG4gICAgICByZXR1cm4gaW5zdGFuY2VUeXBlcy5zb21lKGlzSW5zdGFuY2VUeXBlKTtcbiAgICB9O1xuXG4gICAgZnVuY3Rpb24gaXNFdmVyeUluc3RhbmNlVHlwZSguLi5pbnN0YW5jZVR5cGVzOiBzdHJpbmdbXSk6IEJvb2xlYW4ge1xuICAgICAgcmV0dXJuIGluc3RhbmNlVHlwZXMuc29tZSh0ID0+IGRlZGljYXRlZE1hc3RlclR5cGUuc3RhcnRzV2l0aCh0KSlcbiAgICAgICAgJiYgaW5zdGFuY2VUeXBlcy5zb21lKHQgPT4gaW5zdGFuY2VUeXBlLnN0YXJ0c1dpdGgodCkpO1xuICAgIH07XG5cbiAgICAvLyBWYWxpZGF0ZSBmZWF0dXJlIHN1cHBvcnQgZm9yIHRoZSBnaXZlbiBFbGFzdGljc2VhcmNoIHZlcnNpb24sIHBlclxuICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljc2VhcmNoLXNlcnZpY2UvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2Flcy1mZWF0dXJlcy1ieS12ZXJzaW9uLmh0bWxcbiAgICBpZiAoZWxhc3RpY3NlYXJjaFZlcnNpb25OdW0gPCA1LjEpIHtcbiAgICAgIGlmIChcbiAgICAgICAgcHJvcHMubG9nZ2luZz8uc2xvd0luZGV4TG9nRW5hYmxlZFxuICAgICAgICB8fCBwcm9wcy5sb2dnaW5nPy5hcHBMb2dFbmFibGVkXG4gICAgICAgIHx8IHByb3BzLmxvZ2dpbmc/LnNsb3dTZWFyY2hMb2dFbmFibGVkXG4gICAgICApIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdFcnJvciBhbmQgc2xvdyBsb2dzIHB1Ymxpc2hpbmcgcmVxdWlyZXMgRWxhc3RpY3NlYXJjaCB2ZXJzaW9uIDUuMSBvciBsYXRlci4nKTtcbiAgICAgIH1cbiAgICAgIGlmIChwcm9wcy5lbmNyeXB0aW9uQXRSZXN0Py5lbmFibGVkKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignRW5jcnlwdGlvbiBvZiBkYXRhIGF0IHJlc3QgcmVxdWlyZXMgRWxhc3RpY3NlYXJjaCB2ZXJzaW9uIDUuMSBvciBsYXRlci4nKTtcbiAgICAgIH1cbiAgICAgIGlmIChwcm9wcy5jb2duaXRvS2liYW5hQXV0aCAhPSBudWxsKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQ29nbml0byBhdXRoZW50aWNhdGlvbiBmb3IgS2liYW5hIHJlcXVpcmVzIEVsYXN0aWNzZWFyY2ggdmVyc2lvbiA1LjEgb3IgbGF0ZXIuJyk7XG4gICAgICB9XG4gICAgICBpZiAoaXNTb21lSW5zdGFuY2VUeXBlKCdjNScsICdpMycsICdtNScsICdyNScpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQzUsIEkzLCBNNSwgYW5kIFI1IGluc3RhbmNlIHR5cGVzIHJlcXVpcmUgRWxhc3RpY3NlYXJjaCB2ZXJzaW9uIDUuMSBvciBsYXRlci4nKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAoZWxhc3RpY3NlYXJjaFZlcnNpb25OdW0gPCA2LjApIHtcbiAgICAgIGlmIChwcm9wcy5ub2RlVG9Ob2RlRW5jcnlwdGlvbikge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ05vZGUtdG8tbm9kZSBlbmNyeXB0aW9uIHJlcXVpcmVzIEVsYXN0aWNzZWFyY2ggdmVyc2lvbiA2LjAgb3IgbGF0ZXIuJyk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKGVsYXN0aWNzZWFyY2hWZXJzaW9uTnVtIDwgNi43KSB7XG4gICAgICBpZiAodW5zaWduZWRCYXNpY0F1dGhFbmFibGVkKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignVXNpbmcgdW5zaWduZWQgYmFzaWMgYXV0aCByZXF1aXJlcyBFbGFzdGljc2VhcmNoIHZlcnNpb24gNi43IG9yIGxhdGVyLicpO1xuICAgICAgfVxuICAgICAgaWYgKGFkdmFuY2VkU2VjdXJpdHlFbmFibGVkKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignRmluZS1ncmFpbmVkIGFjY2VzcyBjb250cm9sIHJlcXVpcmVzIEVsYXN0aWNzZWFyY2ggdmVyc2lvbiA2Ljcgb3IgbGF0ZXIuJyk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gVmFsaWRhdGUgYWdhaW5zdCBpbnN0YW5jZSB0eXBlIHJlc3RyaWN0aW9ucywgcGVyXG4gICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNzZWFyY2gtc2VydmljZS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvYWVzLXN1cHBvcnRlZC1pbnN0YW5jZS10eXBlcy5odG1sXG4gICAgaWYgKGlzSW5zdGFuY2VUeXBlKCdpMycpICYmIGVic0VuYWJsZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignSTMgaW5zdGFuY2UgdHlwZXMgZG8gbm90IHN1cHBvcnQgRUJTIHN0b3JhZ2Ugdm9sdW1lcy4nKTtcbiAgICB9XG5cbiAgICBpZiAoaXNTb21lSW5zdGFuY2VUeXBlKCdtMycsICdyMycsICd0MicpICYmIGVuY3J5cHRpb25BdFJlc3RFbmFibGVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ00zLCBSMywgYW5kIFQyIGluc3RhbmNlIHR5cGVzIGRvIG5vdCBzdXBwb3J0IGVuY3J5cHRpb24gb2YgZGF0YSBhdCByZXN0LicpO1xuICAgIH1cblxuICAgIGlmIChpc0luc3RhbmNlVHlwZSgndDIubWljcm8nKSAmJiBlbGFzdGljc2VhcmNoVmVyc2lvbk51bSA+IDIuMykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgdDIubWljcm8uZWxhc3RpY3NlYXJjaCBpbnN0YW5jZSB0eXBlIHN1cHBvcnRzIG9ubHkgRWxhc3RpY3NlYXJjaCAxLjUgYW5kIDIuMy4nKTtcbiAgICB9XG5cbiAgICAvLyBPbmx5IFIzIGFuZCBJMyBzdXBwb3J0IGluc3RhbmNlIHN0b3JhZ2UsIHBlclxuICAgIC8vIGh0dHBzOi8vYXdzLmFtYXpvbi5jb20vZWxhc3RpY3NlYXJjaC1zZXJ2aWNlL3ByaWNpbmcvXG4gICAgaWYgKCFlYnNFbmFibGVkICYmICFpc0V2ZXJ5SW5zdGFuY2VUeXBlKCdyMycsICdpMycpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0VCUyB2b2x1bWVzIGFyZSByZXF1aXJlZCB3aGVuIHVzaW5nIGluc3RhbmNlIHR5cGVzIG90aGVyIHRoYW4gcjMgb3IgaTMuJyk7XG4gICAgfVxuXG4gICAgLy8gRmluZS1ncmFpbmVkIGFjY2VzcyBjb250cm9sIHJlcXVpcmVzIG5vZGUtdG8tbm9kZSBlbmNyeXB0aW9uLCBlbmNyeXB0aW9uIGF0IHJlc3QsXG4gICAgLy8gYW5kIGVuZm9yY2VkIEhUVFBTLlxuICAgIGlmIChhZHZhbmNlZFNlY3VyaXR5RW5hYmxlZCkge1xuICAgICAgaWYgKCFub2RlVG9Ob2RlRW5jcnlwdGlvbkVuYWJsZWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdOb2RlLXRvLW5vZGUgZW5jcnlwdGlvbiBpcyByZXF1aXJlZCB3aGVuIGZpbmUtZ3JhaW5lZCBhY2Nlc3MgY29udHJvbCBpcyBlbmFibGVkLicpO1xuICAgICAgfVxuICAgICAgaWYgKCFlbmNyeXB0aW9uQXRSZXN0RW5hYmxlZCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0VuY3J5cHRpb24tYXQtcmVzdCBpcyByZXF1aXJlZCB3aGVuIGZpbmUtZ3JhaW5lZCBhY2Nlc3MgY29udHJvbCBpcyBlbmFibGVkLicpO1xuICAgICAgfVxuICAgICAgaWYgKCFlbmZvcmNlSHR0cHMpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdFbmZvcmNlIEhUVFBTIGlzIHJlcXVpcmVkIHdoZW4gZmluZS1ncmFpbmVkIGFjY2VzcyBjb250cm9sIGlzIGVuYWJsZWQuJyk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gVmFsaWRhdGUgZmluZSBncmFpbmVkIGFjY2VzcyBjb250cm9sIGVuYWJsZWQgZm9yIGF1ZGl0IGxvZ3MsIHBlclxuICAgIC8vIGh0dHBzOi8vYXdzLmFtYXpvbi5jb20vYWJvdXQtYXdzL3doYXRzLW5ldy8yMDIwLzA5L2VsYXN0aWNzZWFyY2gtYXVkaXQtbG9ncy1ub3ctYXZhaWxhYmxlLW9uLWFtYXpvbi1lbGFzdGljc2VhcmNoLXNlcnZpY2UvXG4gICAgaWYgKHByb3BzLmxvZ2dpbmc/LmF1ZGl0TG9nRW5hYmxlZCAmJiAhYWR2YW5jZWRTZWN1cml0eUVuYWJsZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignRmluZS1ncmFpbmVkIGFjY2VzcyBjb250cm9sIGlzIHJlcXVpcmVkIHdoZW4gYXVkaXQgbG9ncyBwdWJsaXNoaW5nIGlzIGVuYWJsZWQuJyk7XG4gICAgfVxuXG4gICAgbGV0IGNmblZwY09wdGlvbnM6IENmbkRvbWFpbi5WUENPcHRpb25zUHJvcGVydHkgfCB1bmRlZmluZWQ7XG4gICAgaWYgKHByb3BzLnZwY09wdGlvbnMpIHtcbiAgICAgIGNmblZwY09wdGlvbnMgPSB7XG4gICAgICAgIHNlY3VyaXR5R3JvdXBJZHM6IHByb3BzLnZwY09wdGlvbnMuc2VjdXJpdHlHcm91cHMubWFwKChzZykgPT4gc2cuc2VjdXJpdHlHcm91cElkKSxcbiAgICAgICAgc3VibmV0SWRzOiBwcm9wcy52cGNPcHRpb25zLnN1Ym5ldHMubWFwKChzdWJuZXQpID0+IHN1Ym5ldC5zdWJuZXRJZCksXG4gICAgICB9O1xuICAgIH1cblxuICAgIC8vIFNldHVwIGxvZ2dpbmdcbiAgICBjb25zdCBsb2dHcm91cHM6IGxvZ3MuSUxvZ0dyb3VwW10gPSBbXTtcblxuICAgIGlmIChwcm9wcy5sb2dnaW5nPy5zbG93U2VhcmNoTG9nRW5hYmxlZCkge1xuICAgICAgdGhpcy5zbG93U2VhcmNoTG9nR3JvdXAgPSBwcm9wcy5sb2dnaW5nLnNsb3dTZWFyY2hMb2dHcm91cCA/P1xuICAgICAgICBuZXcgbG9ncy5Mb2dHcm91cCh0aGlzLCAnU2xvd1NlYXJjaExvZ3MnLCB7XG4gICAgICAgICAgcmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX01PTlRILFxuICAgICAgICB9KTtcblxuICAgICAgbG9nR3JvdXBzLnB1c2godGhpcy5zbG93U2VhcmNoTG9nR3JvdXApO1xuICAgIH07XG5cbiAgICBpZiAocHJvcHMubG9nZ2luZz8uc2xvd0luZGV4TG9nRW5hYmxlZCkge1xuICAgICAgdGhpcy5zbG93SW5kZXhMb2dHcm91cCA9IHByb3BzLmxvZ2dpbmcuc2xvd0luZGV4TG9nR3JvdXAgPz9cbiAgICAgICAgbmV3IGxvZ3MuTG9nR3JvdXAodGhpcywgJ1Nsb3dJbmRleExvZ3MnLCB7XG4gICAgICAgICAgcmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX01PTlRILFxuICAgICAgICB9KTtcblxuICAgICAgbG9nR3JvdXBzLnB1c2godGhpcy5zbG93SW5kZXhMb2dHcm91cCk7XG4gICAgfTtcblxuICAgIGlmIChwcm9wcy5sb2dnaW5nPy5hcHBMb2dFbmFibGVkKSB7XG4gICAgICB0aGlzLmFwcExvZ0dyb3VwID0gcHJvcHMubG9nZ2luZy5hcHBMb2dHcm91cCA/P1xuICAgICAgICBuZXcgbG9ncy5Mb2dHcm91cCh0aGlzLCAnQXBwTG9ncycsIHtcbiAgICAgICAgICByZXRlbnRpb246IGxvZ3MuUmV0ZW50aW9uRGF5cy5PTkVfTU9OVEgsXG4gICAgICAgIH0pO1xuXG4gICAgICBsb2dHcm91cHMucHVzaCh0aGlzLmFwcExvZ0dyb3VwKTtcbiAgICB9O1xuXG4gICAgaWYgKHByb3BzLmxvZ2dpbmc/LmF1ZGl0TG9nRW5hYmxlZCkge1xuICAgICAgdGhpcy5hdWRpdExvZ0dyb3VwID0gcHJvcHMubG9nZ2luZy5hdWRpdExvZ0dyb3VwID8/XG4gICAgICAgICAgbmV3IGxvZ3MuTG9nR3JvdXAodGhpcywgJ0F1ZGl0TG9ncycsIHtcbiAgICAgICAgICAgIHJldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9NT05USCxcbiAgICAgICAgICB9KTtcblxuICAgICAgbG9nR3JvdXBzLnB1c2godGhpcy5hdWRpdExvZ0dyb3VwKTtcbiAgICB9O1xuXG4gICAgbGV0IGxvZ0dyb3VwUmVzb3VyY2VQb2xpY3k6IExvZ0dyb3VwUmVzb3VyY2VQb2xpY3kgfCBudWxsID0gbnVsbDtcbiAgICBpZiAobG9nR3JvdXBzLmxlbmd0aCA+IDApIHtcbiAgICAgIGNvbnN0IGxvZ1BvbGljeVN0YXRlbWVudCA9IG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbJ2xvZ3M6UHV0TG9nRXZlbnRzJywgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJ10sXG4gICAgICAgIHJlc291cmNlczogbG9nR3JvdXBzLm1hcCgobGcpID0+IGxnLmxvZ0dyb3VwQXJuKSxcbiAgICAgICAgcHJpbmNpcGFsczogW25ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZXMuYW1hem9uYXdzLmNvbScpXSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBVc2UgYSBjdXN0b20gcmVzb3VyY2UgdG8gc2V0IHRoZSBsb2cgZ3JvdXAgcmVzb3VyY2UgcG9saWN5IHNpbmNlIGl0IGlzIG5vdCBzdXBwb3J0ZWQgYnkgQ0RLIGFuZCBjZm4uXG4gICAgICAvLyBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1jZGsvaXNzdWVzLzUzNDNcbiAgICAgIGxvZ0dyb3VwUmVzb3VyY2VQb2xpY3kgPSBuZXcgTG9nR3JvdXBSZXNvdXJjZVBvbGljeSh0aGlzLCBgRVNMb2dHcm91cFBvbGljeSR7dGhpcy5ub2RlLmFkZHJ9YCwge1xuICAgICAgICAvLyBjcmVhdGUgYSBjbG91ZHdhdGNoIGxvZ3MgcmVzb3VyY2UgcG9saWN5IG5hbWUgdGhhdCBpcyB1bmlxdWUgdG8gdGhpcyBkb21haW4gaW5zdGFuY2VcbiAgICAgICAgcG9saWN5TmFtZTogYEVTTG9nUG9saWN5JHt0aGlzLm5vZGUuYWRkcn1gLFxuICAgICAgICBwb2xpY3lTdGF0ZW1lbnRzOiBbbG9nUG9saWN5U3RhdGVtZW50XSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGNvbnN0IGxvZ1B1Ymxpc2hpbmc6IFJlY29yZDxzdHJpbmcsIGFueT4gPSB7fTtcblxuICAgIGlmICh0aGlzLmFwcExvZ0dyb3VwKSB7XG4gICAgICBsb2dQdWJsaXNoaW5nLkVTX0FQUExJQ0FUSU9OX0xPR1MgPSB7XG4gICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgIGNsb3VkV2F0Y2hMb2dzTG9nR3JvdXBBcm46IHRoaXMuYXBwTG9nR3JvdXAubG9nR3JvdXBBcm4sXG4gICAgICB9O1xuICAgIH1cblxuICAgIGlmICh0aGlzLnNsb3dTZWFyY2hMb2dHcm91cCkge1xuICAgICAgbG9nUHVibGlzaGluZy5TRUFSQ0hfU0xPV19MT0dTID0ge1xuICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICBjbG91ZFdhdGNoTG9nc0xvZ0dyb3VwQXJuOiB0aGlzLnNsb3dTZWFyY2hMb2dHcm91cC5sb2dHcm91cEFybixcbiAgICAgIH07XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuc2xvd0luZGV4TG9nR3JvdXApIHtcbiAgICAgIGxvZ1B1Ymxpc2hpbmcuSU5ERVhfU0xPV19MT0dTID0ge1xuICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICBjbG91ZFdhdGNoTG9nc0xvZ0dyb3VwQXJuOiB0aGlzLnNsb3dJbmRleExvZ0dyb3VwLmxvZ0dyb3VwQXJuLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5hdWRpdExvZ0dyb3VwKSB7XG4gICAgICBsb2dQdWJsaXNoaW5nLkFVRElUX0xPR1MgPSB7XG4gICAgICAgIGVuYWJsZWQ6IHRoaXMuYXVkaXRMb2dHcm91cCAhPSBudWxsLFxuICAgICAgICBjbG91ZFdhdGNoTG9nc0xvZ0dyb3VwQXJuOiB0aGlzLmF1ZGl0TG9nR3JvdXA/LmxvZ0dyb3VwQXJuLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICAvLyBDcmVhdGUgdGhlIGRvbWFpblxuICAgIHRoaXMuZG9tYWluID0gbmV3IENmbkRvbWFpbih0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBkb21haW5OYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgIGVsYXN0aWNzZWFyY2hWZXJzaW9uLFxuICAgICAgZWxhc3RpY3NlYXJjaENsdXN0ZXJDb25maWc6IHtcbiAgICAgICAgZGVkaWNhdGVkTWFzdGVyRW5hYmxlZCxcbiAgICAgICAgZGVkaWNhdGVkTWFzdGVyQ291bnQ6IGRlZGljYXRlZE1hc3RlckVuYWJsZWRcbiAgICAgICAgICA/IGRlZGljYXRlZE1hc3RlckNvdW50XG4gICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgIGRlZGljYXRlZE1hc3RlclR5cGU6IGRlZGljYXRlZE1hc3RlckVuYWJsZWRcbiAgICAgICAgICA/IGRlZGljYXRlZE1hc3RlclR5cGVcbiAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgaW5zdGFuY2VDb3VudCxcbiAgICAgICAgaW5zdGFuY2VUeXBlLFxuICAgICAgICB6b25lQXdhcmVuZXNzRW5hYmxlZCxcbiAgICAgICAgem9uZUF3YXJlbmVzc0NvbmZpZzogem9uZUF3YXJlbmVzc0VuYWJsZWRcbiAgICAgICAgICA/IHsgYXZhaWxhYmlsaXR5Wm9uZUNvdW50IH1cbiAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgIH0sXG4gICAgICBlYnNPcHRpb25zOiB7XG4gICAgICAgIGVic0VuYWJsZWQsXG4gICAgICAgIHZvbHVtZVNpemU6IGVic0VuYWJsZWQgPyB2b2x1bWVTaXplIDogdW5kZWZpbmVkLFxuICAgICAgICB2b2x1bWVUeXBlOiBlYnNFbmFibGVkID8gdm9sdW1lVHlwZSA6IHVuZGVmaW5lZCxcbiAgICAgICAgaW9wczogZWJzRW5hYmxlZCA/IHByb3BzLmVicz8uaW9wcyA6IHVuZGVmaW5lZCxcbiAgICAgIH0sXG4gICAgICBlbmNyeXB0aW9uQXRSZXN0T3B0aW9uczoge1xuICAgICAgICBlbmFibGVkOiBlbmNyeXB0aW9uQXRSZXN0RW5hYmxlZCxcbiAgICAgICAga21zS2V5SWQ6IGVuY3J5cHRpb25BdFJlc3RFbmFibGVkXG4gICAgICAgICAgPyBwcm9wcy5lbmNyeXB0aW9uQXRSZXN0Py5rbXNLZXk/LmtleUlkXG4gICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICB9LFxuICAgICAgbm9kZVRvTm9kZUVuY3J5cHRpb25PcHRpb25zOiB7IGVuYWJsZWQ6IG5vZGVUb05vZGVFbmNyeXB0aW9uRW5hYmxlZCB9LFxuICAgICAgbG9nUHVibGlzaGluZ09wdGlvbnM6IGxvZ1B1Ymxpc2hpbmcsXG4gICAgICBjb2duaXRvT3B0aW9uczoge1xuICAgICAgICBlbmFibGVkOiBwcm9wcy5jb2duaXRvS2liYW5hQXV0aCAhPSBudWxsLFxuICAgICAgICBpZGVudGl0eVBvb2xJZDogcHJvcHMuY29nbml0b0tpYmFuYUF1dGg/LmlkZW50aXR5UG9vbElkLFxuICAgICAgICByb2xlQXJuOiBwcm9wcy5jb2duaXRvS2liYW5hQXV0aD8ucm9sZS5yb2xlQXJuLFxuICAgICAgICB1c2VyUG9vbElkOiBwcm9wcy5jb2duaXRvS2liYW5hQXV0aD8udXNlclBvb2xJZCxcbiAgICAgIH0sXG4gICAgICB2cGNPcHRpb25zOiBjZm5WcGNPcHRpb25zLFxuICAgICAgc25hcHNob3RPcHRpb25zOiBwcm9wcy5hdXRvbWF0ZWRTbmFwc2hvdFN0YXJ0SG91clxuICAgICAgICA/IHsgYXV0b21hdGVkU25hcHNob3RTdGFydEhvdXI6IHByb3BzLmF1dG9tYXRlZFNuYXBzaG90U3RhcnRIb3VyIH1cbiAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICBkb21haW5FbmRwb2ludE9wdGlvbnM6IHtcbiAgICAgICAgZW5mb3JjZUh0dHBzLFxuICAgICAgICB0bHNTZWN1cml0eVBvbGljeTogcHJvcHMudGxzU2VjdXJpdHlQb2xpY3kgPz8gVExTU2VjdXJpdHlQb2xpY3kuVExTXzFfMCxcbiAgICAgIH0sXG4gICAgICBhZHZhbmNlZFNlY3VyaXR5T3B0aW9uczogYWR2YW5jZWRTZWN1cml0eUVuYWJsZWRcbiAgICAgICAgPyB7XG4gICAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICBpbnRlcm5hbFVzZXJEYXRhYmFzZUVuYWJsZWQsXG4gICAgICAgICAgbWFzdGVyVXNlck9wdGlvbnM6IHtcbiAgICAgICAgICAgIG1hc3RlclVzZXJBcm46IG1hc3RlclVzZXJBcm4sXG4gICAgICAgICAgICBtYXN0ZXJVc2VyTmFtZTogbWFzdGVyVXNlck5hbWUsXG4gICAgICAgICAgICBtYXN0ZXJVc2VyUGFzc3dvcmQ6IHRoaXMubWFzdGVyVXNlclBhc3N3b3JkPy50b1N0cmluZygpLFxuICAgICAgICAgIH0sXG4gICAgICAgIH1cbiAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgfSk7XG5cbiAgICBpZiAocHJvcHMuZW5hYmxlVmVyc2lvblVwZ3JhZGUpIHtcbiAgICAgIHRoaXMuZG9tYWluLmNmbk9wdGlvbnMudXBkYXRlUG9saWN5ID0ge1xuICAgICAgICAuLi50aGlzLmRvbWFpbi5jZm5PcHRpb25zLnVwZGF0ZVBvbGljeSxcbiAgICAgICAgZW5hYmxlVmVyc2lvblVwZ3JhZGU6IHByb3BzLmVuYWJsZVZlcnNpb25VcGdyYWRlLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBpZiAobG9nR3JvdXBSZXNvdXJjZVBvbGljeSkgeyB0aGlzLmRvbWFpbi5ub2RlLmFkZERlcGVuZGVuY3kobG9nR3JvdXBSZXNvdXJjZVBvbGljeSk7IH1cblxuICAgIGlmIChwcm9wcy5kb21haW5OYW1lKSB7IHRoaXMubm9kZS5hZGRNZXRhZGF0YSgnYXdzOmNkazpoYXNQaHlzaWNhbE5hbWUnLCBwcm9wcy5kb21haW5OYW1lKTsgfVxuXG4gICAgdGhpcy5kb21haW5OYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUodGhpcy5kb21haW4ucmVmKTtcblxuICAgIHRoaXMuZG9tYWluRW5kcG9pbnQgPSB0aGlzLmRvbWFpbi5nZXRBdHQoJ0RvbWFpbkVuZHBvaW50JykudG9TdHJpbmcoKTtcblxuICAgIHRoaXMuZG9tYWluQXJuID0gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZSh0aGlzLmRvbWFpbi5hdHRyQXJuLCB7XG4gICAgICBzZXJ2aWNlOiAnZXMnLFxuICAgICAgcmVzb3VyY2U6ICdkb21haW4nLFxuICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGFjY2Vzc1BvbGljeVN0YXRlbWVudHM6IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXSB8IHVuZGVmaW5lZCA9IHVuc2lnbmVkQmFzaWNBdXRoRW5hYmxlZFxuICAgICAgPyAocHJvcHMuYWNjZXNzUG9saWNpZXMgPz8gW10pLmNvbmNhdCh1bnNpZ25lZEFjY2Vzc1BvbGljeSlcbiAgICAgIDogcHJvcHMuYWNjZXNzUG9saWNpZXM7XG5cbiAgICBpZiAoYWNjZXNzUG9saWN5U3RhdGVtZW50cyAhPSBudWxsKSB7XG4gICAgICBjb25zdCBhY2Nlc3NQb2xpY3kgPSBuZXcgRWxhc3RpY3NlYXJjaEFjY2Vzc1BvbGljeSh0aGlzLCAnRVNBY2Nlc3NQb2xpY3knLCB7XG4gICAgICAgIGRvbWFpbk5hbWU6IHRoaXMuZG9tYWluTmFtZSxcbiAgICAgICAgZG9tYWluQXJuOiB0aGlzLmRvbWFpbkFybixcbiAgICAgICAgYWNjZXNzUG9saWNpZXM6IGFjY2Vzc1BvbGljeVN0YXRlbWVudHMsXG4gICAgICB9KTtcblxuICAgICAgaWYgKHByb3BzLmVuY3J5cHRpb25BdFJlc3Q/Lmttc0tleSkge1xuXG4gICAgICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljc2VhcmNoLXNlcnZpY2UvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2VuY3J5cHRpb24tYXQtcmVzdC5odG1sXG5cbiAgICAgICAgLy8gdGhlc2UgcGVybWlzc2lvbnMgYXJlIGRvY3VtZW50ZWQgYXMgcmVxdWlyZWQgZHVyaW5nIGRvbWFpbiBjcmVhdGlvbi5cbiAgICAgICAgLy8gd2hpbGUgbm90IHN0cmljdGx5IGRvY3VtZW50ZWQgZm9yIHVwZGF0ZXMgYXMgd2VsbCwgaXQgc3RhbmRzIHRvIHJlYXNvbiB0aGF0IGFuIHVwZGF0ZVxuICAgICAgICAvLyBvcGVyYXRpb24gbWlnaHQgcmVxdWlyZSB0aGVzZSBpbiBjYXNlIHRoZSBjbHVzdGVyIHVzZXMgYSBrbXMga2V5LlxuICAgICAgICAvLyBlbXBpcmNhbCBldmlkZW5jZSBzaG93cyB0aGlzIGlzIGluZGVlZCByZXF1aXJlZDogaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2lzc3Vlcy8xMTQxMlxuICAgICAgICBhY2Nlc3NQb2xpY3kuZ3JhbnRQcmluY2lwYWwuYWRkVG9QcmluY2lwYWxQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGFjdGlvbnM6IFsna21zOkxpc3QqJywgJ2ttczpEZXNjcmliZSonLCAna21zOkNyZWF0ZUdyYW50J10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbcHJvcHMuZW5jcnlwdGlvbkF0UmVzdC5rbXNLZXkua2V5QXJuXSxcbiAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgIH0pKTtcbiAgICAgIH1cblxuICAgICAgYWNjZXNzUG9saWN5Lm5vZGUuYWRkRGVwZW5kZW5jeSh0aGlzLmRvbWFpbik7XG4gICAgfVxuICB9XG59XG5cbi8qKlxuICogR2l2ZW4gYW4gRWxhc3RpY3NlYXJjaCBkb21haW4gZW5kcG9pbnQsIHJldHVybnMgYSBDbG91ZEZvcm1hdGlvbiBleHByZXNzaW9uIHRoYXRcbiAqIGV4dHJhY3RzIHRoZSBkb21haW4gbmFtZS5cbiAqXG4gKiBEb21haW4gZW5kcG9pbnRzIGxvb2sgbGlrZSB0aGlzOlxuICpcbiAqICAgaHR0cHM6Ly9leGFtcGxlLWRvbWFpbi1qY2pvdHJ0NmY3b3RlbTRzcWN3YmNoM2M0dS51cy1lYXN0LTEuZXMuYW1hem9uYXdzLmNvbVxuICogICBodHRwczovLzxkb21haW4tbmFtZT4tPHN1ZmZpeD4uPHJlZ2lvbj4uZXMuYW1hem9uYXdzLmNvbVxuICpcbiAqIC4ud2hpY2ggbWVhbnMgdGhhdCBpbiBvcmRlciB0byBleHRyYWN0IHRoZSBkb21haW4gbmFtZSBmcm9tIHRoZSBlbmRwb2ludCwgd2UgY2FuXG4gKiBzcGxpdCB0aGUgZW5kcG9pbnQgdXNpbmcgXCItPHN1ZmZpeD5cIiBhbmQgc2VsZWN0IHRoZSBjb21wb25lbnQgaW4gaW5kZXggMC5cbiAqXG4gKiBAcGFyYW0gZG9tYWluRW5kcG9pbnQgVGhlIEVsYXN0aWNzZWFyY2ggZG9tYWluIGVuZHBvaW50XG4gKi9cbmZ1bmN0aW9uIGV4dHJhY3ROYW1lRnJvbUVuZHBvaW50KGRvbWFpbkVuZHBvaW50OiBzdHJpbmcpIHtcbiAgY29uc3QgeyBob3N0bmFtZSB9ID0gbmV3IFVSTChkb21haW5FbmRwb2ludCk7XG4gIGNvbnN0IGRvbWFpbiA9IGhvc3RuYW1lLnNwbGl0KCcuJylbMF07XG4gIGNvbnN0IHN1ZmZpeCA9ICctJyArIGRvbWFpbi5zcGxpdCgnLScpLnNsaWNlKC0xKVswXTtcbiAgcmV0dXJuIGRvbWFpbi5zcGxpdChzdWZmaXgpWzBdO1xufVxuXG4vKipcbiAqIENvbnZlcnRzIGFuIEVsYXN0aWNzZWFyY2ggdmVyc2lvbiBpbnRvIGEgaW50byBhIGRlY2ltYWwgbnVtYmVyIHdpdGggbWFqb3IgYW5kIG1pbm9yIHZlcnNpb24gaS5lIHgueS5cbiAqXG4gKiBAcGFyYW0gdmVyc2lvbiBUaGUgRWxhc3RpY3NlYXJjaCB2ZXJzaW9uIG9iamVjdFxuICovXG5mdW5jdGlvbiBwYXJzZVZlcnNpb24odmVyc2lvbjogRWxhc3RpY3NlYXJjaFZlcnNpb24pOiBudW1iZXIge1xuICBjb25zdCB2ZXJzaW9uU3RyID0gdmVyc2lvbi52ZXJzaW9uO1xuICBjb25zdCBmaXJzdERvdCA9IHZlcnNpb25TdHIuaW5kZXhPZignLicpO1xuXG4gIGlmIChmaXJzdERvdCA8IDEpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgRWxhc3RpY3NlYXJjaCB2ZXJzaW9uOiAke3ZlcnNpb25TdHJ9LiBWZXJzaW9uIHN0cmluZyBuZWVkcyB0byBzdGFydCB3aXRoIG1ham9yIGFuZCBtaW5vciB2ZXJzaW9uICh4LnkpLmApO1xuICB9XG5cbiAgY29uc3Qgc2Vjb25kRG90ID0gdmVyc2lvblN0ci5pbmRleE9mKCcuJywgZmlyc3REb3QgKyAxKTtcblxuICB0cnkge1xuICAgIGlmIChzZWNvbmREb3QgPT0gLTEpIHtcbiAgICAgIHJldHVybiBwYXJzZUZsb2F0KHZlcnNpb25TdHIpO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4gcGFyc2VGbG9hdCh2ZXJzaW9uU3RyLnN1YnN0cmluZygwLCBzZWNvbmREb3QpKTtcbiAgICB9XG4gIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIEVsYXN0aWNzZWFyY2ggdmVyc2lvbjogJHt2ZXJzaW9uU3RyfS4gVmVyc2lvbiBzdHJpbmcgbmVlZHMgdG8gc3RhcnQgd2l0aCBtYWpvciBhbmQgbWlub3IgdmVyc2lvbiAoeC55KS5gKTtcbiAgfVxufVxuIl19