"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const apputils_1 = require("@jupyterlab/apputils");
const coreutils_1 = require("@jupyterlab/coreutils");
const application_1 = require("@jupyterlab/application");
const docmanager_1 = require("@jupyterlab/docmanager");
const filebrowser_1 = require("@jupyterlab/filebrowser");
const launcher_1 = require("@jupyterlab/launcher");
const mainmenu_1 = require("@jupyterlab/mainmenu");
const widgets_1 = require("@phosphor/widgets");
const requests_helper_1 = require("requests-helper");
require("../style/index.css");
// tslint:disable: variable-name
const extension = {
    activate,
    autoStart: true,
    id: "jupyterlab_autoversion",
    optional: [launcher_1.ILauncher],
    requires: [docmanager_1.IDocumentManager, apputils_1.ICommandPalette, application_1.ILayoutRestorer, mainmenu_1.IMainMenu, filebrowser_1.IFileBrowserFactory],
};
class AutoversionWidget extends widgets_1.Widget {
    constructor(app, context, id, path) {
        const body = document.createElement("div");
        body.style.display = "flex";
        body.style.flexDirection = "column";
        const default_none = document.createElement("option");
        default_none.selected = false;
        default_none.disabled = true;
        default_none.hidden = false;
        default_none.style.display = "none";
        default_none.value = "";
        const type = document.createElement("select");
        type.appendChild(default_none);
        requests_helper_1.request("get", coreutils_1.PageConfig.getBaseUrl() + "autoversion/get?id=" + id + "&path=" + path).then((res) => {
            if (res.ok) {
                const versions = res.json();
                for (const record of versions.versions) {
                    const option = document.createElement("option");
                    option.value = record;
                    const timestamp = new Date(record[1]);
                    option.textContent = timestamp + " -- " + record[0].slice(0, 6);
                    type.appendChild(option);
                }
            }
        });
        type.style.marginBottom = "15px";
        type.style.minHeight = "25px";
        body.appendChild(type);
        super({ node: body });
    }
    getValue() {
        return this.inputNode.value;
    }
    get inputNode() {
        return this.node.getElementsByTagName("select")[0];
    }
}
exports.AutoversionWidget = AutoversionWidget;
function autoversion(app, context) {
    const model = context.model;
    const id = model.metadata.autoversion || "";
    apputils_1.showDialog({
        body: new AutoversionWidget(app, context, id, context.path),
        buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.okButton({ label: "Ok" })],
        focusNodeSelector: "input",
        title: "Autoversion:",
    }).then((result) => {
        if (result.button.label === "Cancel") {
            return;
        }
        else {
            const val = result.value.split(",");
            revision(app, context, val[2], val[3]);
        }
    });
}
exports.autoversion = autoversion;
function revision(app, context, id, version) {
    requests_helper_1.request("get", coreutils_1.PageConfig.getBaseUrl() + "autoversion/restore?id=" +
        id + "&path=" + context.path + "&version=" + version).then((res) => {
        if (res.ok) {
            const data = res.json();
            if (data.version.toString() === version) {
                context.model.fromJSON(data.nb);
            }
        }
    });
}
exports.revision = revision;
// tslint:disable-next-line:max-classes-per-file
class AutoversionExtension {
    constructor(app) {
        this.app = app;
    }
    createNew(panel, context) {
        // Create the toolbar button
        const button = new apputils_1.ToolbarButton({
            className: "autoversionButton",
            iconClassName: "fa fa-fast-backward",
            onClick: () => autoversion(this.app, context),
            tooltip: "Autoversion",
        });
        // Add the toolbar button to the notebook
        panel.toolbar.insertItem(7, "autoversion", button);
        return button;
    }
}
exports.AutoversionExtension = AutoversionExtension;
function activate(app, docManager, palette, restorer, mainMenu, browser, launcher) {
    const avExtension = new AutoversionExtension(app);
    app.docRegistry.addWidgetExtension("Notebook", avExtension);
    app.contextMenu.addItem({
        command: "notebook:autoversion",
        rank: -0.5,
        selector: ".jp-Notebook",
    });
    // tslint:disable-next-line:no-console
    console.log("JupyterLab extension jupyterlab_autoversion is activated!");
}
exports._activate = activate;
exports.default = extension;
