from mstrio.utils.helper import response_handler


def get_privileges(connection, error_msg=None):
    """Get the set of available privileges for the platform.

    Args:
        connection: MicroStrategy REST API connection object
        error_msg (string, optional): Custom Error Message for Error Handling

    Returns:
        Complete HTTP response object.
    """
    response = connection.session.get(url=connection.base_url + '/api/iserver/privileges',
                                      headers={'X-MSTR-ProjectID': None})
    if not response.ok:
        if error_msg is None:
            error_msg = "Error getting privileges"
        response_handler(response, error_msg)
    return response


def get_privilege_categories(connection, error_msg=None):
    """Get the set of available privilege categories for the platform.

    Args:
        connection: MicroStrategy REST API connection object
        error_msg (string, optional): Custom Error Message for Error Handling

    Returns:
        Complete HTTP response object.
    """
    response = connection.session.get(url=connection.base_url + '/api/iserver/privileges/categories',
                                      headers={'X-MSTR-ProjectID': None})
    if not response.ok:
        if error_msg is None:
            error_msg = "Error getting privilege categories"
        response_handler(response, error_msg)
    return response


def get_security_roles(connection, fields=None, error_msg=None):
    """Get information for all security roles. A security role describes the
    ability to do something, such as create, edit, add, delete, view, manage,
    save, search, share, export, and so on. A security role has a name, a
    description, and a privilege.

    Args:
        connection: MicroStrategy REST API connection object
        fields: top-level field whitelist which allows client to selectively
            retrieve part of the response model
        error_msg (string, optional): Custom Error Message for Error Handling

    Returns:
        Complete HTTP response object.
    """
    response = connection.session.get(url=connection.base_url + '/api/securityRoles',
                                      headers={'X-MSTR-ProjectID': None},
                                      params={'fields': fields})
    if not response.ok:
        if error_msg is None:
            error_msg = "Error getting information for set of security roles"
        response_handler(response, error_msg)
    return response


def create_security_role(connection, body, error_msg=None):
    """Create a new security role.

    Args:
        connection: MicroStrategy REST API connection object
        body: JSON-formatted definition of the dataset. Generated by
            `utils.formjson()`.
        error_msg (string, optional): Custom Error Message for Error Handling

    Returns:
        Complete HTTP response object.
    """
    response = connection.session.post(url=connection.base_url + '/api/securityRoles',
                                       headers={'X-MSTR-ProjectID': None},
                                       json=body)
    if not response.ok:
        if error_msg is None:
            error_msg = "Error creating new role"
        response_handler(response, error_msg)
    return response


def get_security_role(connection, id, error_msg=None):
    """Get information for a security role with security role Id.

    Args:
        connection: MicroStrategy REST API connection object
        id (string): Security role ID
        error_msg (string, optional): Custom Error Message for Error Handling

    Returns:
        Complete HTTP response object.
    """
    response = connection.session.get(url=connection.base_url + '/api/securityRoles/' + id,
                                      headers={'X-MSTR-ProjectID': None})
    if not response.ok:
        if error_msg is None:
            error_msg = "Error getting role information"
        response_handler(response, error_msg)
    return response


def delete_security_role(connection, id, error_msg=None):
    """Delete info for a security role with given Id.

    Args:
        connection: MicroStrategy REST API connection object
        id (string): Security role ID
        error_msg (string, optional): Custom Error Message for Error Handling

    Returns:
        Complete HTTP response object.
    """
    response = connection.session.delete(url=connection.base_url + '/api/securityRoles/' + id,
                                         headers={'X-MSTR-ProjectID': None})
    if not response.ok:
        if error_msg is None:
            error_msg = "Error deleting role information"
        response_handler(response, error_msg)
    return response


def update_security_role(connection, id, body, error_msg=None):
    """Update information for a specific security role.

    Args:
        connection: MicroStrategy REST API connection object
        id (string): Security role ID
        body: JSON-formatted definition of the dataset. Generated by
            `utils.formjson()`.
        error_msg (string, optional): Custom Error Message for Error Handling

    Returns:
        Complete HTTP response object.
    """
    response = connection.session.patch(url=connection.base_url + '/api/securityRoles/' + id,
                                        headers={'X-MSTR-ProjectID': None},
                                        json=body)
    if not response.ok:
        if error_msg is None:
            error_msg = "Error updating Security Role information"
        response_handler(response, error_msg)
    return response


def get_security_role_for_project(connection, id, project_id, error_msg=None):
    """Get all users and user groups that are linked to a specific security
    role.

    Args:
        connection: MicroStrategy REST API connection object
        id (string): Security role ID
        project_id (string, optional): Project id string
        error_msg (string, optional): Custom Error Message for Error Handling

    Returns:
        Complete HTTP response object.
    """
    response = connection.session.get(url=connection.base_url + '/api/securityRoles/' + id + '/projects/' +
                                      project_id + '/members',
                                      headers={'X-MSTR-ProjectID': None})
    if not response.ok:
        if error_msg is None:
            error_msg = "Error getting role information"
        response_handler(response, error_msg)
    return response
