# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['sqlpyd']

package_data = \
{'': ['*']}

install_requires = \
['pydantic>=1.10.4,<2.0.0',
 'python-dotenv>=0.21.0,<0.22.0',
 'sqlite-utils>=3.30,<4.0']

setup_kwargs = {
    'name': 'sqlpyd',
    'version': '0.1.0',
    'description': 'Validate data with pydantic for consumption by sqlite-utils',
    'long_description': '# sqlpyd\n\nCombining [sqlite-utils](https://github.com/simonw/sqlite-utils) data management + [Pydantic](https://github.com/pydantic/pydantic) data validation for data that will (later) be deployed in a specific [Datasette](https://datasette.io/) project: [LawData](https://lawdata.xyz).\n\n## Premise\n\nEach data model exists in two dimensions:\n\n1. the app layer via a pythonic interface; and\n2. the persistence layer via a database.\n\nHere we\'ll use sqlite. Though sqlite features are frequently evolving, see *json1*, *fts5*, etc., it lacks a more robust validation mechanism.\n\nPydantic would be useful to:\n\n1. clean and validate a model\'s fields prior to database insertion;\n2. reuse data on the pythonic interface for dynamic queries on database extraction.\n\nSince the database query syntax (SQL) is different from the app syntax (python), a useful bridge is sqlite-utils which allows us, via this package, to use pre-defined *Pydantic* field attributes as a means of creating dynamic SQL statements.\n\nPut another way, this is an attempt to integrate the two tools so that the models declared in *Pydantic* can be consumed directly by *sqlite-utils*.\n\nThe opinionated use of default configurations is intended for a more specific project. Later on, may consider making this more general in scope.\n\n## Connection\n\nConnect to a database declared by an `.env` file through a `DB_FILE` variable, e.g.\n\n```sh\nDB_FILE="code/sqlpyd/test.db"\n```\n\nWith the .env file created, the following sqlite `sqlpyd.Connection` object gets a typed `Table`:\n\n```python\nfrom sqlpyd import Connection\nfrom sqlite_utils.db import Table\n\nconn = Connection()  # will use .env file\'s DB_FILE value\nconn.db["test_table"].insert(\n    {"text": "hello-world"}, pk="id"\n)  # will contain a db object\nisinstance(conn.tbl("test_table"), Table)\nTrue\n```\n\nThere appears to be movement to make *sqlite-utils* more type-friendly, [see issue](https://github.com/simonw/sqlite-utils/issues/496).\n\n## Fields\n\n### Generic Pydantic Model\n\nLet\'s assume a generic pydantic BaseModel with a non-primitive field like `suffix` and `gender`:\n\n```python\n# see sqlpyd/name.py\nclass Gender(str, Enum):\n    male = "male"\n    female = "female"\n    other = "unspecified"\n\n\nclass Suffix(str, Enum):\n    jr = "Jr."\n    sr = "Sr."\n    third = "III"\n    fourth = "IV"\n    fifth = "V"\n    sixth = "VI"\n\n\nclass IndividualBio(BaseModel):\n    first_name: str = Field(None, max_length=50)\n    last_name: str = Field(None, max_length=50)\n    suffix: Suffix | None = Field(None, max_length=4)\n    nick_name: str = Field(None, max_length=50)\n    gender: Gender | None = Field(Gender.other)\n\n    class Config:\n        use_enum_values = True\n```\n\nWith the BaseModel, we can get the types directly using:\n\n```python shell\n>>> IndividualBio.__annotations__\n{\n    "first_name": str,\n    "last_name": str,\n    "suffix": sqlpyd.__main__.Suffix | None,  # is non-primitive / optional\n    "nick_name": str,\n    "gender": sqlpyd.__main__.Gender | None,  # is non-primitive  / optional\n}\n```\n\nUsing the *sqlite-utils* convention of creating tables, this will throw an error:\n\n```python shell\n>>> from sqlpyd import Connection  # thin wrapper over sqlite-utils Database()\n\n>>> conn = Connection(DatabasePath="created.db")\n>>> conn.db["test_tbl"].create(columns=IndividualBio.__annotations__)\n...\nKeyError: sqlpyd.__main__.Suffix | None\n```\n\n### Data Modelling & Input Validation\n\nWe could rewrite the needed columns and use *sqlite-utils*:\n\n```python\nconn.db["test_tbl"].create(\n    columns={\n        "first_name": str,\n        "last_name": str,\n        "suffix": str,\n        "nick_name": str,\n        "gender": str,\n    }\n)\n# <Table test_tbl (first_name, last_name, suffix, nick_name, gender)>\n```\n\nBut we can also modify the initial Pydantic model and co-inherit from  `sqlpyd.TableConfig`, to wit:\n\n```python\nclass RegularName(\n    BaseModel\n):  # separated the name to add a clear pre-root validator, note addition to field attributes\n    full_name: str | None = Field(None, col=str, fts=True, index=True)\n    first_name: str = Field(..., max_length=50, col=str, fts=True)\n    last_name: str = Field(..., max_length=50, col=str, fts=True, index=True)\n    suffix: Suffix | None = Field(None, max_length=4, col=str)\n\n    class Config:\n        use_enum_values = True\n\n    @root_validator(pre=True)\n    def set_full_name(cls, values):\n        if not values.get("full_name"):\n            first = values.get("first_name")\n            last = values.get("last_name")\n            if first and last:\n                values["full_name"] = f"{first} {last}"\n                if sfx := values.get("suffix"):\n                    values["full_name"] += f", {sfx}"\n        return values\n\n\nclass IndividualBio(\n    TableConfig\n):  # mandatory step:  inherit from TableConfig (which inherits from BaseModel)\n    __tablename__ = "person_tbl"  # optional: may declare a tablename\n    __indexes__ = [["first_name", "last_name"]]  # optional: may declare joined indexes\n    nick_name: str | None = Field(None, max_length=50, col=str, fts=True)\n    gender: Gender | None = Field(Gender.other, max_length=15, col=str)\n\n    @validator("gender", pre=True)\n    def lower_cased_gender(cls, v):\n        return Gender(v.lower()) if v else None\n```\n\nWith this setup, we can use the connection to create the table. Note that the primary key `id` is auto-generated in this scenario:\n\n```python\nconn = Connection(DatabasePath="test.db", WAL=False)\n\nconn.create_table(IndividualBio)\n# <Table person_tbl (id, full_name, first_name, last_name, suffix, nick_name, gender)>\n\nperson2 = {  # dict\n    "first_name": "Jane",\n    "last_name": "Doe",\n    "suffix": None,\n    "gender": "FEMALE",  # all caps\n    "nick_name": "Jany",\n}\n\nIndividualBio.__validators__  # note that we created a validator for \'gender\'\n# {\'gender\': [<pydantic.class_validators.Validator object at 0x10c497510>]}\n\nIndividualBio.__pre_root_validators__()  # we also have one to create a \'full_name\'\n# [<function RegularName.set_full_name at 0x10c4b43a0>]\n\ntbl = conn.add_record(\n    IndividualBio, person2\n)  # under the hood, the dict is instantiated to a Pydantic model and the resulting `tbl` value is an sqlite-utils Table\n\nassert list(tbl.rows) == [\n    {\n        "id": 1,  # auto-generated\n        "full_name": "Jane Doe",  # since the model contains a pre root-validator, it adds a full name\n        "first_name": "Jane",\n        "last_name": "Doe",\n        "suffix": None,\n        "nick_name": "Jany",\n        "gender": "female",  # since the model contains a validator, it cleans the same prior to database entry\n    }\n]\nTrue\n```\n\n### Attributes\n\n`sqlite-utils` is a far more powerful solution than the limited subset of features provided here. Again, this abstraction is for the purpose of easily reusing the functionality for a specific project rather than for something more generalized.\n\n#### Columns In General\n\nUsing `col` in the Pydantic Field signals the need to add the field to an sqlite database table:\n\n```python\nconn = Connection(DatabasePath="test.db", WAL=False)\nkls = IndividualBio\ntbl = conn.db[kls.__tablename__]\ncols = kls.extract_cols(kls.__fields__)  # possible fields to use\n"""\n{\'first_name\': str,\n \'last_name\': str,\n \'suffix\': str,\n \'nick_name\': str,\n \'gender\': str}\n"""\ntbl.create(cols)  # customize tablename and column types\n# <Table individual_bio_tbl (first_name, last_name, suffix, nick_name, gender)>\n```\n\n#### Primary Key\n\nTo auto-generate, use the `TableConfig.config_tbl()` helper. It auto-creates the `id` field as an `int`-based primary key.\n\n> Note: if an `id` is declared as a `str` in the pydantic model, the `str` declaration takes precedence over the implicit `int` default.\n\n```python\nconn = Connection(DatabasePath="test_db.db")\nkls = IndividualBio\ntbl = conn.db[kls.__tablename__]\ntbl_created = kls.config_tbl(tbl=tbl, cols=kls.__fields__)\n# <Table individual_bio_tbl (id, first_name, last_name, suffix, nick_name, gender)> # id now added\n```\n\nThis results in the following sql schema:\n\n```sql\nCREATE TABLE [individual_bio_tbl] (\n   [id] INTEGER PRIMARY KEY, -- added as integer since no field specified\n   [first_name] TEXT NOT NULL, -- required via Pydantic\'s ...\n   [last_name] TEXT NOT NULL, -- required via Pydantic\'s ...\n   [suffix] TEXT,\n   [nick_name] TEXT,\n   [gender] TEXT\n)\n```\n\n#### Full-Text Search (fts) Fields\n\nSince we indicated, in the above declaration of `Fields`, that some columns are to be used for `fts`, we enable *sqlite-utils* to auto-generate the tables required. This makes possible the [prescribed approach of querying fts tables](https://sqlite-utils.datasette.io/en/stable/python-api.html#building-sql-queries-with-table-search-sql):\n\n```python\n# Using the same variable for `tbl` described above, can yield a query string, viz.\nprint(tbl.search_sql(columns=["first_name", "last_name"]))\n```\n\nproduces:\n\n```sql\nwith original as (\n    select\n        rowid,\n        [first_name],\n        [last_name]\n    from [individual_bio_tbl]\n)\nselect\n    [original].[first_name],\n    [original].[last_name]\nfrom\n    [original]\n    join [individual_bio_tbl_fts] on [original].rowid = [individual_bio_tbl_fts].rowid\nwhere\n    [individual_bio_tbl_fts] match :query\norder by\n    [individual_bio_tbl_fts].rank\n```\n\n#### Foreign Keys\n\nTo add foreign keys, can use the `fk` attribute on a ModelField, assigning the same to a 2-tuple, e.g.:\n\n```python\nclass GroupedIndividuals(TableConfig):\n    __tablename__ = "grouping_tbl"\n    __indexes__ = [["member_id", "name"]]\n\n    name: str = Field(..., max_length=50, col=str)\n    member_id: int = Field(\n        ..., col=int, fk=(IndividualBio.__tablename__, "id"), index=True\n    )\n```\n\nParts of `fk` tuple:\n\n- The first part of the `fk` tuple is the referenced table name *X*.\n- The second part of the `fk` tuple is the id of *X*.\n\nSo in the above example, `member_id`, the Pydantic field, is constrained to the "id" column of the table "individual_bio_tbl"\n\n#### Indexes\n\nNote that we can add an index to each field as well with a boolean `True` to a ModelField attribute `index`. In case we want to use a combination of columns for the index, can include this when subclassing `TableConfig`:\n\n```python\nclass GroupedIndividuals(TableConfig):\n    __tablename__ = "grouping_tbl"\n    __indexes__ = [["member_id", "name"]]  # follow sqlite-utils convention\n```\n\nWhen combined, the sql generated amounts to the following:\n\n```sql\nCREATE TABLE [grouping_tbl] (\n   [id] INTEGER PRIMARY KEY,\n   [name] TEXT NOT NULL,\n   [member_id] INTEGER NOT NULL REFERENCES [individual_bio_tbl]([id])\n);\nCREATE UNIQUE INDEX [idx_grouping_tbl_member_id]\n    ON [grouping_tbl] ([member_id]);\nCREATE UNIQUE INDEX [idx_grouping_tbl_name_member_id]\n    ON [grouping_tbl] ([name], [member_id]);\n```\n',
    'author': 'Marcelino G. Veloso III',
    'author_email': 'mars@veloso.one',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
