"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssertionsProvider = void 0;
const path = require("path");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_cdk_lib_1 = require("aws-cdk-lib");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const constructs_1 = require("constructs");
let SDK_METADATA = undefined;
/**
 * Represents an assertions provider. The creates a singletone
 * Lambda Function that will create a single function per stack
 * that serves as the custom resource provider for the various
 * assertion providers
 */
class AssertionsProvider extends constructs_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        const handler = new lambda.SingletonFunction(this, 'AssertionsProvider', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            uuid: '1488541a-7b23-4664-81b6-9b4408076b81',
            timeout: aws_cdk_lib_1.Duration.minutes(2),
        });
        this.grantPrincipal = handler.grantPrincipal;
        this.serviceToken = handler.functionArn;
    }
    encode(obj) {
        if (!obj) {
            return obj;
        }
        return JSON.parse(JSON.stringify(obj), (_k, v) => {
            switch (v) {
                case true:
                    return 'TRUE:BOOLEAN';
                case false:
                    return 'FALSE:BOOLEAN';
                default:
                    return v;
            }
        });
    }
    addPolicyStatementFromSdkCall(service, api, resources) {
        if (SDK_METADATA === undefined) {
            // eslint-disable-next-line
            SDK_METADATA = require('./sdk-api-metadata.json');
        }
        const srv = service.toLowerCase();
        const iamService = (SDK_METADATA[srv] && SDK_METADATA[srv].prefix) || srv;
        const iamAction = api.charAt(0).toUpperCase() + api.slice(1);
        const statement = new iam.PolicyStatement({
            actions: [`${iamService}:${iamAction}`],
            resources: resources || ['*'],
        });
        this.grantPrincipal.addToPolicy(statement);
        return statement;
    }
}
exports.AssertionsProvider = AssertionsProvider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJvdmlkZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJwcm92aWRlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFDN0IsMkNBQTJDO0FBQzNDLGlEQUFpRDtBQUNqRCw2Q0FBdUM7QUFFdkMsaUdBQWlHO0FBQ2pHLDhEQUE4RDtBQUM5RCwyQ0FBdUM7QUFFdkMsSUFBSSxZQUFZLEdBQVEsU0FBUyxDQUFDO0FBRWxDOzs7OztHQUtHO0FBQ0gsTUFBYSxrQkFBbUIsU0FBUSxzQkFBUztJQUkvQyxZQUFZLEtBQWdCLEVBQUUsRUFBVTtRQUN0QyxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sT0FBTyxHQUFHLElBQUksTUFBTSxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSxvQkFBb0IsRUFBRTtZQUN2RSxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztZQUNuRSxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLElBQUksRUFBRSxzQ0FBc0M7WUFDNUMsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztTQUM3QixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsY0FBYyxHQUFHLE9BQU8sQ0FBQyxjQUFjLENBQUM7UUFDN0MsSUFBSSxDQUFDLFlBQVksR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0tBQ3pDO0lBRU0sTUFBTSxDQUFDLEdBQVE7UUFDcEIsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNSLE9BQU8sR0FBRyxDQUFDO1NBQ1o7UUFDRCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLEVBQUUsRUFBRTtZQUMvQyxRQUFRLENBQUMsRUFBRTtnQkFDVCxLQUFLLElBQUk7b0JBQ1AsT0FBTyxjQUFjLENBQUM7Z0JBQ3hCLEtBQUssS0FBSztvQkFDUixPQUFPLGVBQWUsQ0FBQztnQkFDekI7b0JBQ0UsT0FBTyxDQUFDLENBQUM7YUFDWjtRQUNILENBQUMsQ0FBQyxDQUFDO0tBQ0o7SUFFTSw2QkFBNkIsQ0FBQyxPQUFlLEVBQUUsR0FBVyxFQUFFLFNBQW9CO1FBQ3JGLElBQUksWUFBWSxLQUFLLFNBQVMsRUFBRTtZQUM5QiwyQkFBMkI7WUFDM0IsWUFBWSxHQUFHLE9BQU8sQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1NBQ25EO1FBQ0QsTUFBTSxHQUFHLEdBQUcsT0FBTyxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBQ2xDLE1BQU0sVUFBVSxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQyxNQUFNLENBQUMsSUFBSSxHQUFHLENBQUM7UUFDMUUsTUFBTSxTQUFTLEdBQUcsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzdELE1BQU0sU0FBUyxHQUFHLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN4QyxPQUFPLEVBQUUsQ0FBQyxHQUFHLFVBQVUsSUFBSSxTQUFTLEVBQUUsQ0FBQztZQUN2QyxTQUFTLEVBQUUsU0FBUyxJQUFJLENBQUMsR0FBRyxDQUFDO1NBQzlCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sU0FBUyxDQUFDO0tBQ2xCO0NBQ0Y7QUFsREQsZ0RBa0RDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCB7IER1cmF0aW9uIH0gZnJvbSAnYXdzLWNkay1saWInO1xuXG4vLyBrZWVwIHRoaXMgaW1wb3J0IHNlcGFyYXRlIGZyb20gb3RoZXIgaW1wb3J0cyB0byByZWR1Y2UgY2hhbmNlIGZvciBtZXJnZSBjb25mbGljdHMgd2l0aCB2Mi1tYWluXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgbm8tZHVwbGljYXRlLWltcG9ydHMsIGltcG9ydC9vcmRlclxuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5cbmxldCBTREtfTUVUQURBVEE6IGFueSA9IHVuZGVmaW5lZDtcblxuLyoqXG4gKiBSZXByZXNlbnRzIGFuIGFzc2VydGlvbnMgcHJvdmlkZXIuIFRoZSBjcmVhdGVzIGEgc2luZ2xldG9uZVxuICogTGFtYmRhIEZ1bmN0aW9uIHRoYXQgd2lsbCBjcmVhdGUgYSBzaW5nbGUgZnVuY3Rpb24gcGVyIHN0YWNrXG4gKiB0aGF0IHNlcnZlcyBhcyB0aGUgY3VzdG9tIHJlc291cmNlIHByb3ZpZGVyIGZvciB0aGUgdmFyaW91c1xuICogYXNzZXJ0aW9uIHByb3ZpZGVyc1xuICovXG5leHBvcnQgY2xhc3MgQXNzZXJ0aW9uc1Byb3ZpZGVyIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgcHVibGljIHJlYWRvbmx5IHNlcnZpY2VUb2tlbjogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IGhhbmRsZXIgPSBuZXcgbGFtYmRhLlNpbmdsZXRvbkZ1bmN0aW9uKHRoaXMsICdBc3NlcnRpb25zUHJvdmlkZXInLCB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ2xhbWJkYS1oYW5kbGVyJykpLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE0X1gsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICB1dWlkOiAnMTQ4ODU0MWEtN2IyMy00NjY0LTgxYjYtOWI0NDA4MDc2YjgxJyxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMiksXG4gICAgfSk7XG5cbiAgICB0aGlzLmdyYW50UHJpbmNpcGFsID0gaGFuZGxlci5ncmFudFByaW5jaXBhbDtcbiAgICB0aGlzLnNlcnZpY2VUb2tlbiA9IGhhbmRsZXIuZnVuY3Rpb25Bcm47XG4gIH1cblxuICBwdWJsaWMgZW5jb2RlKG9iajogYW55KTogYW55IHtcbiAgICBpZiAoIW9iaikge1xuICAgICAgcmV0dXJuIG9iajtcbiAgICB9XG4gICAgcmV0dXJuIEpTT04ucGFyc2UoSlNPTi5zdHJpbmdpZnkob2JqKSwgKF9rLCB2KSA9PiB7XG4gICAgICBzd2l0Y2ggKHYpIHtcbiAgICAgICAgY2FzZSB0cnVlOlxuICAgICAgICAgIHJldHVybiAnVFJVRTpCT09MRUFOJztcbiAgICAgICAgY2FzZSBmYWxzZTpcbiAgICAgICAgICByZXR1cm4gJ0ZBTFNFOkJPT0xFQU4nO1xuICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgIHJldHVybiB2O1xuICAgICAgfVxuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIGFkZFBvbGljeVN0YXRlbWVudEZyb21TZGtDYWxsKHNlcnZpY2U6IHN0cmluZywgYXBpOiBzdHJpbmcsIHJlc291cmNlcz86IHN0cmluZ1tdKTogaWFtLlBvbGljeVN0YXRlbWVudCB7XG4gICAgaWYgKFNES19NRVRBREFUQSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmVcbiAgICAgIFNES19NRVRBREFUQSA9IHJlcXVpcmUoJy4vc2RrLWFwaS1tZXRhZGF0YS5qc29uJyk7XG4gICAgfVxuICAgIGNvbnN0IHNydiA9IHNlcnZpY2UudG9Mb3dlckNhc2UoKTtcbiAgICBjb25zdCBpYW1TZXJ2aWNlID0gKFNES19NRVRBREFUQVtzcnZdICYmIFNES19NRVRBREFUQVtzcnZdLnByZWZpeCkgfHwgc3J2O1xuICAgIGNvbnN0IGlhbUFjdGlvbiA9IGFwaS5jaGFyQXQoMCkudG9VcHBlckNhc2UoKSArIGFwaS5zbGljZSgxKTtcbiAgICBjb25zdCBzdGF0ZW1lbnQgPSBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbYCR7aWFtU2VydmljZX06JHtpYW1BY3Rpb259YF0sXG4gICAgICByZXNvdXJjZXM6IHJlc291cmNlcyB8fCBbJyonXSxcbiAgICB9KTtcbiAgICB0aGlzLmdyYW50UHJpbmNpcGFsLmFkZFRvUG9saWN5KHN0YXRlbWVudCk7XG4gICAgcmV0dXJuIHN0YXRlbWVudDtcbiAgfVxufVxuIl19