"""
CLI tool for creating a test document from a test directory or JSON
"""

import argparse
import os
import logging
import tempfile
import shutil

import k3logging

from k3testdocumentation_generator import __version__, generate_json_from_file_system, render_doc_template_with_dict, generate_pdf_document_from_html# generate_html_test_documentation_from_json, generate_pdf_test_documentation_from_json
from k3testdocumentation_generator.resources import TEST_DOCUMENTATION_TEMPLATE_PATH
import json



__author__ = 'Joachim Kestner <joachim.kestner@khoch3.de>'

logger = logging.getLogger(__name__)

def main():
    parser = argparse.ArgumentParser(description=__doc__+"\n\nAuthor: {}\nVersion: {}".format(__author__,__version__), formatter_class=argparse.RawDescriptionHelpFormatter)
    #parser.add_argument("-f", "--flag", action="store_true", help="Example argparse of a choice")

    parser.add_argument("-te", "--template_engine", default="jinja2", choices=["jinja2"], help="The templating engine. Currently only jinja2 is supported.")
    parser.add_argument("-to", "--template_output", default="html", choices=["html", "latex"], help="The content language within the template. Only html is supported.")
    parser.add_argument("-ot", "--output_type", default="pdf", choices=["pdf", "json", "raw"], help="The output format. JSON would output the data before it goes into the templating engine. Raw is the raw result after the templating engine has run. Default is 'pdf'")
    parser.add_argument("-o", "--output", help="Output file path. If not set a name will be generated by: basename(input) + '.' + output_type.lower()")
    parser.add_argument("--template", help="The path to an alternative template")
    parser.add_argument("input", help="Input to generate documentation from. Can either be a directory containing the specified structure or an appropriate JSON")
    
    k3logging.set_parser_log_arguments(parser)
    
    args = parser.parse_args()
    
    k3logging.eval_parser_log_arguments(args)
    
    if args.template_output == "latex":
        logger.error("Latex as a template output is not supported atm.")
        return
    
    with tempfile.TemporaryDirectory() as tmpdir:
    
        if os.path.isdir(args.input):
            tmpjsonfile = os.path.join(tmpdir, "t.json")
            generate_json_from_file_system(args.input, tmpjsonfile)
            logger.info(f"Generated json file '{tmpjsonfile}'")
            jsonFilePath = tmpjsonfile
        else:
            jsonFilePath = args.input
        
        if not args.output:
            if args.input.endswith("/"):
                outputFilePath = os.path.basename(args.input[:-1]) + "." + args.output_type.lower()
            else:
                outputFilePath = os.path.basename(args.input) + "." + args.output_type.lower()
        else:
            outputFilePath = args.output
        
        if args.output_type == "json":
            shutil.copy(jsonFilePath, outputFilePath)
        else:
            tmplPath = args.template if args.template else TEST_DOCUMENTATION_TEMPLATE_PATH
            with open(tmplPath) as fh:
                tmplStr = fh.read()
            with open(jsonFilePath) as fh:
                inputDict = json.load(fh)
            rawStr = render_doc_template_with_dict(inputDict, tmplStr, args.template_engine)
            if args.output_type == "raw":
                with open(outputFilePath, "w") as fh:
                    fh.write(rawStr)
            elif args.output_type == "pdf":
                generate_pdf_document_from_html(rawStr, outputFilePath)
            else:
                raise RuntimeError("Invalid output type")
        logger.info(f"Output written to '{outputFilePath}'")
