# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['raindropiopy',
 'raindropiopy.cli',
 'raindropiopy.cli.commands',
 'raindropiopy.cli.models']

package_data = \
{'': ['*']}

install_requires = \
['email-validator>=1.3.1,<2.0.0',
 'humanize>=4.4.0,<5.0.0',
 'prompt-toolkit>=3.0.36,<4.0.0',
 'pydantic>=1.10.4,<2.0.0',
 'pyfiglet>=0.8.post1,<0.9',
 'python-dateutil>=2.8.2,<3.0.0',
 'python-dotenv>=0.21.0,<0.22.0',
 'requests-oauthlib>=1.3.1,<2.0.0',
 'rich>=13.2.0,<14.0.0',
 'tomli>=2.0.1,<3.0.0']

entry_points = \
{'console_scripts': ['raindrop-io-py = raindropiopy.cli.cli:main']}

setup_kwargs = {
    'name': 'raindrop-io-py',
    'version': '0.1.0',
    'description': 'API and terminal-based CLI for Raindrop.io bookmark manager',
    'long_description': '[![version](https://img.shields.io/badge/python-3.10+-green)](https://www.python.org/)\n[![pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit)](https://github.com/pre-commit/pre-commit)\n[![license](https://img.shields.io/badge/License-MIT-green.svg)](https://github.com/PBorocz/raindrop-io-py/blob/trunk/LICENSE)\n|docs|\n\n# Raindrop-IO-py\n\nPython wrapper for the API to the [Raindrop.io](https://raindrop.io) Bookmark Manager as well as a simple command-line interface to perform common operations.\n\n## Background\n\nI wanted to use an existing API for the Raindrop Bookmark Manager ([python-raindropio](https://github.com/atsuoishimoto/python-raindropio)) to perform some bulk operations through a simple command-line interface. However, the API available was incomplete and didn\'t contain any user-interface. Thus, this is a _fork_ and significant extension of [python-raindropio](https://github.com/atsuoishimoto/python-raindropio) (ht [Atsuo Ishimoto](https://github.com/atsuoishimoto)).\n\nThis package includes:\n\n-   An API providing access to the Raindrop environment. For instance: create, update, delete link/file-based Raindrops; create, update delete Raindrop collections, tags etc.\n-   A terminal-based user-interface that both tests the API as well as providing (me) a fast, simple interface to my Raindrop collections.\n\n## Status\n\nAs the API layer is based on a fork of an existing package, it\'s reasonably stable. However, the command-line interface (CLI) is brand new (and lacking tests, ie. "works for me!" ;-).\n\n## Requirements\n\nRequires Python 3.10 or later (well, at least I\'m developing against 3.10.9).\n\n## Install\n\n```shell\n[.venv] python -m pip install raindrop-io-py\n```\n\n## Setup\n\nTo use this package, besides your own account on [Raindrop](https://raindrop.io), you\'ll need to create an `integration app` on the Raindrop.io site from which you can create API token(s).\n\n-   Go to [<https://app.draindrop.api/settings/integrations>](https://app.raindrop.io/settings/integrations) and select `+ create new app`.\n\n-   Give it a descriptive name and then select the app you just created.\n\n-   Select `Create test token` and copy the token provided. Note that the basis for calling it a _test_ token is that it only gives you access to bookmarks within *your own account*. Raindrop allows you to use their API against other people\'s environments using oAuth (see untested/unsupported `flask_oauth.py` file in /examples)\n\n-   Save your token into your environment (we use python-dotenv so a simple .env/.envrc file containing your token should suffice), for example:\n\n```shell\n# If you use direnv or it\'s equivalent, place something like this in a .env file:\nRAINDROP_TOKEN=01234567890-abcdefghf-aSample-API-Token-01234567890-abcdefghf\n\n# Or for bash:\nexport RAINDROP_TOKEN=01234567890-abcdefghf-aSample-API-Token-01234567890-abcdefghf\n\n# Or for fish:\nset -gx RAINDROP_TOKEN 01234567890-abcdefghf-aSample-API-Token-01234567890-abcdefghf\n\n# etc...\n```\n\n## Examples\n\nA full suite of examples are provided in the `examples` directory. Each can be run independently as:\n\n```shell\n[.venv] % python examples/list_collections.py\n```\n\nor a wrapper script is available to run all of them, in logical order with a small wait to be nice to Raindrop\'s API:\n\n```shell\n[.venv] % python examples/RUN_ALL.py\n```\n\n### API Examples\n\nHere are a few examples of API usage (all of these should be able to be executed "as-is"):\n\n#### Display All Collections and **Unsorted** Bookmarks:\n\nThis example shows the intended usage of the API as a context-manager, from which any number of calls can be made:\n\n```python\nimport os\n\nfrom dotenv import load_dotenv\n\nfrom raindropiopy.api import API, Collection, CollectionRef, Raindrop\n\nload_dotenv()\n\nwith API(os.environ["RAINDROP_TOKEN"]) as api:\n\n    print("Current Collections:"\n    for collection in Collection.get_collections(api):\n        print(collection.title)\n\n    print("\\nUnsorted Raindrop Bookmarks:"\n    for item in Raindrop.search(api, collection=CollectionRef.Unsorted):\n        print(item.title)\n```\n\n#### Create a New Raindrop Bookmark to a URL\n\n```python\nimport os\n\nfrom dotenv import load_dotenv\n\nfrom raindropiopy.api import API, Raindrop\n\nload_dotenv()\n\nwith API(os.environ["RAINDROP_TOKEN"]) as api:\n    link, title = "https://www.python.org/", "Our Benevolent Dictator\'s Creation"\n    print(f"Creating Raindrop to: \'{link}\' with title: \'{title}\'...", flush=True, end="")\n    raindrop = Raindrop.create_link(api, link=link, title=title, tags=["abc", "def"])\n    print(f"Done, id={raindrop.id}")\n\n```\n\n(after this has executed, go to your Raindrop.io environment (site or app) and you should see this Raindrop to python.org available)\n\n#### Create a New Raindrop Collection\n\n```python\nimport os\nimport sys\nfrom datetime import datetime\nfrom getpass import getuser\n\nfrom dotenv import load_dotenv\n\nfrom raindropiopy.api import API, Collection\n\nload_dotenv()\n\nwith API(os.environ["RAINDROP_TOKEN"]) as api:\n    title = f"TEST Collection ({getuser()}@{datetime.now():%Y-%m-%dT%H:%M:%S})"\n    print(f"Creating collection: \'{title}\'...", flush=True, end="")\n    collection = Collection.create(api, title=title)\n    print(f"Done, {collection.id=}.")\n```\n\n(after this has executed, go to your Raindrop.io environment (site or app) and you should see this collection available)\n\n## Command-Line Interface Usage\n\n```shell\n[.venv] % raindropiopy\n```\n\nNote: remember to setup `RAINDROP-TOKEN` in your environment!\n\n## Documentation\n\nWe use [Sphinx](https://www.sphinx-doc.org/en/master/index.html) with [Google-style docstrings](https://www.sphinx-doc.org/en/master/usage/extensions/example_google.html) to document our API. Documentation is hosted by [ReadTheDocs](https://readthedocs.org/) and can be found [here](https://raindrop-io-py.readthedocs.io/en/latest/).\n\n## Acknowledgments\n\n[python-raindropio](https://github.com/atsuoishimoto/python-raindropio) from [Atsuo Ishimoto](https://github.com/atsuoishimoto).\n\n## License\n\nThe project is licensed under the MIT License.\n\n## Release History\n\n### Unreleased\n\n### v0.1.0 - 2023-02-16\n\n- CHANGED: `Raindrop.create_file` to handle `collection` argument consistent with `Raindrop.create_link`, specifically, either a `Collection`, `CollectionRef` or direct integer collection_id.\n- ADDED: Beginning of documentation suite on Read-The-Docs.\n\n### v0.0.15 - 2023-02-11\n\n- CHANGED: `Raindrop.search_paged` is now hidden (can\'t see a reason to explicitly use it over `Raindrop.search`)\n- CHANGED: Several attributes that, while allowed to be set by RaindropIO\'s API, are now *not* able to be set by this API. For example, you shouldn\'t be able to change "time" by setting `created` or `last_update` fields on a Raindrop or Collection.\n- CHANGED: The `Collection`, `Raindrop` and `Tag` "remove" method is now "delete" to more accurately match with RaindropIO\'s API).\n\n### v0.0.14 - 2023-02-09\n\n- FIXED: `Raindrop.cache.size` and `Raindrop.cache.created` attributes are now optional (RaindropIO\'s API doesn\'t always provide them).\n- FIXED: README examples corrected to reflect simpler Raindrop.search call.\n\n### v0.0.13 - 2023-02-07\n\n- CHANGED: Cross-referenced the fields available from the Raindrop API with our API; most available but several optional ones skipped for now.\n- CHANGED: (Internal) Remove dependency on ["jashin"](https://github.com/sojin-project/jashin) library by moving to [pydantic](https://docs.pydantic.dev/) for all Raindrop API models.\n\n### v0.0.12 - 2023-02-06\n\n- CHANGED: (Internal) Move from README.org to README.md to allow PyPI to display project information correctly.\n\n### v0.0.11 - 2023-02-06\n\n- CHANGED: Raindrop search API call is now non-paged (the "paged" version is still available as `Raindrop.search_paged`).\n\n### v0.0.10 - 2023-02-05\n\n- ADDED: Ability to specify raindrop field: Description on a created Raindrop (either file or link-based).\n- ADDED: Ability to re-query existing search results (eg. after changes) and smoothed out post-search interactions.\n\n### v0.0.9 - 2023-02-04\n\n- ADDED: An ability to view, edit and delete raindrops returned from a search.\n- ADDED: A simple `RUN_ALL.py` script to the examples directory to...well, run all the examples in order!\n- CHANGED: The display of raindrops returned from a search to include tags and to only show Collection name if all raindrops are across multiple collections.\n\n### v0.0.8 - 2023-01-25\n\n- CHANGED: Added simple version method in root package:\n\n```python\nfrom raindropiopy import version\nprint(version())\n```\n\n### v0.0.7 - 2023-01-25\n\n- CHANGED: Moved from keeping README in markdown to org file format. Incorporated package\'s ChangeLog into README as well (at the bottom).\n- CHANGED: Added new manage.py release automation capability (internal only, nothing public-facing).\n\n### v0.0.6 - 2023-01-22\n\n- FIXED: CLI autocomplete now works again after adding support for "single-letter" command-shortcuts.\n- ADDED: A set of missing attributes to the Raindrop API model type, eg. file, cache etc. Only attribute still missing is "highlights".\n\n### v0.0.5 - 2023-01-21\n\n- ADDED: Support use of [Vulture](https://github.com/jendrikseipp/vulture) for dead-code analysis (not in pre-commit through due to conflict with ruff\'s McCabe complexity metric)\n- CHANGED: Moved internal module name to match that of package name. Since we couldn\'t use raindroppy as a package name on PyPI due to similarities with existing packages (one of which was for a **crypto** package), we renamed this package to raindrop-io-py. In concert, the internal module is now `raindropiopy`:\n\n```python\nfrom raindroiopy.api import API\n```\n\n- FIXED: Sample file upload specification in `examples/create_raindrop_file.py` is now correct.\n\n\n.. |docs| image:: https://readthedocs.org/projects/docs/badge/?version=latest\n    :alt: Documentation Status\n    :scale: 100%\n    :target: https://docs.readthedocs.io/en/latest/?badge=latest\n',
    'author': 'Peter Borocz',
    'author_email': 'peter.borocz+raindrop-io-py@google.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/PBorocz/raindrop-io-py',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
