"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const caching_1 = require("../../caching");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const tags_1 = require("../shared/tags");
const tarball_lambda_shared_1 = require("../shared/tarball.lambda-shared");
const constants_1 = require("./constants");
const framework_detection_lambda_shared_1 = require("./framework-detection.lambda-shared");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const CONFIG_BUCKET_NAME = env_lambda_shared_1.requireEnv('CONFIG_BUCKET_NAME');
    const CONFIG_FILE_KEY = env_lambda_shared_1.requireEnv('CONFIG_FILE_KEY');
    // Load configuration
    const { packageTags: packageTagsConfig, packageLinks: allowedLinks } = await getConfig(CONFIG_BUCKET_NAME, CONFIG_FILE_KEY);
    const codeArtifactProps = (function () {
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            return undefined;
        }
        const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
        const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
        const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
        return { endpoint, domain, domainOwner, apiEndpoint };
    })();
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws
            .s3()
            .getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        })
            .promise();
        const { integrity: integrityCheck } = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        let dotJsii;
        let packageJson;
        let licenseText;
        try {
            ({ dotJsii, packageJson, licenseText } = await tarball_lambda_shared_1.extractObjects(Buffer.from(tarball.Body), {
                dotJsii: { path: 'package/.jsii', required: true },
                packageJson: { path: 'package/package.json', required: true },
                licenseText: { filter: isLicenseFile },
            }));
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        let parsedAssembly;
        let constructFramework;
        let packageLicense;
        let packageName;
        let packageVersion;
        try {
            parsedAssembly = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            // needs `dependencyClosure`
            constructFramework = framework_detection_lambda_shared_1.detectConstructFramework(parsedAssembly);
            const { license, name, version } = parsedAssembly;
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            // Delete some fields not used by the client to reduce the size of the assembly.
            // See https://github.com/cdklabs/construct-hub-webapp/issues/691
            delete parsedAssembly.types;
            delete parsedAssembly.readme;
            delete parsedAssembly.dependencyClosure;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const packageJsonObj = JSON.parse(packageJson.toString('utf-8'));
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense, constructHub, } = packageJsonObj;
        if (packageJsonName !== packageName ||
            packageJsonVersion !== packageVersion ||
            packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        const packageLinks = allowedLinks.reduce((accum, { configKey, allowedDomains }) => {
            const pkgValue = constructHub === null || constructHub === void 0 ? void 0 : constructHub.packageLinks[configKey];
            if (!pkgValue) {
                return accum;
            }
            // check if value is in allowed domains list
            const url = new url_1.URL(pkgValue);
            if ((allowedDomains === null || allowedDomains === void 0 ? void 0 : allowedDomains.length) && !allowedDomains.includes(url.host)) {
                return accum;
            }
            // if no allow list is provided
            return { ...accum, [configKey]: pkgValue };
        }, {});
        const packageTags = packageTagsConfig.reduce((accum, tagConfig) => {
            const { condition, ...tagData } = tagConfig;
            if (tags_1.isTagApplicable(condition, packageJsonObj)) {
                return [...accum, tagData];
            }
            return accum;
        }, []);
        if (codeArtifactProps) {
            console.log('Publishing to the internal CodeArtifact...');
            try {
                const { publishConfig } = packageJsonObj;
                if (publishConfig) {
                    console.log('Not publishing to CodeArtifact due to the presence of publishConfig in package.json: ', publishConfig);
                }
                else {
                    await code_artifact_lambda_shared_1.codeArtifactPublishPackage(Buffer.from(tarball.Body), codeArtifactProps);
                }
            }
            catch (err) {
                console.error('Failed publishing to CodeArtifact: ', err);
            }
        }
        const metadata = {
            constructFramework,
            date: payload.time,
            licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8'),
            packageLinks,
            packageTags,
        };
        const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
        console.log(`Writing assembly at ${assemblyKey}`);
        console.log(`Writing package at  ${packageKey}`);
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: caching_1.CacheStrategy.default().toString(),
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: caching_1.CacheStrategy.default().toString(),
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws
            .s3()
            .putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: Buffer.from(JSON.stringify(parsedAssembly), 'utf-8'),
            CacheControl: caching_1.CacheStrategy.default().toString(),
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        })
            .promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws
            .stepFunctions()
            .startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        })
            .promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return (possibleExtensions.has(ext.toLowerCase()) &&
        path_1.basename(fileName, ext).toUpperCase() === 'LICENSE');
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/gi, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
/**
 * Looks for the ingestion configuration file in the passed bucket and parses
 * it. If it is not found or invalid then a default is returned.
 */
async function getConfig(bucket, key) {
    var _a;
    const defaultConfig = {
        packageTags: [],
        packageLinks: [],
    };
    try {
        const req = await aws.s3().getObject({
            Bucket: bucket,
            Key: key,
        }).promise();
        const body = (_a = req === null || req === void 0 ? void 0 : req.Body) === null || _a === void 0 ? void 0 : _a.toString();
        if (body) {
            return JSON.parse(body);
        }
        return defaultConfig;
    }
    catch (e) {
        console.error(e);
        return defaultConfig;
    }
}
//# sourceMappingURL=data:application/json;base64,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