"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CouchChanges = void 0;
const events_1 = require("events");
const https_1 = require("https");
const url_1 = require("url");
const zlib_1 = require("zlib");
/**
 * A utility class that helps with traversing CouchDB database changes streams
 * in a promise-based, page-by-page manner.
 */
class CouchChanges extends events_1.EventEmitter {
    /**
     * @param baseUrl  the CouchDB endpoint URL.
     * @param database the name of the database for which changes are fetched.
     */
    constructor(baseUrl, database) {
        super();
        // Setting up for keep-alive connections.
        this.agent = new https_1.Agent({
            keepAlive: true,
            keepAliveMsecs: 5000,
            maxSockets: 4,
            timeout: 60000,
        });
        this.baseUrl = new url_1.URL(database, baseUrl);
    }
    /**
     * @returns summary informations about the database.
     */
    async info() {
        return await this.https('get', this.baseUrl);
    }
    /**
     * Obtains a batch of changes from the database.
     *
     * @param since     the sequence value since when history should be fetched.
     * @param batchSize the maximum amount of changes to return in a single page.
     *
     * @returns a page of changes.
     */
    async changes(since, { batchSize = 100 } = {}) {
        const changesUrl = new url_1.URL('_changes', this.baseUrl);
        changesUrl.searchParams.set('limit', batchSize.toFixed());
        changesUrl.searchParams.set('include_docs', 'true');
        changesUrl.searchParams.set('selector', '_filter');
        changesUrl.searchParams.set('since', since.toString());
        const filter = { name: { $gt: null } };
        return await this.https('post', changesUrl, filter);
    }
    /**
     * Makes an HTTPs request using the provided method, url, and optionally payload. This function
     * properly handles input that is received with `Content-Type: gzip` and automatically retries
     * typical transient errors (HTTP 5XX, ECONNRESET, etc...) with linear back-off and no maximum
     * retry count (this is used in Lambda functions, which de-facto caps the amount of attempts
     * that will be made due to the function time out).
     *
     * @param method the HTTP method used for the request (e.g: 'get', 'post', ...).
     * @param url    the URL to request.
     * @param body   an optional HTTP request payload, which will be JSON-encoded.
     *
     * @param attempt the request attempt number (used to determine back-off / retry).
     *
     * @returns the JSON-decoded response body.
     */
    https(method, url, body, attempt = 1) {
        return new Promise((ok, ko) => {
            const retry = () => setTimeout(() => {
                console.log(`Retrying ${method.toUpperCase()} ${url.toString()}`);
                this.https(method, url, body, attempt + 1).then(ok, ko);
            }, Math.min(500 * attempt, 5000));
            const headers = {
                'Accept': 'application/json',
                'Accept-Encoding': 'gzip',
            };
            if (body) {
                headers['Content-Type'] = 'application/json';
            }
            const req = https_1.request(url, {
                agent: this.agent,
                headers,
                method,
                port: 443,
                servername: url.hostname,
            }, (res) => {
                if (res.statusCode == null) {
                    const error = new Error(`[FATAL] Request failed: ${method.toUpperCase()} ${url.toString()}`);
                    Error.captureStackTrace(error);
                    return ko(error);
                }
                // Transient (server) errors:
                if (res.statusCode >= 500 && res.statusCode < 600) {
                    console.error(`[RETRYABLE] HTTP ${res.statusCode} (${res.statusMessage}) - ${method.toUpperCase()} ${url.toString()}`);
                    // Call again after a short back-off
                    return retry();
                }
                // Permanent (client) errors:
                if (res.statusCode >= 400 && res.statusCode < 500) {
                    const error = new Error(`[FATAL] HTTP ${res.statusCode} (${res.statusMessage}) - ${method.toUpperCase()} ${url.toString()}`);
                    Error.captureStackTrace(error);
                    return ko(error);
                }
                let data = Buffer.alloc(typeof res.headers['content-length'] === 'string'
                    ? Number.parseInt(res.headers['content-length'])
                    : 4096);
                let dataLength = 0;
                res.once('error', (err) => {
                    if (err.code === 'ECONNRESET') {
                        // Transient networking problem?
                        console.error(`[RETRYABLE] ${err.code} - ${method.toUpperCase()} ${url.toString()}`);
                        res.removeAllListeners('close');
                        retry();
                    }
                    else {
                        debugger;
                        ko(err);
                    }
                });
                res.on('data', (chunk) => {
                    const buffer = chunk = Buffer.from(chunk);
                    if (dataLength + buffer.length > data.length) {
                        // Buffer is not large enough, extend it to fit new data...
                        const existing = data;
                        data = Buffer.alloc(data.length + Math.max(buffer.length, 4096));
                        existing.copy(data);
                    }
                    buffer.copy(data, dataLength);
                    dataLength += buffer.length;
                });
                res.once('close', () => {
                    // Ensure buffer is trimmed to correct length.
                    data = data.subarray(0, dataLength);
                    try {
                        if (res.headers['content-encoding'] === 'gzip') {
                            data = zlib_1.gunzipSync(data);
                        }
                        ok(JSON.parse(data.toString('utf-8')));
                    }
                    catch (error) {
                        if (error.code === 'Z_BUF_ERROR') {
                            // Truncated payload... Connection cut too early?
                            console.error(`[RETRYABLE] Z_BUF_ERROR (${error.message}) - ${method.toUpperCase()} ${url.toString()}`);
                            retry();
                        }
                        else {
                            debugger;
                            ko(error);
                        }
                    }
                });
            });
            req.end(body && JSON.stringify(body, null, 2));
        });
    }
}
exports.CouchChanges = CouchChanges;
//# sourceMappingURL=data:application/json;base64,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