"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const packaging_1 = require("./packaging");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
class Bundling {
    constructor(props) {
        const { entry, runtime, architecture = aws_lambda_1.Architecture.X86_64, outputPathSuffix = '', image, poetryIncludeHashes, commandHooks, } = props;
        const outputPath = path.posix.join(aws_cdk_lib_1.AssetStaging.BUNDLING_OUTPUT_DIR, outputPathSuffix);
        const bundlingCommands = this.createBundlingCommand({
            entry,
            inputDir: aws_cdk_lib_1.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: outputPath,
            poetryIncludeHashes,
            commandHooks,
        });
        this.image = image ?? aws_cdk_lib_1.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...props.buildArgs,
                IMAGE: runtime.bundlingImage.image,
            },
            platform: architecture.dockerPlatform,
        });
        this.command = props.command ?? ['bash', '-c', chain(bundlingCommands)];
        this.entrypoint = props.entrypoint;
        this.volumes = props.volumes;
        this.volumesFrom = props.volumesFrom;
        this.environment = props.environment;
        this.workingDirectory = props.workingDirectory;
        this.user = props.user;
        this.securityOpt = props.securityOpt;
        this.network = props.network;
    }
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.entry, {
            assetHash: options.assetHash,
            assetHashType: options.assetHashType,
            exclude: exports.DEPENDENCY_EXCLUDES,
            bundling: options.skip ? undefined : new Bundling(options),
        });
    }
    createBundlingCommand(options) {
        const packaging = packaging_1.Packaging.fromEntry(options.entry, options.poetryIncludeHashes);
        let bundlingCommands = [];
        bundlingCommands.push(...options.commandHooks?.beforeBundling(options.inputDir, options.outputDir) ?? []);
        bundlingCommands.push(`cp -rTL ${options.inputDir}/ ${options.outputDir}`);
        bundlingCommands.push(`cd ${options.outputDir}`);
        bundlingCommands.push(packaging.exportCommand ?? '');
        if (packaging.dependenciesFile) {
            bundlingCommands.push(`python -m pip install -r ${packaging_1.DependenciesFile.PIP} -t ${options.outputDir}`);
        }
        bundlingCommands.push(...options.commandHooks?.afterBundling(options.inputDir, options.outputDir) ?? []);
        return bundlingCommands;
    }
}
exports.Bundling = Bundling;
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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