"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Device Servicer*

:details: DeviceServicer:
    Used to acquire general device status and individual stirrer position status of the bioREACTOR48.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 12.02.2020

:file:    DeviceServicer_simulation.py
:authors: Lukas Bromig

:date: (creation)          2020-04-16T10:18:48.774234
:date: (last modification) 2020-04-16T10:18:48.774234

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import inspect      # used for status determination
import time         # used for observables
import uuid         # used for observables
import grpc         # used for type hinting only
import random
import types 

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import DeviceServicer_pb2 as DeviceServicer_pb2
# from .gRPC import DeviceServicer_pb2_grpc as DeviceServicer_pb2_grpc

# import default arguments
from .DeviceServicer_default_arguments import default_dict


# noinspection PyPep8Naming,PyUnusedLocal
class DeviceServicerSimulation:
    """
    Implementation of the *Device Servicer* in *Simulation* mode
        This is a BioREACTOR48 Service
    """

    def __init__(self, status, start_time):
        """Class initializer"""
        self.start_time = start_time
        self.status = status
        self.TotalReactors = 48
        self.BarReactors = 8
        self.BarNumber = 6
        self.DesignatedToFail = []
        for i in range(self.TotalReactors):
            k=random.choice([True, False, False, False, False, False, False, False]) # 1/8th of the reactors will fail
            if k == True:
                FailTime = random.random() # defines the time at which the stirred will, fail, ranges from 0 to 1 hour process time 
                self.DesignatedToFail.append(FailTime)
            else:
                self.DesignatedToFail.append(k)
 
        logging.debug(f'The fail scheme is {self.DesignatedToFail}')
        logging.debug('Started server in mode: {mode}'.format(mode='Simulation'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def _get_function_name(self):
        return inspect.stack()[1][3]

    def GetLog(self, request, context: grpc.ServicerContext) \
            -> silaFW_pb2.CommandConfirmation:
        """
        Executes the observable command "Get Log"
            Get the current status of the device from the state machine of the SiLA server.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A command confirmation object with the following information:
            commandId: A command id with which this observable command can be referenced in future calls
            lifetimeOfExecution: The (maximum) lifetime of this command call.
        """
    
        # initialise default values
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        lifetime_of_execution: silaFW_pb2.Duration = None
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        # TODO:
        #   Execute the actual command
        #   Optional: Generate a lifetime_of_execution
    
        # respond with UUID and lifetime of execution
        command_uuid = silaFW_pb2.CommandExecutionUUID(value=str(uuid.uuid4()))
        if lifetime_of_execution is not None:
            return silaFW_pb2.CommandConfirmation(
                commandExecutionUUID=command_uuid,
                lifetimeOfExecution=lifetime_of_execution
            )
        else:
            return silaFW_pb2.CommandConfirmation(
                commandExecutionUUID=command_uuid
            )
    
    def GetLog_Info(self, request, context: grpc.ServicerContext) \
            -> silaFW_pb2.ExecutionInfo:
        """
        Returns execution information regarding the command call :meth:`~.GetLog`.
    
        :param request: A request object with the following properties
            commandId: The UUID of the command executed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: An ExecutionInfo response stream for the command with the following fields:
            commandStatus: Status of the command (enumeration)
            progressInfo: Information on the progress of the command (0 to 1)
            estimatedRemainingTime: Estimate of the remaining time required to run the command
            updatedLifetimeOfExecution: An update on the execution lifetime
        """
        # Get the UUID of the command
        command_uuid = request.value
    
        # Get the current state
        execution_info = self._get_command_state(command_uuid=command_uuid)
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        # construct the initial return dictionary in case while is not executed
        return_values = {'commandStatus': execution_info.commandStatus}
        if execution_info.HasField('progressInfo'):
            return_values['progressInfo'] = execution_info.progressInfo
        if execution_info.HasField('estimatedRemainingTime'):
            return_values['estimatedRemainingTime'] = execution_info.estimatedRemainingTime
        if execution_info.HasField('updatedLifetimeOfExecution'):
            return_values['updatedLifetimeOfExecution'] = execution_info.updatedLifetimeOfExecution
    
        # we loop only as long as the command is running
        while execution_info.commandStatus == silaFW_pb2.ExecutionInfo.CommandStatus.waiting \
                or execution_info.commandStatus == silaFW_pb2.ExecutionInfo.CommandStatus.running:
            # TODO:
            #   Evaluate the command status --> command_status. Options:
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.running
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.finishedSuccessfully
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.finishedWithError
            #   Optional:
            #       * Determine the progress (progressInfo)
            #       * Determine the estimated remaining time
            #       * Update the Lifetime of execution
    
            # Update all values
            execution_info = self._get_command_state(command_uuid=command_uuid)
    
            # construct the return dictionary
            return_values = {'commandStatus': execution_info.commandStatus}
            if execution_info.HasField('progressInfo'):
                return_values['progressInfo'] = execution_info.progressInfo
            if execution_info.HasField('estimatedRemainingTime'):
                return_values['estimatedRemainingTime'] = execution_info.estimatedRemainingTime
            if execution_info.HasField('updatedLifetimeOfExecution'):
                return_values['updatedLifetimeOfExecution'] = execution_info.updatedLifetimeOfExecution
    
            yield silaFW_pb2.ExecutionInfo(**return_values)
    
            # we add a small delay to give the client a chance to keep up.
            time.sleep(0.5)
        else:
            # one last time yield the status
            yield silaFW_pb2.ExecutionInfo(**return_values)
    
    def GetLog_Result(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetLog_Responses:
        """
        Returns the final result of the command call :meth:`~.GetLog`.
    
        :param request: A request object with the following properties
            CommandExecutionUUID: The UUID of the command executed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentLogLevel (Current Log Level): The current log level of the latest logs , retrieved from the SiLA server log file.
            request.CurrentLogTimestamp (Current Log Timestamp): The current log timestamp of the latest logs , retrieved from the SiLA server log file.
            request.CurrentLogMessage (Current Log Level): The current log level of the latest logs , retrieved from the SiLA server log file.
        """
    
        # initialise the return value
        return_value: DeviceServicer_pb2.GetLog_Responses = None
    
        # Get the UUID of the command
        command_uuid = request.value

        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        log_line = globals.get_log_from_queue()

        par_dict = {
            'CurrentLogLevel': silaFW_pb2.Integer(value=0),
            # 'CurrentLogTimestamp': silaFW_pb2.Timestamp(value=version),
            'CurrentLogMessage': silaFW_pb2.String(value=log_line)
        }
        return_value = DeviceServicer_pb2.GetLog_Responses(**par_dict)

        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.GetLog_Responses(
                **default_dict['GetLog_Responses']
            )
    
        return return_value

    def GetDeviceStatus(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetDeviceStatus_Responses:
        """
        Executes the unobservable command "Get Device Status"
            Get the current status of the device, software version, control mode, bar number and unit address.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): The overall device status. ER or OK.
            request.Version (Software Version): Software version number
            request.Mode (Mode): Operation mode. REM (remote) or MAN (manual) or OFF (offline).
            request.BarConnection (Bar Connection): Bar number connected or not connected. (1 = connected, 0 = not connected)
            request.Address (Address): UnitAddress
        """
    
        # initialise the return value
        return_value = None
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        command = 'SendStatus\r\n'

        try:
            read = 'OK_8602_REM101010_A'
            split = read.split('_')
            status = split[0]
            version = int(split[1])
            mode = split[2][0:3]
            connection = split[2][3:]
            address = split[3]
            address = 1  # Change address data type to string if response is A-Z!

            par_dict = {
                    'Status': silaFW_pb2.String(value=status),
                    'Version': silaFW_pb2.Integer(value=version),
                    'Mode': silaFW_pb2.String(value=mode),
                    'BarConnection': silaFW_pb2.String(value=connection),
                    'Address': silaFW_pb2.Integer(value=address)
            }
            return_value = DeviceServicer_pb2.GetDeviceStatus_Responses(**par_dict)
            logging.debug('Executed command GetDeviceStatus in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except ConnectionError:
            logging.exception('Communication failed executing the command: {command}'.format(command=command))
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.GetDeviceStatus_Responses(
                **default_dict['GetDeviceStatus_Responses']
            )
    
        return return_value

    def GetReactorStatus(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetReactorStatus_Responses:
        """
        Executes the unobservable command "Get Reactor Status"
            Get the current status of all 48 reactors. Check if stirrer is still running. 1 = stirring, 0 = not stirring
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ReactorStatus (Reactor Status): Get the current status of all 48 reactors. Check if stirrer is still running. 1 = stirring, 0 = not stirring
        """
                
            
        # initialise the return value
        return_value = None
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        par_dict = {'ReactorStatus': []}
        l = 0
        for j in range(1, self.BarNumber+1, 1):
            command = f'SendStatusBar{j}\r\n'
            try:
                read = f'OK_{j}_01111101_A'
                split = read.split('_')
                status = split[0]
                bar = int(split[1])
                reactors = split[2]
                address = split[3]
                response = list()
                
                
                for i in range(0, len(reactors), 1):
                    runtime_h = (time.time() - self.start_time) / (60*60)
                    logging.debug(f'The current runtime is : {runtime_h}')
                    logging.debug(f'The curr bool is {self.DesignatedToFail[l]}')
                    
                    if bool(self.DesignatedToFail[l]) != False: # outer if statement checks if the entry in the list isnt a bool (the stirrer will never fail)
                        if runtime_h > self.DesignatedToFail[l]: # inner if statement checks if the runtime of the stirrer has elapsed
                            par_dict['ReactorStatus'].append(silaFW_pb2.Boolean(value= False))
                        else: 
                            par_dict['ReactorStatus'].append(silaFW_pb2.Boolean(value= True))
                        
                    else:
                        par_dict['ReactorStatus'].append(silaFW_pb2.Boolean(value= True))
                    l = l +1 # additional loop counter to iterate over all reactor positions
                    #par_dict['ReactorStatus'].append(silaFW_pb2.Boolean(value=bool(int(reactors[i]))))
                    
                return_value = DeviceServicer_pb2.GetReactorStatus_Responses(**par_dict)
                logging.debug('Executed command GetReactorStatus in mode: {mode} with response: {response}'
                              .format(mode='Simulation', response=read))
            except ConnectionError:
                logging.exception('Communication failed executing the command: {command}'.format(command=command))
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.GetReactorStatus_Responses(
                **default_dict['GetReactorStatus_Responses']
            )
    
        return return_value

    def Subscribe_CurrentStatus(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.Subscribe_CurrentStatus_Responses:
        """
        Requests the observable property Current Status
            Get the current status of the device from the internal state machine of the SiLA server.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.CurrentStatus (Current Status): Get the current status of the device from the internal state machine of the SiLA server.
        """
    
        # initialise the return value
        return_value: DeviceServicer_pb2.Subscribe_CurrentStatus_Responses = None
    
        # we could use a timeout here if we wanted
        while True:
            par_dict = {
                'CurrentStatus': silaFW_pb2.String(value=self.status)
            }
            return_value = DeviceServicer_pb2.Subscribe_CurrentStatus_Responses(**par_dict)
    
            # create the default value
            if return_value is None:
                return_value = DeviceServicer_pb2.Subscribe_CurrentStatus_Responses(
                    **default_dict['Subscribe_CurrentStatus_Responses']
                )
            time.sleep(1)
            yield return_value

    def Get_BarNumber(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.Get_BarNumber_Responses:
        """
        Requests the unobservable property Bar Number
            Number of stirrer bars available. Default = 6.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.BarNumber (Bar Number): Number of stirrer bars available. Default = 6.
        """
    
        # initialise the return value
        return_value: DeviceServicer_pb2.Get_BarNumber_Responses = None
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        par_dict = {
            'BarNumber': silaFW_pb2.Integer(value=self.BarNumber)
        }
        return_value = DeviceServicer_pb2.Get_BarNumber_Responses(**par_dict)
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.Get_BarNumber_Responses(
                **default_dict['Get_BarNumber_Responses']
            )
    
        return return_value
    
    def Get_BarReactors(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.Get_BarReactors_Responses:
        """
        Requests the unobservable property BarReactorsr
            Number of reactors per bar. Default = 8.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.BarReactors (BarReactorsr): Number of reactors per bar. Default = 8.
        """
    
        # initialise the return value
        return_value: DeviceServicer_pb2.Get_BarReactors_Responses = None
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        par_dict = {
            'BarReactors': silaFW_pb2.Integer(value=self.BarReactors)
        }
        return_value = DeviceServicer_pb2.Get_BarReactors_Responses(**par_dict)
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.Get_BarReactors_Responses(
                **default_dict['Get_BarReactors_Responses']
            )
    
        return return_value
    
    def Get_TotalReactors(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.Get_TotalReactors_Responses:
        """
        Requests the unobservable property Total Reactors
            Number of total reactors. Default = 6*8 = 48.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.TotalReactors (Total Reactors): Number of total reactors. Default = 6*8 = 48.
        """
    
        # initialise the return value
        return_value: DeviceServicer_pb2.Get_TotalReactors_Responses = None
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')
        par_dict = {
            'TotalReactors': silaFW_pb2.Integer(value=self.TotalReactors)
        }
        return_value = DeviceServicer_pb2.Get_TotalReactors_Responses(**par_dict)
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.Get_TotalReactors_Responses(
                **default_dict['Get_TotalReactors_Responses']
            )
    
        return return_value
