#!/usr/bin/env python3
"""
________________________________________________________________________

:PROJECT: SiLA2_python

*BlueVaryService*

:details: BlueVaryService:
    This is a BlueVary Off-Gas Analytics Service
           
:file:    BlueVaryService_server.py
:authors: Lukas Bromig

:date: (creation)          2020-04-16T10:18:59.332927
:date: (last modification) 2020-04-16T10:18:59.332927

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""
__version__ = "1.0"

import os
import sys
import logging
import argparse

# Import the main SiLA library
from sila2lib.sila_server import SiLA2Server

# Import gRPC libraries of features
from sila2lib_implementations.BlueVary.BlueVaryService.DeviceServicer.gRPC import DeviceServicer_pb2
from sila2lib_implementations.BlueVary.BlueVaryService.DeviceServicer.gRPC import DeviceServicer_pb2_grpc
# import default arguments for this feature
from sila2lib_implementations.BlueVary.BlueVaryService.DeviceServicer.DeviceServicer_default_arguments import default_dict as DeviceServicer_default_dict
from sila2lib_implementations.BlueVary.BlueVaryService.SensorServicer.gRPC import SensorServicer_pb2
from sila2lib_implementations.BlueVary.BlueVaryService.SensorServicer.gRPC import SensorServicer_pb2_grpc
# import default arguments for this feature
from sila2lib_implementations.BlueVary.BlueVaryService.SensorServicer.SensorServicer_default_arguments import default_dict as SensorServicer_default_dict
from sila2lib_implementations.BlueVary.BlueVaryService.CalibrationServicer.gRPC import CalibrationServicer_pb2
from sila2lib_implementations.BlueVary.BlueVaryService.CalibrationServicer.gRPC import CalibrationServicer_pb2_grpc
# import default arguments for this feature
from sila2lib_implementations.BlueVary.BlueVaryService.CalibrationServicer.CalibrationServicer_default_arguments import default_dict as CalibrationServicer_default_dict

# Import the servicer modules for each feature
from sila2lib_implementations.BlueVary.BlueVaryService.DeviceServicer.DeviceServicer_servicer import DeviceServicer
from sila2lib_implementations.BlueVary.BlueVaryService.SensorServicer.SensorServicer_servicer import SensorServicer
from sila2lib_implementations.BlueVary.BlueVaryService.CalibrationServicer.CalibrationServicer_servicer import CalibrationServicer

# import custom implementation libraries
import glob
import serial
import serial.tools.list_ports
import threading


class BlueVaryServiceServer(SiLA2Server):
    """
    This is a BlueVary Off-Gas Analytics Service
    """

    def __init__(self, cmd_args, ip: str = '127.0.0.1', port: int = 50001, simulation_mode: bool = False):
        """Class initializer"""
        super().__init__(
            name=cmd_args.server_name, description=cmd_args.description,
            server_type=cmd_args.server_type, server_uuid=None,
            version=__version__,
            vendor_url="",
            ip=ip, port=port,
            # ip="10.152.248.22", port=50001,
            key_file=cmd_args.encryption_key, cert_file=cmd_args.encryption_cert
        )

        logging.info(
            "Starting SiLA2 server with server name: {server_name}".format(
                server_name=cmd_args.server_name
            )
        )

        # Create a lock that prevent simultaneous requests to Device
        self.lock = threading.Lock()
        self.simulation_mode = simulation_mode

        # when re-writing this server decide if 'sys.platform.startswith' (used for BioR. & PreSens) is better than os.name
        if not self.simulation_mode:
            ports = []
            # Opening a serial port for data communication
            if os.name == 'nt':  # For Windows OS
                # Not available on BeagleBone/Debian
                ports = list(serial.tools.list_ports.comports(include_links=True))
            elif os.name == 'posix': # For linux based distributions
                # Available on BeagleBone
                ports = glob.glob('/dev/tty[U][S][B]*')
                # untested for OSX. Might be different to stated above
                # ports = ['/dev/tty.SLAB_USBtoUART*']

            if ports == []:
                logging.info(
                    "No available COM/USB port to establish connection: {ports}".format(ports=ports))
            else:
                self.ser = serial.Serial(port=ports[0], baudrate=19200, parity=serial.PARITY_NONE,
                                         stopbits=serial.STOPBITS_ONE, bytesize=serial.EIGHTBITS, timeout=5)
                logging.info(
                    "Connection established on port: {ports}".format(ports=ports[0]))

            # On Debian use the following and specify the Comport name!
            # ser = serial.Serial(port="/dev/ttyS0", baudrate = 9600,parity=serial.PARITY_NONE, stopbits=serial.STOPBITS_
            # ONE,bytesize=8,timeout=1, xonxoff=0)
        else:
            self.ser = []
        # registering features
        #  Register DeviceServicer
        self.DeviceServicer_servicer = DeviceServicer(ser=self.ser, lock=self.lock,
                                                      simulation_mode=self.simulation_mode)
        DeviceServicer_pb2_grpc.add_DeviceServicerServicer_to_server(
            self.DeviceServicer_servicer,
            self.grpc_server
        )
        self.add_feature(feature_id='DeviceServicer',
                         servicer=self.DeviceServicer_servicer,
                         data_path=os.path.join(
                             os.path.dirname(__file__),
                             'meta')
                         )
        #  Register SensorServicer
        self.SensorServicer_servicer = SensorServicer(ser=self.ser, lock=self.lock,
                                                      simulation_mode=self.simulation_mode)
        SensorServicer_pb2_grpc.add_SensorServicerServicer_to_server(
            self.SensorServicer_servicer,
            self.grpc_server
        )
        self.add_feature(feature_id='SensorServicer',
                         servicer=self.SensorServicer_servicer,
                         data_path=os.path.join(
                             os.path.dirname(__file__),
                             'meta')
                         )
        #  Register CalibrationServicer
        self.CalibrationServicer_servicer = CalibrationServicer(ser=self.ser, lock=self.lock,
                                                                simulation_mode=self.simulation_mode)
        CalibrationServicer_pb2_grpc.add_CalibrationServicerServicer_to_server(
            self.CalibrationServicer_servicer,
            self.grpc_server
        )
        self.add_feature(feature_id='CalibrationServicer',
                         servicer=self.CalibrationServicer_servicer,
                         data_path=os.path.join(
                             os.path.dirname(__file__),
                             'meta')
                         )

        self.simulation_mode = simulation_mode
        
        # starting and running the gRPC/SiLA2 server
        self.run()


def parse_command_line():
    """
    Just looking for commandline arguments
    """
    parser = argparse.ArgumentParser(description="A SiLA2 service: BlueVaryService")

    # Simple arguments for the server identification
    parser.add_argument('-s', '--server-name', action='store',
                        default="BlueVaryServiceDasgip2", help='start SiLA server with [server-name]')
    parser.add_argument('-t', '--server-type', action='store',
                        default="Unknown Type", help='start SiLA server with [server-type]')
    parser.add_argument('-d', '--description', action='store',
                        default="This is a BlueVary Off-Gas Analytics Service", help='SiLA server description')

    # Encryption
    parser.add_argument('-X', '--encryption', action='store', default=None,
                        help='The name of the private key and certificate file (without extension).')
    parser.add_argument('--encryption-key', action='store', default=None,
                        help='The name of the encryption key (*with* extension). Can be used if key and certificate '
                             'vary or non-standard file extensions are used.')
    parser.add_argument('--encryption-cert', action='store', default=None,
                        help='The name of the encryption certificate (*with* extension). Can be used if key and '
                             'certificate vary or non-standard file extensions are used.')

    parser.add_argument('-v', '--version', action='version', version='%(prog)s ' + __version__)

    parsed_args = parser.parse_args()

    # validate/update some settings
    #   encryption
    if parsed_args.encryption is not None:
        # only overwrite the separate keys if not given manually
        if parsed_args.encryption_key is None:
            parsed_args.encryption_key = parsed_args.encryption + '.key'
        if parsed_args.encryption_cert is None:
            parsed_args.encryption_cert = parsed_args.encryption + '.cert'

    return parsed_args
    

def init():
    if __name__ == '__main__':
        # or use logging.ERROR for less output
        logging.basicConfig(format='%(levelname)-8s| %(module)s.%(funcName)s: %(message)s', level=logging.DEBUG)

        args = parse_command_line()

        # generate SiLA2Server
        sila_server = BlueVaryServiceServer(cmd_args=args, ip='127.0.0.1', port=50001, simulation_mode=False)
        sys.exit(sila_server.run())


init()
