"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Sensor Servicer*

:details: SensorServicer:
    Read out the off-gas analytics sensor data for gas1 and gas2.Gas concentrations [vol.-%], pressure [mbar], humidity
    [%], temperature[C] and absolute humidity [vol.-%] are provided.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 20.05.2019
           
:file:    SensorServicer_servicer.py
:authors: Lukas Bromig

:date: (creation)          2020-04-16T10:18:59.311978
:date: (last modification) 2020-04-16T10:18:59.311978

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import grpc

# meta packages
from typing import Union

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2
from sila2lib.error_handling.server_err import SiLAError

# import gRPC modules for this feature
from .gRPC import SensorServicer_pb2 as SensorServicer_pb2
from .gRPC import SensorServicer_pb2_grpc as SensorServicer_pb2_grpc

# import simulation and real implementation
from .SensorServicer_simulation import SensorServicerSimulation
from .SensorServicer_real import SensorServicerReal


class SensorServicer(SensorServicer_pb2_grpc.SensorServicerServicer):
    """
    This is a BlueVary Off-Gas Analytics Service
    """
    implementation: Union[SensorServicerSimulation, SensorServicerReal]
    simulation_mode: bool

    def __init__(self,ser, lock, simulation_mode: bool = True):
        """
        Class initialiser.

        :param simulation_mode: Sets whether at initialisation the simulation mode is active or the real mode.
        """
        self.ser = ser
        self.lock = lock
        self.simulation_mode = simulation_mode
        if simulation_mode:
            self._inject_implementation(SensorServicerSimulation())
        else:
            self._inject_implementation(SensorServicerReal(ser = self.ser, lock = self.lock))

    def _inject_implementation(self,
                               implementation: Union[SensorServicerSimulation,
                                                     SensorServicerReal]
                               ) -> bool:
        """
        Dependency injection of the implementation used.
            Allows to set the class used for simulation/real mode.

        :param implementation: A valid implementation of the BlueVaryServiceServicer.
        """

        self.implementation = implementation
        return True

    def switch_to_simulation_mode(self):
        """Method that will automatically be called by the server when the simulation mode is requested."""
        self.simulation_mode = True
        self._inject_implementation(SensorServicerSimulation())

    def switch_to_real_mode(self):
        """Method that will automatically be called by the server when the real mode is requested."""
        self.simulation_mode = False
        self._inject_implementation(SensorServicerReal(ser = self.ser, lock = self.lock))

    def GetResults(self, request, context: grpc.ServicerContext) \
            -> SensorServicer_pb2.GetResults_Responses:
        """
        Executes the unobservable command "Get Sensor Results"
            Get the current gas concentrations for CO2 and O2 in [vol.-%] and the pressure readings in [mbar].
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CO2 (Current CO2 Concentration): Get the current CO2 concentration
            request.O2 (Current O2 Concentration): Get the current O2 concentration
            request.Pressure (Current Air Pressure): Get the current air pressure reading
        """
    
        logging.debug(
            "GetResults called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetResults(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetHumidity(self, request, context: grpc.ServicerContext) \
            -> SensorServicer_pb2.GetHumidity_Responses:
        """
        Executes the unobservable command "Get Humidity Results"
            Get the current humidity [%], temperature [C] and absolute humidity [vol.-%].
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Humidity (Current Humidity): Get the current humidity in [%]
            request.Temperature (Current Temperature): Get the current temperature in [C]
            request.AbsoluteHumidity (Current Absolute Humidity): Get the current absolute humidity in [vol.-%]
        """
    
        logging.debug(
            "GetHumidity called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetHumidity(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)

    
