#!/usr/bin/env python3
"""
________________________________________________________________________

:PROJECT: SiLA2_python

*BlueVaryService*

:details: BlueVaryService:
    This is a BlueVary Off-Gas Analytics Service
           
:file:    BlueVaryService_server.py
:authors: Lukas Bromig

:date: (creation)          2020-04-16T10:18:59.332927
:date: (last modification) 2020-04-16T10:18:59.332927

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""
__version__ = "1.0"

import logging
import argparse

# Import the main SiLA library
from sila2lib.sila_server import SiLA2Server

# Import gRPC libraries of features
from DeviceServicer.gRPC import DeviceServicer_pb2
from DeviceServicer.gRPC import DeviceServicer_pb2_grpc
# import default arguments for this feature
from DeviceServicer.DeviceServicer_default_arguments import default_dict as DeviceServicer_default_dict
from SensorServicer.gRPC import SensorServicer_pb2
from SensorServicer.gRPC import SensorServicer_pb2_grpc
# import default arguments for this feature
from SensorServicer.SensorServicer_default_arguments import default_dict as SensorServicer_default_dict
from CalibrationServicer.gRPC import CalibrationServicer_pb2
from CalibrationServicer.gRPC import CalibrationServicer_pb2_grpc
# import default arguments for this feature
from CalibrationServicer.CalibrationServicer_default_arguments import default_dict as CalibrationServicer_default_dict

# Import the servicer modules for each feature
from DeviceServicer.DeviceServicer_servicer import DeviceServicer
from SensorServicer.SensorServicer_servicer import SensorServicer
from CalibrationServicer.CalibrationServicer_servicer import CalibrationServicer

# import custom implementation libraries
import os
import glob
import serial
import serial.tools.list_ports
import threading
from serial.tools import list_ports

class BlueVaryServiceServer(SiLA2Server):
    """
    This is a BlueVary Off-Gas Analytics Service
    """

    def __init__(self, cmd_args, simulation_mode: bool = False):
        """Class initialiser"""
        super().__init__(
            name=cmd_args.server_name, description=cmd_args.description,
            server_type=cmd_args.server_type,
            version=__version__,
            vendor_url="",
            ip="10.152.248.108", port=50004,
            key_file=cmd_args.encryption_key, cert_file=cmd_args.encryption_cert
        )

        logging.info(
            "Starting SiLA2 server with server name: {server_name}".format(
                server_name=cmd_args.server_name
            )
        )

        # Create a lock that prevent simultaneous requests to Device
        self.lock = threading.Lock()

        ports = []
        # Opening a serial port for data communication
        if os.name == "nt":
            # Available on Windows
            ports = list(serial.tools.list_ports.comports(include_links=True))
            print("Using serial port: %s" % ports[port_index][0])
            self.ser = serial.Serial(port=ports[port_index][0], baudrate=19200, parity=serial.PARITY_NONE,
                                     stopbits=serial.STOPBITS_ONE, bytesize=serial.EIGHTBITS, timeout=5)
        elif os.name == "posix":
            # Available on BeagleBone
            id4 = '88 CO2_30773 O2_30701 HUM_30587 :I,C5'
            command = '&i\r\n'
            # Available on BeagleBone
            portlist = list_ports.comports()
            for connec, k, i in portlist:
                    port = serial.Serial(port=connec, baudrate=19200, parity=serial.PARITY_NONE,
                    stopbits=serial.STOPBITS_ONE, bytesize=serial.EIGHTBITS, timeout=1)
                    read = None
                    port.write(str.encode(command))
                    read = str(bytes.decode(port.readline().rstrip()))
                    print(read)
                    port.close()
                    if read == id4:
                        print("Found Unit 4")
                        used_port = connec
            print("Using serial port: %s" %used_port)
            self.ser = serial.Serial(port=used_port, baudrate = 19200,parity=serial.PARITY_NONE, stopbits=serial.STOPBITS_ONE, bytesize=serial.EIGHTBITS,timeout=5)
        else:
            print("Error establishing serial connection! Check port configuration and auto-detection. Detection may vary depending on th ecurrent OS.")
        self.simulation_mode = simulation_mode
        # registering features
        #  Register DeviceServicer
        self.DeviceServicer_servicer = DeviceServicer(ser = self.ser, lock = self.lock, simulation_mode=self.simulation_mode)
        DeviceServicer_pb2_grpc.add_DeviceServicerServicer_to_server(
            self.DeviceServicer_servicer,
            self.grpc_server
        )
        self.add_feature(feature_id='DeviceServicer',
                         servicer=self.DeviceServicer_servicer,
                         data_path=os.path.join(os.path.dirname(os.path.abspath(__file__)), 'meta'))
        #  Register SensorServicer
        self.SensorServicer_servicer = SensorServicer(ser = self.ser, lock = self.lock, simulation_mode=self.simulation_mode)
        SensorServicer_pb2_grpc.add_SensorServicerServicer_to_server(
            self.SensorServicer_servicer,
            self.grpc_server
        )
        self.add_feature(feature_id='SensorServicer',
                         servicer=self.SensorServicer_servicer,
                         data_path=os.path.join(os.path.dirname(os.path.abspath(__file__)), 'meta'))
        #  Register CalibrationServicer
        self.CalibrationServicer_servicer = CalibrationServicer(ser = self.ser, lock = self.lock, simulation_mode=self.simulation_mode)
        CalibrationServicer_pb2_grpc.add_CalibrationServicerServicer_to_server(
            self.CalibrationServicer_servicer,
            self.grpc_server
        )
        self.add_feature(feature_id='CalibrationServicer',
                         servicer=self.CalibrationServicer_servicer,
                         data_path=os.path.join(os.path.dirname(os.path.abspath(__file__)), 'meta'))

        self.simulation_mode = simulation_mode

        # starting and running the gRPC/SiLA2 server
        self.run()


def parse_command_line():
    """
    Just looking for commandline arguments
    """
    parser = argparse.ArgumentParser(description="A SiLA2 service: BlueVaryService")

    # Simple arguments for the server identification
    parser.add_argument('-s', '--server-name', action='store',
                        default="BlueVaryService_4", help='start SiLA server with [server-name]')
    parser.add_argument('-t', '--server-type', action='store',
                        default="Unknown Type", help='start SiLA server with [server-type]')
    parser.add_argument('-d', '--description', action='store',
                        default="This is a BlueVary Off-Gas Analytics Service", help='SiLA server description')

    # Encryption
    parser.add_argument('-X', '--encryption', action='store', default=None,
                        help='The name of the private key and certificate file (without extension).')
    parser.add_argument('--encryption-key', action='store', default=None,
                        help='The name of the encryption key (*with* extension). Can be used if key and certificate '
                             'vary or non-standard file extensions are used.')
    parser.add_argument('--encryption-cert', action='store', default=None,
                        help='The name of the encryption certificate (*with* extension). Can be used if key and '
                             'certificate vary or non-standard file extensions are used.')

    parser.add_argument('-v', '--version', action='version', version='%(prog)s ' + __version__)

    parsed_args = parser.parse_args()

    # validate/update some settings
    #   encryption
    if parsed_args.encryption is not None:
        # only overwrite the separate keys if not given manually
        if parsed_args.encryption_key is None:
            parsed_args.encryption_key = parsed_args.encryption + '.key'
        if parsed_args.encryption_cert is None:
            parsed_args.encryption_cert = parsed_args.encryption + '.cert'

    return parsed_args
    
        
if __name__ == '__main__':
    # or use logging.ERROR for less output
    logging.basicConfig(format='%(levelname)-8s| %(module)s.%(funcName)s: %(message)s', level=logging.DEBUG)
    
    args = parse_command_line()

    # generate SiLA2Server
    sila_server = BlueVaryServiceServer(cmd_args=args, simulation_mode=False)
