"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Device Servicer*

:details: DeviceServicer:
    General device settings of the DASGIP reactor system can be retrieved and changed within this function.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 20.05.2019

:file:    DeviceServicer_real.py
:authors: Lukas Bromig

:date: (creation)          2020-04-16T10:19:13.456748
:date: (last modification) 2020-04-16T10:19:13.456748

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "0.0.1"

# import general packages
import logging
import time         # used for observables
import uuid         # used for observables
import grpc         # used for type hinting only

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import DeviceServicer_pb2 as DeviceServicer_pb2
# from .gRPC import DeviceServicer_pb2_grpc as DeviceServicer_pb2_grpc

# import default arguments
from .DeviceServicer_default_arguments import default_dict


# noinspection PyPep8Naming,PyUnusedLocal
class DeviceServicerReal:
    """
    Implementation of the *Device Servicer* in *Real* mode
        This is a DASGIP Service
    """

    def __init__(self, reactors):
        """Class initialiser"""
        self.status = "Test-status. Not implemented yet!"
        self.reactors = reactors
        logging.debug('Started server in mode: {mode}'.format(mode='Real'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def GetLog(self, request, context: grpc.ServicerContext) \
            -> silaFW_pb2.CommandConfirmation:
        """
        Executes the observable command "Get Log"
            Get the current status of the device from the state machine of the SiLA server.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A command confirmation object with the following information:
            commandId: A command id with which this observable command can be referenced in future calls
            lifetimeOfExecution: The (maximum) lifetime of this command call.
        """
    
        # initialise default values
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        lifetime_of_execution: silaFW_pb2.Duration = None
    
        # TODO:
        #   Execute the actual command
        #   Optional: Generate a lifetime_of_execution
    
        # respond with UUID and lifetime of execution
        command_uuid = silaFW_pb2.CommandExecutionUUID(value=str(uuid.uuid4()))
        if lifetime_of_execution is not None:
            return silaFW_pb2.CommandConfirmation(
                commandExecutionUUID=command_uuid,
                lifetimeOfExecution=lifetime_of_execution
            )
        else:
            return silaFW_pb2.CommandConfirmation(
                commandExecutionUUID=command_uuid
            )
    
    def GetLog_Info(self, request, context: grpc.ServicerContext) \
            -> silaFW_pb2.ExecutionInfo:
        """
        Returns execution information regarding the command call :meth:`~.GetLog`.
    
        :param request: A request object with the following properties
            commandId: The UUID of the command executed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: An ExecutionInfo response stream for the command with the following fields:
            commandStatus: Status of the command (enumeration)
            progressInfo: Information on the progress of the command (0 to 1)
            estimatedRemainingTime: Estimate of the remaining time required to run the command
            updatedLifetimeOfExecution: An update on the execution lifetime
        """
        # Get the UUID of the command
        command_uuid = request.value
    
        # Get the current state
        execution_info = self._get_command_state(command_uuid=command_uuid)
    
        # construct the initial return dictionary in case while is not executed
        return_values = {'commandStatus': execution_info.commandStatus}
        if execution_info.HasField('progressInfo'):
            return_values['progressInfo'] = execution_info.progressInfo
        if execution_info.HasField('estimatedRemainingTime'):
            return_values['estimatedRemainingTime'] = execution_info.estimatedRemainingTime
        if execution_info.HasField('updatedLifetimeOfExecution'):
            return_values['updatedLifetimeOfExecution'] = execution_info.updatedLifetimeOfExecution
    
        # we loop only as long as the command is running
        while execution_info.commandStatus == silaFW_pb2.ExecutionInfo.CommandStatus.waiting \
                or execution_info.commandStatus == silaFW_pb2.ExecutionInfo.CommandStatus.running:
            # TODO:
            #   Evaluate the command status --> command_status. Options:
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.running
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.finishedSuccessfully
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.finishedWithError
            #   Optional:
            #       * Determine the progress (progressInfo)
            #       * Determine the estimated remaining time
            #       * Update the Lifetime of execution
    
            # Update all values
            execution_info = self._get_command_state(command_uuid=command_uuid)
    
            # construct the return dictionary
            return_values = {'commandStatus': execution_info.commandStatus}
            if execution_info.HasField('progressInfo'):
                return_values['progressInfo'] = execution_info.progressInfo
            if execution_info.HasField('estimatedRemainingTime'):
                return_values['estimatedRemainingTime'] = execution_info.estimatedRemainingTime
            if execution_info.HasField('updatedLifetimeOfExecution'):
                return_values['updatedLifetimeOfExecution'] = execution_info.updatedLifetimeOfExecution
    
            yield silaFW_pb2.ExecutionInfo(**return_values)
    
            # we add a small delay to give the client a chance to keep up.
            time.sleep(0.5)
        else:
            # one last time yield the status
            yield silaFW_pb2.ExecutionInfo(**return_values)
    
    def GetLog_Result(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetLog_Responses:
        """
        Returns the final result of the command call :meth:`~.GetLog`.
    
        :param request: A request object with the following properties
            CommandExecutionUUID: The UUID of the command executed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentLogLevel (Current Log Level): The current log level of the latest logs , retrieved from the SiLA server log file.
            request.CurrentLogTimestamp (Current Log Timestamp): The current log timestamp of the latest logs , retrieved from the SiLA server log file.
            request.CurrentLogMessage (Current Log Level): The current log level of the latest logs , retrieved from the SiLA server log file.
        """
    
        # initialise the return value
        return_value: DeviceServicer_pb2.GetLog_Responses = None
    
        # Get the UUID of the command
        command_uuid = request.value
    
        # TODO:
        #   Add implementation of Real for command GetLog here and write the resulting response
        #   in return_value
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.GetLog_Responses(
                **default_dict['GetLog_Responses']
            )
    
        return return_value

    def GetRuntimeClock(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetRuntimeClock_Responses:
        """
        Executes the unobservable command "Get Runtime Clock"
            Get the current local time of the system.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentRuntime (Current Runtime): The current runtime of the DASGIP reactor system.
        """
    
        # initialise the return value
        return_value = None
    
        #Get command
        node = self.reactors[request.UnitID.value].unit.Device.service.RuntimeClock
        CurrentRuntime = node.get_value()
        return_value = DeviceServicer_pb2.GetRuntimeClock_Responses(CurrentRuntime=silaFW_pb2.String(
            value=CurrentRuntime))
    
        return return_value
    
    
    def GetStartedUTC(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetStartedUTC_Responses:
        """
        Executes the unobservable command "Get Started UTC"
            Get time stamp of system start in UTC
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentStartedUTC (Current Started UTC): The current timestamnp of the system start in UTC
        """
    
        # initialise the return value
        return_value = None
    
        #Get command
        node = self.reactors[request.UnitID.value].unit.Device.service.StartedUTC
        CurrentStartedUTC = node.get_value()
        return_value =DeviceServicer_pb2.GetStartedUTC_Responses(CurrentStartedUTC=silaFW_pb2.Timestamp(
            second=CurrentStartedUTC.second,
            minute=CurrentStartedUTC.minute,
            hour=CurrentStartedUTC.hour,
            day=CurrentStartedUTC.day,
            month=CurrentStartedUTC.month,
            year=CurrentStartedUTC.year,
           ))
        return return_value
    
    
    def GetStarted(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetStarted_Responses:
        """
        Executes the unobservable command "Get Started"
            Get time stamp of system start
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentStarted (Current Started): The current timestamnp of the system start
        """
    
        # initialise the return value
        return_value = None

        #Get command
        node = self.reactors[request.UnitID.value].unit.Device.service.Started
        CurrentStarted = node.get_value()
        return_value =DeviceServicer_pb2.GetStarted_Responses(CurrentStarted=silaFW_pb2.Timestamp(
            second=CurrentStarted.second,
            minute=CurrentStarted.minute,
            hour=CurrentStarted.hour,
            day=CurrentStarted.day,
            month=CurrentStarted.month,
            year=CurrentStarted.year,
           ))
        return return_value
    
    
    def GetStoppedUTC(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetStoppedUTC_Responses:
        """
        Executes the unobservable command "Get Stopped UTC"
            Get time stamp of system stop in UTC
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentStoppedUTC (Current Stopped UTC): The current timestamnp of the system stop in UTC
        """
    
        # initialise the return value
        return_value = None
    
        #Get command
        node = self.reactors[request.UnitID.value].unit.Device.service.StoppedUTC
        CurrentStoppedUTC = node.get_value()
        return_value =DeviceServicer_pb2.GetStoppedUTC_Responses(CurrentStoppedUTC=silaFW_pb2.Timestamp(
            second=CurrentStoppedUTC.second,
            minute=CurrentStoppedUTC.minute,
            hour=CurrentStoppedUTC.hour,
            day=CurrentStoppedUTC.day,
            month=CurrentStoppedUTC.month,
            year=CurrentStoppedUTC.year,
           ))
        return return_value
    
    
    def GetStopped(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetStopped_Responses:
        """
        Executes the unobservable command "Get Stopped"
            Get time stamp of system stop
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentStopped (Current Stopped): The current timestamnp of the system stop
        """
    
        # initialise the return value
        return_value = None
    
        #Get command
        node = self.reactors[request.UnitID.value].unit.Device.service.Stopped
        CurrentStopped = node.get_value()
        return_value =DeviceServicer_pb2.GetStopped_Responses(CurrentStopped=silaFW_pb2.Timestamp(
            second=CurrentStopped.second,
            minute=CurrentStopped.minute,
            hour=CurrentStopped.hour,
            day=CurrentStopped.day,
            month=CurrentStopped.month,
            year=CurrentStopped.year,
           ))
        return return_value
    
    
    def GetUserId(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetUserId_Responses:
        """
        Executes the unobservable command "Get UserId"
            Get User Id of the System
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentUserId (Current User ID): The current user ID of the system
        """
    
        # initialise the return value
        return_value = None
    
        #Get command
        node = self.reactors[request.UnitID.value].unit.Device.service.UserId
        CurrentUserId = node.get_value()
        return_value =DeviceServicer_pb2.GetUserId_Responses(CurrentUserId=silaFW_pb2.String(value=CurrentUserId))
    
        return return_value
    
    
    def GetBatchId(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetBatchId_Responses:
        """
        Executes the unobservable command "Get BatchId"
            Get Batch Id of the System
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentBatchId (Current Batch ID): The current batch ID of the system
        """
    
        # initialise the return value
        return_value = None
    
        #Get command
        node = self.reactors[request.UnitID.value].unit.Device.service.BatchId
        CurrentBatchId = node.get_value()
        return_value =DeviceServicer_pb2.GetBatchId_Responses(CurrentBatchId=silaFW_pb2.String(value=CurrentBatchId))
    
        return return_value
    
    
    def GetInoculatedUTC(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetInoculatedUTC_Responses:
        """
        Executes the unobservable command "Get InoculatedUTC"
            Get Inoculation time in UTC
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentInoculationUTC (Current InoculationUTC): The current inoculation time in UTC
        """
    
        # initialise the return value
        return_value = None
    
        #Get command
        node = self.reactors[request.UnitID.value].unit.Device.service.InoculatedUTC
        CurrentInoculationUTC = node.get_value()
        return_value =DeviceServicer_pb2.GetInoculatedUTC_Responses(CurrentInoculationUTC=silaFW_pb2.Timestamp(
            second=CurrentInoculationUTC.second,
            minute=CurrentInoculationUTC.minute,
            hour=CurrentInoculationUTC.hour,
            day=CurrentInoculationUTC.day,
            month=CurrentInoculationUTC.month,
            year=CurrentInoculationUTC.year,
           ))
        return return_value
    
    
    def GetInoculated(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetInoculated_Responses:
        """
        Executes the unobservable command "Get Inoculated"
            Get Inoculation time
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentInoculation (Current Inoculation): The current inoculation time
        """
    
        # initialise the return value
        return_value = None
    
        #Get command
        node = self.reactors[request.UnitID.value].unit.Device.service.Inoculated
        CurrentInoculation = node.get_value()
        return_value = DeviceServicer_pb2.GetInoculated_Responses(CurrentInoculation=silaFW_pb2.Timestamp(
            second=CurrentInoculation.second,
            minute=CurrentInoculation.minute,
            hour=CurrentInoculation.hour,
            day=CurrentInoculation.day,
            month=CurrentInoculation.month,
            year=CurrentInoculation.year,
           ))

        return return_value
    
    
    def GetAvailable(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetAvailable_Responses:
        """
        Executes the unobservable command "Get Available"
            Get System Availability
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAvailable (Current Available): The current system availability
        """
    
        # initialise the return value
        return_value = None

        # Todo: Not implemented in OPC-UA? Remove from FDL? Not available for DeviceService?
        CurrentAvailable = 0
        return_value = DeviceServicer_pb2.GetAvailable_Responses(
            CurrentAvailable=silaFW_pb2.Integer(value=CurrentAvailable))
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.GetAvailable_Responses(
                **default_dict['GetAvailable_Responses']
            )
    
        return return_value
    
    
    def GetName(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetName_Responses:
        """
        Executes the unobservable command "Get Name"
            Get System Name
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentName (Current Name): The current system name
        """
    
        # initialise the return value
        return_value = None
    
        #Get command
        node = self.reactors[request.UnitID.value].unit.Device.service.Name
        CurrentName = node.get_value()
        return_value =DeviceServicer_pb2.GetName_Responses(CurrentName=silaFW_pb2.String(value=CurrentName))
    
        return return_value
    
    
    def GetVersion(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetVersion_Responses:
        """
        Executes the unobservable command "Get Version"
            Get System Version
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVersion (Current Version): The current system version
        """
    
        # initialise the return value
        return_value = None
    
        #Get command
        node = self.reactors[request.UnitID.value].unit.Device.service.Version
        CurrentVersion = node.get_value()
        return_value =DeviceServicer_pb2.GetVersion_Responses(CurrentVersion=silaFW_pb2.String(value=CurrentVersion))
    
        return return_value
    

    def Subscribe_CurrentStatus(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.Subscribe_CurrentStatus_Responses:
        """
        Requests the observable property Current Status
            Get the current status of the device from the internal state machine of the SiLA server.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.CurrentStatus (Current Status): Get the current status of the device from the internal state machine of the SiLA server.
        """
    
        # initialise the return value
        return_value: DeviceServicer_pb2.Subscribe_CurrentStatus_Responses = None
    
        # we could use a timeout here if we wanted
        while True:
            par_dict = {
                'CurrentStatus': silaFW_pb2.String(value=self.status)
            }
            return_value = DeviceServicer_pb2.Subscribe_CurrentStatus_Responses(**par_dict)
            # TODO:
            #   Add implementation of Real for property CurrentStatus here and write the resulting
            #   response in return_value
    
            # create the default value
            if return_value is None:
                return_value = DeviceServicer_pb2.Subscribe_CurrentStatus_Responses(
                    **default_dict['Subscribe_CurrentStatus_Responses']
                )
    
    
            yield return_value
