"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Gassing Servicer*

:details: GassingServicer:
    Control a DASGIP gassing module. Enables read and write operations for various parameters.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 20.05.2019
           
:file:    GassingServicer_servicer.py
:authors: Lukas Bromig

:date: (creation)          2020-04-16T10:19:13.710280
:date: (last modification) 2020-04-16T10:19:13.710280

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "0.0.1"

# import general packages
import logging
import grpc

# meta packages
from typing import Union

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2
from sila2lib.error_handling.server_err import SiLAError

# import gRPC modules for this feature
from .gRPC import GassingServicer_pb2 as GassingServicer_pb2
from .gRPC import GassingServicer_pb2_grpc as GassingServicer_pb2_grpc

# import simulation and real implementation
from .GassingServicer_simulation import GassingServicerSimulation
from .GassingServicer_real import GassingServicerReal


class GassingServicer(GassingServicer_pb2_grpc.GassingServicerServicer):
    """
    This is a DASGIP Service
    """
    implementation: Union[GassingServicerSimulation, GassingServicerReal]
    simulation_mode: bool

    def __init__(self, reactors, simulation_mode: bool = True):
        """
        Class initialiser.

        :param simulation_mode: Sets whether at initialisation the simulation mode is active or the real mode.
        """
        self.reactors = reactors
        self.simulation_mode = simulation_mode
        if simulation_mode:
            self._inject_implementation(GassingServicerSimulation(self.reactors))
        else:
            self._inject_implementation(GassingServicerReal(self.reactors))

    def _inject_implementation(self,
                               implementation: Union[GassingServicerSimulation,
                                                     GassingServicerReal]
                               ) -> bool:
        """
        Dependency injection of the implementation used.
            Allows to set the class used for simulation/real mode.

        :param implementation: A valid implementation of the DASGIP_ServiceServicer.
        """

        self.implementation = implementation
        return True

    def switch_to_simulation_mode(self):
        """Method that will automatically be called by the server when the simulation mode is requested."""
        self.simulation_mode = True
        self._inject_implementation(GassingServicerSimulation(self.reactors))

    def switch_to_real_mode(self):
        """Method that will automatically be called by the server when the real mode is requested."""
        self.simulation_mode = False
        self._inject_implementation(GassingServicerReal(self.reactors))

    def GetPV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetPV_Responses:
        """
        Executes the unobservable command "Get PV"
            Get present value. Gassing total flow process value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPV (Current PV): Current present value. Gassing total flow process value.
        """
    
        logging.debug(
            "GetPV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetPV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetSP(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetSP_Responses:
        """
        Executes the unobservable command "Get SP"
            Get setpoint value. Gassing total flow setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSP (Current SP): Current setpoint value. Gassing total flow setpoint.
        """
    
        logging.debug(
            "GetSP called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetSP(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFPV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFPV_Responses:
        """
        Executes the unobservable command "Get F PV"
            Get present value. Gassing total flow process value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFPV (Current F PV): Current present value. Gassing total flow process value.
        """
    
        logging.debug(
            "GetFPV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFPV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetVPV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetVPV_Responses:
        """
        Executes the unobservable command "Get V PV"
            Get present value. Gassing total volume process value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVPV (Current V PV): Current present value. Gassing total volume process value.
        """
    
        logging.debug(
            "GetVPV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetVPV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFSP(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFSP_Responses:
        """
        Executes the unobservable command "Get Flow SP"
            Get gassing total flow setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFSP (Current Flow SP): Current gassing total flow setpoint.
        """
    
        logging.debug(
            "GetFSP called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFSP(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFSPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFSPM_Responses:
        """
        Executes the unobservable command "Set Flow SPM"
            Set gassing total flow setpoint manual.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.FSPM (Flow Setpoint Manual):
            The gassing total flow setpoint manual.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FSPMSet (Flow Setpoint Manual Set): The set manual gassing total flow setpoint.
        """
    
        logging.debug(
            "SetFSPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFSPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFSPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFSPM_Responses:
        """
        Executes the unobservable command "Get Flow SP Manual"
            Get gassing total flow manual setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFSPM (Current Flow SP Manual): Current gassing total flow manual setpoint.
        """
    
        logging.debug(
            "GetFSPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFSPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFSPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFSPE_Responses:
        """
        Executes the unobservable command "Set Flow SPE"
            Set gassing total flow setpoint external.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.FSPE (Flow Setpoint External):
            The gassing total flow setpoint external.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FSPESet (Flow Setpoint External Set): The set external gassing total flow setpoint.
        """
    
        logging.debug(
            "SetFSPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFSPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFSPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFSPE_Responses:
        """
        Executes the unobservable command "Get Flow SP External"
            Get gassing total flow external setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFSPE (Current Flow SP External): Current gassing total flow external setpoint.
        """
    
        logging.debug(
            "GetFSPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFSPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFSPA(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFSPA_Responses:
        """
        Executes the unobservable command "Get Flow SP Automatic"
            Get gassing total flow automatic setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFSPA (Current Flow SP Automatic): Current gassing total flow automatic setpoint.
        """
    
        logging.debug(
            "GetFSPA called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFSPA(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFSPR(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFSPR_Responses:
        """
        Executes the unobservable command "Get Flow SP Remote"
            Get gassing total flow remote setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFSPR (Current Flow SP Remote): Current gassing total flow remote setpoint.
        """
    
        logging.debug(
            "GetFSPR called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFSPR(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFSPL(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFSPL_Responses:
        """
        Executes the unobservable command "Get Flow SP Local"
            Get gassing total flow local setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFSPL (Current Flow SP Local): Current gassing total flow local setpoint.
        """
    
        logging.debug(
            "GetFSPL called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFSPL(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFMode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFMode_Responses:
        """
        Executes the unobservable command "Set Flow Mode"
            Set gas total flow operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.FMode (Flow Mode):
            Set gas total flow operation mode. Controller mode (Manual, Automatic).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FModeSet (Flow Mode Set): The set gas total flow operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "SetFMode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFMode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFMode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFMode_Responses:
        """
        Executes the unobservable command "Get Flow Mode"
            Get gas total flow operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFMode (Current Flow Mode): Current gas total flow operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "GetFMode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFMode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFSetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFSetpointSelect_Responses:
        """
        Executes the unobservable command "Set Flow SetpointSelect"
            Set gas total flow setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.FSetpointSelect (Flow SetpointSelect):
            Set gas total flow setpoint selection (Local, Manual, Internal, Script, External).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FSetpointSelectSet (Flow SetpointSelect Set): The set gas total flow setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "SetFSetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFSetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFSetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFSetpointSelect_Responses:
        """
        Executes the unobservable command "Get Flow SetpointSelect"
            Get gas total flow setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFSetpointSelect (Current Flow SetpointSelect): Current gas total flow setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "GetFSetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFSetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXCO2PV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXCO2PV_Responses:
        """
        Executes the unobservable command "Get XCO2 PV"
            Get present value. Gassing XCO2 concentration process value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXCO2PV (Current XCO2 PV): Current present value. Gassing XCO2 concentration process value.
        """
    
        logging.debug(
            "GetXCO2PV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXCO2PV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXCO2SP(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXCO2SP_Responses:
        """
        Executes the unobservable command "Get XCO2 SP"
            Get gassing XCO2 concentration setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXCO2SP (Current XCO2 SP): Current gassing XCO2 concentration setpoint.
        """
    
        logging.debug(
            "GetXCO2SP called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXCO2SP(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetXCO2SPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetXCO2SPM_Responses:
        """
        Executes the unobservable command "Set XCO2 SPM"
            Set gassing XCO2 concentration setpoint manual.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.XCO2SPM (XCO2 Setpoint Manual):
            The gassing XCO2 concentration setpoint manual.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.XCO2SPMSet (XCO2 Setpoint Manual Set): The set manual gassing XCO2 concentration setpoint.
        """
    
        logging.debug(
            "SetXCO2SPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetXCO2SPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXCO2SPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXCO2SPM_Responses:
        """
        Executes the unobservable command "Get XCO2 SP Manual"
            Get gassing XCO2 concentration manual setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXCO2SPM (Current XCO2 SP Manual): Current gassing XCO2 concentration manual setpoint.
        """
    
        logging.debug(
            "GetXCO2SPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXCO2SPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetXCO2SPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetXCO2SPE_Responses:
        """
        Executes the unobservable command "Set XCO2 SPE"
            Set gassing XCO2 concentration setpoint external.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.XCO2SPE (XCO2 Setpoint External):
            The gassing XCO2 concentration setpoint external.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.XCO2SPESet (XCO2 Setpoint External Set): The set external gassing XCO2 concentration setpoint.
        """
    
        logging.debug(
            "SetXCO2SPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetXCO2SPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXCO2SPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXCO2SPE_Responses:
        """
        Executes the unobservable command "Get XCO2 SP External"
            Get gassing XCO2 concentration external setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXCO2SPE (Current XCO2 SP External): Current gassing XCO2 concentration external setpoint.
        """
    
        logging.debug(
            "GetXCO2SPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXCO2SPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXCO2SPA(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXCO2SPA_Responses:
        """
        Executes the unobservable command "Get XCO2 SP Automatic"
            Get gassing XCO2 concentration automatic setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXCO2SPA (Current XCO2 SP Automatic): Current gassing XCO2 concentration automatic setpoint.
        """
    
        logging.debug(
            "GetXCO2SPA called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXCO2SPA(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXCO2SPR(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXCO2SPR_Responses:
        """
        Executes the unobservable command "Get XCO2 SP Remote"
            Get gassing XCO2 concentration remote setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXCO2SPR (Current XCO2 SP Remote): Current gassing XCO2 concentration remote setpoint.
        """
    
        logging.debug(
            "GetXCO2SPR called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXCO2SPR(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXCO2SPL(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXCO2SPL_Responses:
        """
        Executes the unobservable command "Get XCO2 SP Local"
            Get gassing XCO2 concentration local setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXCO2SPL (Current XCO2 SP Local): Current gassing XCO2 concentration local setpoint.
        """
    
        logging.debug(
            "GetXCO2SPL called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXCO2SPL(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetXCO2Mode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetXCO2Mode_Responses:
        """
        Executes the unobservable command "Set XCO2 Mode"
            Set gas XCO2 concentration operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.XCO2Mode (XCO2 Mode):
            Set gas XCO2 concentration operation mode. Controller mode (Manual, Automatic).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.XCO2ModeSet (XCO2 Mode Set): The set gas XCO2 concentration operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "SetXCO2Mode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetXCO2Mode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXCO2Mode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXCO2Mode_Responses:
        """
        Executes the unobservable command "Get XCO2 Mode"
            Get gas XCO2 concentration operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXCO2Mode (Current XCO2 Mode): Current gas XCO2 concentration operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "GetXCO2Mode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXCO2Mode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetXCO2SetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetXCO2SetpointSelect_Responses:
        """
        Executes the unobservable command "Set XCO2 SetpointSelect"
            Set gas XCO2 concentration setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.XCO2SetpointSelect (XCO2 SetpointSelect):
            Set gas XCO2 concentration setpoint selection (Local, Manual, Internal, Script, External).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.XCO2SetpointSelectSet (XCO2 SetpointSelect Set): The set gas XCO2 concentration setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "SetXCO2SetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetXCO2SetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXCO2SetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXCO2SetpointSelect_Responses:
        """
        Executes the unobservable command "Get XCO2 SetpointSelect"
            Get gas XCO2 concentration setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXCO2SetpointSelect (Current XCO2 SetpointSelect): Current gas XCO2 concentration setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "GetXCO2SetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXCO2SetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXO2PV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXO2PV_Responses:
        """
        Executes the unobservable command "Get XO2 PV"
            Get present value. Gassing XO2 concentration process value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXO2PV (Current XO2 PV): Current present value. Gassing XO2 concentration process value.
        """
    
        logging.debug(
            "GetXO2PV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXO2PV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXO2SP(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXO2SP_Responses:
        """
        Executes the unobservable command "Get XO2 SP"
            Get gassing XO2 concentration setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXO2SP (Current XO2 SP): Current gassing XO2 concentration setpoint.
        """
    
        logging.debug(
            "GetXO2SP called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXO2SP(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetXO2SPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetXO2SPM_Responses:
        """
        Executes the unobservable command "Set XO2 SPM"
            Set gassing XO2 concentration setpoint manual.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.XO2SPM (XO2 Setpoint Manual):
            The gassing XO2 concentration setpoint manual.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.XO2SPMSet (XO2 Setpoint Manual Set): The set manual gassing XO2 concentration setpoint.
        """
    
        logging.debug(
            "SetXO2SPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetXO2SPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXO2SPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXO2SPM_Responses:
        """
        Executes the unobservable command "Get XO2 SP Manual"
            Get gassing XO2 concentration manual setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXO2SPM (Current XO2 SP Manual): Current gassing XO2 concentration manual setpoint.
        """
    
        logging.debug(
            "GetXO2SPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXO2SPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetXO2SPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetXO2SPE_Responses:
        """
        Executes the unobservable command "Set XO2 SPE"
            Set gassing XO2 concentration setpoint external.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.XO2SPE (XO2 Setpoint External):
            The gassing XO2 concentration setpoint external.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.XO2SPESet (XO2 Setpoint External Set): The set external gassing XO2 concentration setpoint.
        """
    
        logging.debug(
            "SetXO2SPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetXO2SPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXO2SPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXO2SPE_Responses:
        """
        Executes the unobservable command "Get XO2 SP External"
            Get gassing XO2 concentration external setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXO2SPE (Current XO2 SP External): Current gassing XO2 concentration external setpoint.
        """
    
        logging.debug(
            "GetXO2SPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXO2SPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXO2SPA(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXO2SPA_Responses:
        """
        Executes the unobservable command "Get XO2 SP Automatic"
            Get gassing XO2 concentration automatic setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXO2SPA (Current XO2 SP Automatic): Current gassing XO2 concentration automatic setpoint.
        """
    
        logging.debug(
            "GetXO2SPA called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXO2SPA(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXO2SPR(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXO2SPR_Responses:
        """
        Executes the unobservable command "Get XO2 SP Remote"
            Get gassing XO2 concentration remote setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXO2SPR (Current XO2 SP Remote): Current gassing XO2 concentration remote setpoint.
        """
    
        logging.debug(
            "GetXO2SPR called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXO2SPR(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXO2SPL(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXO2SPL_Responses:
        """
        Executes the unobservable command "Get XO2 SP Local"
            Get gassing XO2 concentration local setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXO2SPL (Current XO2 SP Local): Current gassing XO2 concentration local setpoint.
        """
    
        logging.debug(
            "GetXO2SPL called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXO2SPL(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetXO2Mode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetXO2Mode_Responses:
        """
        Executes the unobservable command "Set XO2 Mode"
            Set gas XO2 concentration operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.XO2Mode (XO2 Mode):
            Set gas XO2 concentration operation mode. Controller mode (Manual, Automatic).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.XO2ModeSet (XO2 Mode Set): The set gas XO2 concentration operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "SetXO2Mode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetXO2Mode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXO2Mode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXO2Mode_Responses:
        """
        Executes the unobservable command "Get XO2 Mode"
            Get gas XO2 concentration operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXO2Mode (Current XO2 Mode): Current gas XO2 concentration operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "GetXO2Mode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXO2Mode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetXO2SetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetXO2SetpointSelect_Responses:
        """
        Executes the unobservable command "Set XO2 SetpointSelect"
            Set gas XO2 concentration setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.XO2SetpointSelect (XO2 SetpointSelect):
            Set gas XO2 concentration setpoint selection (Local, Manual, Internal, Script, External).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.XO2SetpointSelectSet (XO2 SetpointSelect Set): The set gas XO2 concentration setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "SetXO2SetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetXO2SetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetXO2SetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetXO2SetpointSelect_Responses:
        """
        Executes the unobservable command "Get XO2 SetpointSelect"
            Get gas XO2 concentration setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentXO2SetpointSelect (Current XO2 SetpointSelect): Current gas XO2 concentration setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "GetXO2SetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetXO2SetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFAirPV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFAirPV_Responses:
        """
        Executes the unobservable command "Get FAir PV"
            Get present value. Gassing air flow process value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFAirPV (Current FAir PV): Current present value. Gassing air flow process value.
        """
    
        logging.debug(
            "GetFAirPV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFAirPV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetVAirPV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetVAirPV_Responses:
        """
        Executes the unobservable command "Get V Air PV"
            Get present value. Gassing air flow volume value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVAirPV (Current V Air PV): Current present value. Gassing air flow volume value
        """
    
        logging.debug(
            "GetVAirPV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetVAirPV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFAirSP(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFAirSP_Responses:
        """
        Executes the unobservable command "Get FAir SP"
            Get gassing air flow setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFAirSP (Current FAir SP): Current gassing air flow setpoint.
        """
    
        logging.debug(
            "GetFAirSP called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFAirSP(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFAirSPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFAirSPM_Responses:
        """
        Executes the unobservable command "Set FAir SPM"
            Set gassing air flow setpoint manual.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.FAirSPM (FAir Setpoint Manual):
            The gassing air flow setpoint manual.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FAirSPMSet (FAir Setpoint Manual Set): The set manual gassing air flow setpoint.
        """
    
        logging.debug(
            "SetFAirSPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFAirSPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFAirSPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFAirSPM_Responses:
        """
        Executes the unobservable command "Get FAir SP Manual"
            Get gassing air flow manual setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFAirSPM (Current FAir SP Manual): Current gassing air flow manual setpoint.
        """
    
        logging.debug(
            "GetFAirSPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFAirSPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFAirSPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFAirSPE_Responses:
        """
        Executes the unobservable command "Set FAir SPE"
            Set gassing air flow setpoint external.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.FAirSPE (FAir Setpoint External):
            The gassing air flow setpoint external.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FAirSPESet (FAir Setpoint External Set): The set external gassing air flow setpoint.
        """
    
        logging.debug(
            "SetFAirSPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFAirSPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFAirSPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFAirSPE_Responses:
        """
        Executes the unobservable command "Get FAir SP External"
            Get gassing air flow external setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFAirSPE (Current FAir SP External): Current gassing air flow external setpoint.
        """
    
        logging.debug(
            "GetFAirSPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFAirSPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFAirSPA(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFAirSPA_Responses:
        """
        Executes the unobservable command "Get FAir SP Automatic"
            Get gassing air flow automatic setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFAirSPA (Current FAir SP Automatic): Current gassing air flow automatic setpoint.
        """
    
        logging.debug(
            "GetFAirSPA called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFAirSPA(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFAirSPR(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFAirSPR_Responses:
        """
        Executes the unobservable command "Get FAir SP Remote"
            Get gassing air flow remote setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFAirSPR (Current FAir SP Remote): Current gassing air flow remote setpoint.
        """
    
        logging.debug(
            "GetFAirSPR called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFAirSPR(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFAirMode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFAirMode_Responses:
        """
        Executes the unobservable command "Set FAir Mode"
            Set gas air flow operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.FAirMode (FAir Mode):
            Set gas air flow operation mode. Controller mode (Manual, Automatic).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FAirModeSet (FAir Mode Set): The set gas air flow operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "SetFAirMode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFAirMode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFAirMode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFAirMode_Responses:
        """
        Executes the unobservable command "Get FAir Mode"
            Get gas air flow operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFAirMode (Current FAir Mode): Current gas air flow operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "GetFAirMode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFAirMode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFAirSetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFAirSetpointSelect_Responses:
        """
        Executes the unobservable command "Set FAir SetpointSelect"
            Set gas air flow setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.FAirSetpointSelect (FAir SetpointSelect):
            Set gas air flow setpoint selection (Local, Manual, Internal, Script, External).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FAirSetpointSelectSet (FAir SetpointSelect Set): The set gas air flow setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "SetFAirSetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFAirSetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFAirSetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFAirSetpointSelect_Responses:
        """
        Executes the unobservable command "Get FAir SetpointSelect"
            Get gas air flow setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFAirSetpointSelect (Current FAir SetpointSelect): Current gas air flow setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "GetFAirSetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFAirSetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFO2PV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFO2PV_Responses:
        """
        Executes the unobservable command "Get FO2 PV"
            Get present value. Gassing O2 flow process value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFO2PV (Current FO2 PV): Current present value. Gassing O2 flow process value.
        """
    
        logging.debug(
            "GetFO2PV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFO2PV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetVO2PV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetVO2PV_Responses:
        """
        Executes the unobservable command "Get V O2 PV"
            Get present value. Gassing O2 volume process value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVO2PV (Current V O2 PV): Current present value. Gassing O2 volume process value.
        """
    
        logging.debug(
            "GetVO2PV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetVO2PV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFO2SP(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFO2SP_Responses:
        """
        Executes the unobservable command "Get FO2 SP"
            Get gassing O2 flow setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFO2SP (Current FO2 SP): Current gassing O2 flow setpoint.
        """
    
        logging.debug(
            "GetFO2SP called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFO2SP(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFO2SPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFO2SPM_Responses:
        """
        Executes the unobservable command "Set FO2 SPM"
            Set gassing O2 flow setpoint manual.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.FO2SPM (FO2 Setpoint Manual):
            The gassing O2 flow setpoint manual.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FO2SPMSet (FO2 Setpoint Manual Set): The set manual gassing O2 flow setpoint.
        """
    
        logging.debug(
            "SetFO2SPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFO2SPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFO2SPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFO2SPM_Responses:
        """
        Executes the unobservable command "Get FO2 SP Manual"
            Get gassing O2 flow manual setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFO2SPM (Current FO2 SP Manual): Current gassing O2 flow manual setpoint.
        """
    
        logging.debug(
            "GetFO2SPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFO2SPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFO2SPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFO2SPE_Responses:
        """
        Executes the unobservable command "Set FO2 SPE"
            Set gassing O2 flow setpoint external.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.FO2SPE (FO2 Setpoint External):
            The gassing O2 flow setpoint external.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FO2SPESet (FO2 Setpoint External Set): The set external gassing O2 flow setpoint.
        """
    
        logging.debug(
            "SetFO2SPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFO2SPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFO2SPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFO2SPE_Responses:
        """
        Executes the unobservable command "Get FO2 SP External"
            Get gassing O2 flow external setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFO2SPE (Current FO2 SP External): Current gassing O2 flow external setpoint.
        """
    
        logging.debug(
            "GetFO2SPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFO2SPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFO2SPA(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFO2SPA_Responses:
        """
        Executes the unobservable command "Get FO2 SP Automatic"
            Get gassing O2 flow automatic setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFO2SPA (Current FO2 SP Automatic): Current gassing O2 flow automatic setpoint.
        """
    
        logging.debug(
            "GetFO2SPA called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFO2SPA(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFO2SPR(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFO2SPR_Responses:
        """
        Executes the unobservable command "Get FO2 SP Remote"
            Get gassing O2 flow remote setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFO2SPR (Current FO2 SP Remote): Current gassing O2 flow remote setpoint.
        """
    
        logging.debug(
            "GetFO2SPR called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFO2SPR(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFO2Mode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFO2Mode_Responses:
        """
        Executes the unobservable command "Set FO2 Mode"
            Set gas O2 flow operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.FO2Mode (FO2 Mode):
            Set gas O2 flow operation mode. Controller mode (Manual, Automatic).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FO2ModeSet (FO2 Mode Set): The set gas O2 flow operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "SetFO2Mode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFO2Mode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFO2Mode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFO2Mode_Responses:
        """
        Executes the unobservable command "Get FO2 Mode"
            Get gas O2 flow operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFO2Mode (Current FO2 Mode): Current gas O2 flow operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "GetFO2Mode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFO2Mode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFO2SetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFO2SetpointSelect_Responses:
        """
        Executes the unobservable command "Set FO2 SetpointSelect"
            Set gas O2 flow setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.FO2SetpointSelect (FO2 SetpointSelect):
            Set gas O2 flow setpoint selection (Local, Manual, Internal, Script, External).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FO2SetpointSelectSet (FO2 SetpointSelect Set): The set gas O2 flow setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "SetFO2SetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFO2SetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFO2SetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFO2SetpointSelect_Responses:
        """
        Executes the unobservable command "Get FO2 SetpointSelect"
            Get gas O2 flow setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFO2SetpointSelect (Current FO2 SetpointSelect): Current gas O2 flow setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "GetFO2SetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFO2SetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFCO2PV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFCO2PV_Responses:
        """
        Executes the unobservable command "Get FCO2 PV"
            Get present value. Gassing CO2  flow process value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFCO2PV (Current FCO2 PV): Current present value. Gassing CO2  flow process value.
        """
    
        logging.debug(
            "GetFCO2PV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFCO2PV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetVCO2PV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetVCO2PV_Responses:
        """
        Executes the unobservable command "Get V CO2 PV"
            Get present value. Gassing CO2 volume process value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVCO2PV (Current V CO2 PV): Current present value. Gassing CO2 volume process value.
        """
    
        logging.debug(
            "GetVCO2PV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetVCO2PV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFCO2SP(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFCO2SP_Responses:
        """
        Executes the unobservable command "Get FCO2 SP"
            Get gassing CO2  flow setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFCO2SP (Current FCO2 SP): Current gassing CO2  flow setpoint.
        """
    
        logging.debug(
            "GetFCO2SP called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFCO2SP(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFCO2SPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFCO2SPM_Responses:
        """
        Executes the unobservable command "Set FCO2 SPM"
            Set gassing CO2  flow setpoint manual.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.FCO2SPM (FCO2 Setpoint Manual):
            The gassing CO2  flow setpoint manual.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FCO2SPMSet (FCO2 Setpoint Manual Set): The set manual gassing CO2  flow setpoint.
        """
    
        logging.debug(
            "SetFCO2SPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFCO2SPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFCO2SPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFCO2SPM_Responses:
        """
        Executes the unobservable command "Get FCO2 SP Manual"
            Get gassing CO2  flow manual setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFCO2SPM (Current FCO2 SP Manual): Current gassing CO2  flow manual setpoint.
        """
    
        logging.debug(
            "GetFCO2SPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFCO2SPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFCO2SPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFCO2SPE_Responses:
        """
        Executes the unobservable command "Set FCO2 SPE"
            Set gassing CO2  flow setpoint external.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.FCO2SPE (FCO2 Setpoint External):
            The gassing CO2  flow setpoint external.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FCO2SPESet (FCO2 Setpoint External Set): The set external gassing CO2  flow setpoint.
        """
    
        logging.debug(
            "SetFCO2SPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFCO2SPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFCO2SPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFCO2SPE_Responses:
        """
        Executes the unobservable command "Get FCO2 SP External"
            Get gassing CO2  flow external setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFCO2SPE (Current FCO2 SP External): Current gassing CO2  flow external setpoint.
        """
    
        logging.debug(
            "GetFCO2SPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFCO2SPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFCO2SPA(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFCO2SPA_Responses:
        """
        Executes the unobservable command "Get FCO2 SP Automatic"
            Get gassing CO2  flow automatic setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFCO2SPA (Current FCO2 SP Automatic): Current gassing CO2  flow automatic setpoint.
        """
    
        logging.debug(
            "GetFCO2SPA called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFCO2SPA(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFCO2SPR(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFCO2SPR_Responses:
        """
        Executes the unobservable command "Get FCO2 SP Remote"
            Get gassing CO2  flow remote setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFCO2SPR (Current FCO2 SP Remote): Current gassing CO2  flow remote setpoint.
        """
    
        logging.debug(
            "GetFCO2SPR called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFCO2SPR(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFCO2Mode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFCO2Mode_Responses:
        """
        Executes the unobservable command "Set FCO2 Mode"
            Set gas CO2  flow operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.FCO2Mode (FCO2 Mode):
            Set gas CO2  flow operation mode. Controller mode (Manual, Automatic).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FCO2ModeSet (FCO2 Mode Set): The set gas CO2  flow operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "SetFCO2Mode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFCO2Mode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFCO2Mode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFCO2Mode_Responses:
        """
        Executes the unobservable command "Get FCO2 Mode"
            Get gas CO2  flow operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFCO2Mode (Current FCO2 Mode): Current gas CO2  flow operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "GetFCO2Mode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFCO2Mode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFCO2SetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFCO2SetpointSelect_Responses:
        """
        Executes the unobservable command "Set FCO2 SetpointSelect"
            Set gas CO2  flow setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.FCO2SetpointSelect (FCO2 SetpointSelect):
            Set gas CO2  flow setpoint selection (Local, Manual, Internal, Script, External).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FCO2SetpointSelectSet (FCO2 SetpointSelect Set): The set gas CO2  flow setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "SetFCO2SetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFCO2SetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFCO2SetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFCO2SetpointSelect_Responses:
        """
        Executes the unobservable command "Get FCO2 SetpointSelect"
            Get gas CO2  flow setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFCO2SetpointSelect (Current FCO2 SetpointSelect): Current gas CO2  flow setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "GetFCO2SetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFCO2SetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFN2PV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFN2PV_Responses:
        """
        Executes the unobservable command "Get FN2 PV"
            Get present value. Gassing N2 flow process value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFN2PV (Current FN2 PV): Current present value. Gassing N2 flow process value.
        """
    
        logging.debug(
            "GetFN2PV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFN2PV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetVN2PV(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetVN2PV_Responses:
        """
        Executes the unobservable command "Get V N2 PV"
            Get present value. Gassing N2 volume process value.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVN2PV (Current V N2 PV): Current present value. Gassing N2 volume process value.
        """
    
        logging.debug(
            "GetVN2PV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetVN2PV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFN2SP(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFN2SP_Responses:
        """
        Executes the unobservable command "Get FN2 SP"
            Get gassing N2 flow setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFN2SP (Current FN2 SP): Current gassing N2 flow setpoint.
        """
    
        logging.debug(
            "GetFN2SP called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFN2SP(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFN2SPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFN2SPM_Responses:
        """
        Executes the unobservable command "Set FN2 SPM"
            Set gassing N2 flow setpoint manual.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.FN2SPM (FN2 Setpoint Manual):
            The gassing N2 flow setpoint manual.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FN2SPMSet (FN2 Setpoint Manual Set): The set manual gassing N2 flow setpoint.
        """
    
        logging.debug(
            "SetFN2SPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFN2SPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFN2SPM(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFN2SPM_Responses:
        """
        Executes the unobservable command "Get FN2 SP Manual"
            Get gassing N2 flow manual setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFN2SPM (Current FN2 SP Manual): Current gassing N2 flow manual setpoint.
        """
    
        logging.debug(
            "GetFN2SPM called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFN2SPM(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFN2SPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFN2SPE_Responses:
        """
        Executes the unobservable command "Set FN2 SPE"
            Set gassing N2 flow setpoint external.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor.
            request.FN2SPE (FN2 Setpoint External):
            The gassing N2 flow setpoint external.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FN2SPESet (FN2 Setpoint External Set): The set external gassing N2 flow setpoint.
        """
    
        logging.debug(
            "SetFN2SPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFN2SPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFN2SPE(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFN2SPE_Responses:
        """
        Executes the unobservable command "Get FN2 SP External"
            Get gassing N2 flow external setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFN2SPE (Current FN2 SP External): Current gassing N2 flow external setpoint.
        """
    
        logging.debug(
            "GetFN2SPE called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFN2SPE(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFN2SPA(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFN2SPA_Responses:
        """
        Executes the unobservable command "Get FN2 SP Automatic"
            Get gassing N2 flow automatic setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFN2SPA (Current FN2 SP Automatic): Current gassing N2 flow automatic setpoint.
        """
    
        logging.debug(
            "GetFN2SPA called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFN2SPA(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFN2SPR(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFN2SPR_Responses:
        """
        Executes the unobservable command "Get FN2 SP Remote"
            Get gassing N2 flow remote setpoint.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFN2SPR (Current FN2 SP Remote): Current gassing N2 flow remote setpoint.
        """
    
        logging.debug(
            "GetFN2SPR called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFN2SPR(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFN2Mode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFN2Mode_Responses:
        """
        Executes the unobservable command "Set FN2 Mode"
            Set gas N2 flow operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.FN2Mode (FN2 Mode):
            Set gas N2 flow operation mode. Controller mode (Manual, Automatic).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FN2ModeSet (FN2 Mode Set): The set gas N2 flow operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "SetFN2Mode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFN2Mode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFN2Mode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFN2Mode_Responses:
        """
        Executes the unobservable command "Get FN2 Mode"
            Get gas N2 flow operation mode. Controller mode (Manual, Automatic).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFN2Mode (Current FN2 Mode): Current gas N2 flow operation mode. Controller mode (Manual, Automatic).
        """
    
        logging.debug(
            "GetFN2Mode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFN2Mode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFN2SetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetFN2SetpointSelect_Responses:
        """
        Executes the unobservable command "Set FN2 SetpointSelect"
            Set gas N2 flow setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.FN2SetpointSelect (FN2 SetpointSelect):
            Set gas N2 flow setpoint selection (Local, Manual, Internal, Script, External).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.FN2SetpointSelectSet (FN2 SetpointSelect Set): The set gas N2 flow setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "SetFN2SetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFN2SetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFN2SetpointSelect(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetFN2SetpointSelect_Responses:
        """
        Executes the unobservable command "Get FN2 SetpointSelect"
            Get gas N2 flow setpoint selection (Local, Manual, Internal, Script, External).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFN2SetpointSelect (Current FN2 SetpointSelect): Current gas N2 flow setpoint selection (Local, Manual, Internal, Script, External).
        """
    
        logging.debug(
            "GetFN2SetpointSelect called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFN2SetpointSelect(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetAccess(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetAccess_Responses:
        """
        Executes the unobservable command "Get Access Mode"
            Get access mode value. Controller access (Local, Remote).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAccess (Current Access Mode): Current access mode value. Controller access (Local, Remote)
        """
    
        logging.debug(
            "GetAccess called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetAccess(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetCmd(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetCmd_Responses:
        """
        Executes the unobservable command "Set Controller Command"
            Set the controller command. Controller command (Nothing, Stop, Start).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.Cmd (Controller Command):
            The controller command of the gassing module. Controller command (Nothing, Stop, Start).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CmdSet (Cmd Set): The set controller command.
        """
    
        logging.debug(
            "SetCmd called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetCmd(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetCmd(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetCmd_Responses:
        """
        Executes the unobservable command "Get Controller Command"
            Get the controller command. Controller command (Nothing, Stop, Start).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentCmd (Current Controller Command): Current controller command value of the gassing module. Controller command (Nothing, Stop, Start)
        """
    
        logging.debug(
            "GetCmd called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetCmd(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetGassingMode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.SetGassingMode_Responses:
        """
        Executes the unobservable command "Set Gassing Mode"
            Gassing mode (Unknown, Mix, Flow).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.GassingMode (Gassing Mode):
            Gassing mode (Unknown, Mix, Flow).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.GassingModeSet (Gassing Mode Set): The set gassing mode.
        """
    
        logging.debug(
            "SetGassingMode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetGassingMode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetGassingMode(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetGassingMode_Responses:
        """
        Executes the unobservable command "Get Gassing Mode"
            Get the gassing mode. Gassing mode (Unknown, Mix, Flow).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentGassingMode (Current Gassing Mode): Current gassing mode value of the gassing module. Gassing mode (Unknown, Mix, Flow).
        """
    
        logging.debug(
            "GetGassingMode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetGassingMode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetState(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetState_Responses:
        """
        Executes the unobservable command "Get State"
            Get controller state. Controller state (Off, On, Error).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentState (Current Controller State): Current controller state value of the gassing module. Controller state (Off, On, Error).
        """
    
        logging.debug(
            "GetState called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetState(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetType(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetType_Responses:
        """
        Executes the unobservable command "Get Function Type"
            Get function type.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentType (Current Function Type): Current function type value of the gassing module.
        """
    
        logging.debug(
            "GetType called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetType(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetAvailable(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetAvailable_Responses:
        """
        Executes the unobservable command "Get Function Availability"
            Get function availability.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAvailable (Current Function Availability): Current function availability value of the gassing module.
        """
    
        logging.debug(
            "GetAvailable called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetAvailable(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetName(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetName_Responses:
        """
        Executes the unobservable command "Get Function Name"
            Get function name.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentName (Current Function Name): Current function name of the gassing module.
        """
    
        logging.debug(
            "GetName called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetName(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetVersion(self, request, context: grpc.ServicerContext) \
            -> GassingServicer_pb2.GetVersion_Responses:
        """
        Executes the unobservable command "Get Function Version"
            Get function model version number.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVersion (Current Function Version): Current function model version number of the gassing module.
        """
    
        logging.debug(
            "GetVersion called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetVersion(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)

    
