"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Reactor Servicer*

:details: ReactorServicer:
    Control a DASGIP reactor module. Enables read and write operations for various parameters, including reactor sensor,
    controller, and alarm.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 20.05.2019
           
:file:    ReactorServicer_servicer.py
:authors: Lukas Bromig

:date: (creation)          2020-04-16T10:19:14.975957
:date: (last modification) 2020-04-16T10:19:14.975957

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "0.0.1"

# import general packages
import logging
import grpc

# meta packages
from typing import Union

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2
from sila2lib.error_handling.server_err import SiLAError

# import gRPC modules for this feature
from .gRPC import ReactorServicer_pb2 as ReactorServicer_pb2
from .gRPC import ReactorServicer_pb2_grpc as ReactorServicer_pb2_grpc

# import simulation and real implementation
from .ReactorServicer_simulation import ReactorServicerSimulation
from .ReactorServicer_real import ReactorServicerReal


class ReactorServicer(ReactorServicer_pb2_grpc.ReactorServicerServicer):
    """
    This is a DASGIP Service
    """
    implementation: Union[ReactorServicerSimulation, ReactorServicerReal]
    simulation_mode: bool

    def __init__(self, reactors, simulation_mode: bool = True):
        """
        Class initialiser.

        :param simulation_mode: Sets whether at initialisation the simulation mode is active or the real mode.
        """
        self.reactors = reactors
        self.simulation_mode = simulation_mode
        if simulation_mode:
            self._inject_implementation(ReactorServicerSimulation(self.reactors))
        else:
            self._inject_implementation(ReactorServicerReal(self.reactors))

    def _inject_implementation(self,
                               implementation: Union[ReactorServicerSimulation,
                                                     ReactorServicerReal]
                               ) -> bool:
        """
        Dependency injection of the implementation used.
            Allows to set the class used for simulation/real mode.

        :param implementation: A valid implementation of the DASGIP_ServiceServicer.
        """

        self.implementation = implementation
        return True

    def switch_to_simulation_mode(self):
        """Method that will automatically be called by the server when the simulation mode is requested."""
        self.simulation_mode = True
        self._inject_implementation(ReactorServicerSimulation(self.reactors))

    def switch_to_real_mode(self):
        """Method that will automatically be called by the server when the real mode is requested."""
        self.simulation_mode = False
        self._inject_implementation(ReactorServicerReal(self.reactors))

    def GetVInitial(self, request, context: grpc.ServicerContext) \
            -> ReactorServicer_pb2.GetVInitial_Responses:
        """
        Executes the unobservable command "Get VInitial"
            Get initial liquid volume value. Initial liquid volume (initial working volume).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVInitial (Current VInitial): Current initial liquid volume value. Initial liquid volume (initial working volume).
        """
    
        logging.debug(
            "GetVInitial called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetVInitial(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    
    def GetVLiquid(self, request, context: grpc.ServicerContext) \
            -> ReactorServicer_pb2.GetVLiquid_Responses:
        """
        Executes the unobservable command "Get VLiquid"
            Get actual liquid volume value. Actual liquid volume (working volume).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVLiquid (Current VLiquid): Current actual liquid volume value. Actual liquid volume (working volume).
        """
    
        logging.debug(
            "GetVLiquid called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetVLiquid(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetVLiquid(self, request, context: grpc.ServicerContext) \
            -> ReactorServicer_pb2.SetVLiquid_Responses:
        """
        Executes the unobservable command "Set VLiquid"
            Set the reactor actual liquid volume value. Actual liquid volume (working volume).
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.VLiquid (Volume Liquid):
            The reactor actual liquid volume value. Actual liquid volume (working volume).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.VLiquidSet (VLiquid Set): The set actual liquid volume value. Actual liquid volume (working volume).
        """
    
        logging.debug(
            "SetVLiquid called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetVLiquid(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetVMax(self, request, context: grpc.ServicerContext) \
            -> ReactorServicer_pb2.GetVMax_Responses:
        """
        Executes the unobservable command "Get VMax"
            Get max liquid volume value. Max allowed liquid volume.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVMax (Current VMax): Current max liquid volume value. Max allowed liquid volume.
        """
    
        logging.debug(
            "GetVMax called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetVMax(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetVMax(self, request, context: grpc.ServicerContext) \
            -> ReactorServicer_pb2.SetVMax_Responses:
        """
        Executes the unobservable command "Set VMax"
            Set the reactor max liquid volume value. Max allowed liquid volume.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.VMax (Volume Max):
            The reactor max liquid volume value. Max allowed liquid volume.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.VMaxSet (VMax Set): The set max liquid volume value. Max allowed liquid volume.
        """
    
        logging.debug(
            "SetVMax called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetVMax(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetVMin(self, request, context: grpc.ServicerContext) \
            -> ReactorServicer_pb2.GetVMin_Responses:
        """
        Executes the unobservable command "Get VMin"
            Get min liquid volume value. Min allowed liquid volume.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVMin (Current VMin): Current min liquid volume value. Min allowed liquid volume.
        """
    
        logging.debug(
            "GetVMin called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetVMin(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetVMin(self, request, context: grpc.ServicerContext) \
            -> ReactorServicer_pb2.SetVMin_Responses:
        """
        Executes the unobservable command "Set VMin"
            Set the reactor min liquid volume value. Min allowed liquid volume.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.VMin (Volume Min):
            The reactor min liquid volume value. Min allowed liquid volume.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.VMinSet (VMin Set): The set min liquid volume value. Min allowed liquid volume.
        """
    
        logging.debug(
            "SetVMin called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetVMin(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetVTotal(self, request, context: grpc.ServicerContext) \
            -> ReactorServicer_pb2.GetVTotal_Responses:
        """
        Executes the unobservable command "Get VTotal"
            Get total liquid volume value. Total vessel volume.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVTotal (Current VTotal): Current total liquid volume value. Total vessel volume.
        """
    
        logging.debug(
            "GetVTotal called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetVTotal(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetVTotal(self, request, context: grpc.ServicerContext) \
            -> ReactorServicer_pb2.SetVTotal_Responses:
        """
        Executes the unobservable command "Set VTotal"
            Set the reactor total liquid volume value. Total vessel volume.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
            request.VTotal (Volume Total):
            The reactor total liquid volume value. Total vessel volume.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.VTotalSet (VTotal Set): The set total liquid volume value. Total vessel volume.
        """
    
        logging.debug(
            "SetVTotal called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetVTotal(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)

    
