"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Turbidity Servicer*

:details: TurbidityServicer:
    Control a DASGIP turbidity module. Enables read and write operations for various parameters, including turbidity
    sensor, controller, and alarm.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 20.05.2019
           
:file:    TurbidityServicer_servicer.py
:authors: Lukas Bromig

:date: (creation)          2020-04-16T10:19:15.302172
:date: (last modification) 2020-04-16T10:19:15.302172

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "0.0.1"

# import general packages
import logging
import grpc

# meta packages
from typing import Union

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2
from sila2lib.error_handling.server_err import SiLAError

# import gRPC modules for this feature
from .gRPC import TurbidityServicer_pb2 as TurbidityServicer_pb2
from .gRPC import TurbidityServicer_pb2_grpc as TurbidityServicer_pb2_grpc

# import simulation and real implementation
from .TurbidityServicer_simulation import TurbidityServicerSimulation
from .TurbidityServicer_real import TurbidityServicerReal


class TurbidityServicer(TurbidityServicer_pb2_grpc.TurbidityServicerServicer):
    """
    This is a DASGIP Service
    """
    implementation: Union[TurbidityServicerSimulation, TurbidityServicerReal]
    simulation_mode: bool

    def __init__(self, reactors, simulation_mode: bool = True):
        """
        Class initialiser.

        :param simulation_mode: Sets whether at initialisation the simulation mode is active or the real mode.
        """
        self.reactors = reactors
        self.simulation_mode = simulation_mode
        if simulation_mode:
            self._inject_implementation(TurbidityServicerSimulation(self.reactors))
        else:
            self._inject_implementation(TurbidityServicerReal(self.reactors))

    def _inject_implementation(self,
                               implementation: Union[TurbidityServicerSimulation,
                                                     TurbidityServicerReal]
                               ) -> bool:
        """
        Dependency injection of the implementation used.
            Allows to set the class used for simulation/real mode.

        :param implementation: A valid implementation of the DASGIP_ServiceServicer.
        """

        self.implementation = implementation
        return True

    def switch_to_simulation_mode(self):
        """Method that will automatically be called by the server when the simulation mode is requested."""
        self.simulation_mode = True
        self._inject_implementation(TurbidityServicerSimulation(self.reactors))

    def switch_to_real_mode(self):
        """Method that will automatically be called by the server when the real mode is requested."""
        self.simulation_mode = False
        self._inject_implementation(TurbidityServicerReal(self.reactors))

    def GetAUPV(self, request, context: grpc.ServicerContext) \
            -> TurbidityServicer_pb2.GetAUPV_Responses:
        """
        Executes the unobservable command "Get AU PV"
            Get present value in absorption unit. Turbidity signal in absorption units.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAUPV (Current AU PV): Current present value in absorption unit. Turbidity signal in absorption units.
        """
    
        logging.debug(
            "GetAUPV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetAUPV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetCXPV(self, request, context: grpc.ServicerContext) \
            -> TurbidityServicer_pb2.GetCXPV_Responses:
        """
        Executes the unobservable command "Get CX PV"
            Get calculated turbidity present value. Calculated turbidity signal, i.e. OD600, CDW or others.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentCXPV (Current CX PV): Current calculated turbidity present value. Calculated turbidity signal, i.e. OD600, CDW or others.
        """
    
        logging.debug(
            "GetCXPV called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetCXPV(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetType(self, request, context: grpc.ServicerContext) \
            -> TurbidityServicer_pb2.GetType_Responses:
        """
        Executes the unobservable command "Get Function Type"
            Get function type.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentType (Current Function Type): Current function type value of the turbidity module.
        """
    
        logging.debug(
            "GetType called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetType(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetAvailable(self, request, context: grpc.ServicerContext) \
            -> TurbidityServicer_pb2.GetAvailable_Responses:
        """
        Executes the unobservable command "Get Function Availability"
            Get function availability.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAvailable (Current Function Availability): Current function availability value of the turbidity module.
        """
    
        logging.debug(
            "GetAvailable called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetAvailable(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetName(self, request, context: grpc.ServicerContext) \
            -> TurbidityServicer_pb2.GetName_Responses:
        """
        Executes the unobservable command "Get Function Name"
            Get function name.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentName (Current Function Name): Current function name of the turbidity module.
        """
    
        logging.debug(
            "GetName called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetName(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetVersion(self, request, context: grpc.ServicerContext) \
            -> TurbidityServicer_pb2.GetVersion_Responses:
        """
        Executes the unobservable command "Get Function Version"
            Get function model version number.
    
        :param request: gRPC request containing the parameters passed:
            request.UnitID (UnitID):
            The UnitID of the adressed reactor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVersion (Current Function Version): Current function model version number of the turbidity module.
        """
    
        logging.debug(
            "GetVersion called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetVersion(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)

    
