"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Alarm Service*

:details: AlarmService:
    Allows full control of the alarms and limits of the flowmeter
    By Lukas Bromig and Jose de Jesus Pina, Institute of Biochemical Engineering, Technical University of Munich,
    02.12.2020
           
:file:    AlarmService_servicer.py
:authors: Lukas Bromig and Jose de Jesus Pina

:date: (creation)          2021-03-19T13:54:01.461485
:date: (last modification) 2021-03-19T13:54:01.461485

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import grpc

# meta packages
from typing import Union

# import SiLA2 library
from sila2lib.error_handling.server_err import SiLAError

# import gRPC modules for this feature
from .gRPC import AlarmService_pb2 as AlarmService_pb2
from .gRPC import AlarmService_pb2_grpc as AlarmService_pb2_grpc

# import simulation and real implementation
from .AlarmService_simulation import AlarmServiceSimulation
from .AlarmService_real import AlarmServiceReal


class AlarmService(AlarmService_pb2_grpc.AlarmServiceServicer):
    """
    This is a flowmeter service
    """
    implementation: Union[AlarmServiceSimulation, AlarmServiceReal]
    simulation_mode: bool

    def __init__(self, ser, lock, status, simulation_mode: bool = True):
        """
        Class initialiser.

        :param simulation_mode: Sets whether at initialisation the simulation mode is active or the real mode.
        """
        self.lock = lock
        self.ser = ser
        self.status = status
        self.simulation_mode = simulation_mode
        if simulation_mode:
            self._inject_implementation(AlarmServiceSimulation(status=self.status))
        else:
            self._inject_implementation(AlarmServiceReal(ser=self.ser, status=self.status))

    def _inject_implementation(self,
                               implementation: Union[AlarmServiceSimulation,
                                                     AlarmServiceReal]
                               ) -> bool:
        """
        Dependency injection of the implementation used.
            Allows to set the class used for simulation/real mode.

        :param implementation: A valid implementation of the FlowmeterServiceServicer.
        """

        self.implementation = implementation
        return True

    def switch_to_simulation_mode(self):
        """Method that will automatically be called by the server when the simulation mode is requested."""
        self.simulation_mode = True
        self._inject_implementation(AlarmServiceSimulation(status=self.status))

    def switch_to_real_mode(self):
        """Method that will automatically be called by the server when the real mode is requested."""
        self.simulation_mode = False
        self._inject_implementation(AlarmServiceReal(ser=self.ser, status=self.status))

    def GetAlarmLimitMaximum(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.GetAlarmLimitMaximum_Responses:
        """
        Executes the unobservable command "Get alarm limit maximum"
            The current maximum limit for sensor signal to trigger alarm situation.
            Value in percentage. Range between 0 and 100
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAlarmLimitMaximum (Current alarm limit maximum): Current alarm limit maximum value. In percentage.
        """
    
        logging.debug(
            "GetAlarmLimitMaximum called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            self.lock.acquire()
            self.status = "Flowmeter executing AlarmLimitMaximum"
            return self.implementation.GetAlarmLimitMaximum(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
        finally:
            self.lock.release()


    def SetAlarmLimitMaximum(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.SetAlarmLimitMaximum_Responses:
        """
        Executes the unobservable command "Set alarm limit maximum"
            Set maximum limit for sensor to trigger alarm situation
    
        :param request: gRPC request containing the parameters passed:
            request.SetAlarmLimitMaximum (Set alarm limit maximum):
            Maximum limit for sensor to trigger alarm situation to be set. Value in percentage.
            Range from 0.00 to 100
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command alarm limit maximum
            request.IndexPointing (Index pointing): Index pointing to the first byte in the send message for which the alarm limit maximum status applies
        """
    
        logging.debug(
            "SetAlarmLimitMaximum called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetAlarmLimitMaximum(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetAlarmLimitMinimum(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.GetAlarmLimitMinimum_Responses:
        """
        Executes the unobservable command "Get alarm limit minimum"
            The current minimum limit for sensor signal to trigger alarm situation.
            Value in percentage. Range between 0,0 and 100,0
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAlarmLimitMinimum (Current alarm limit minimum): Current alarm limit minimum. In percentage
        """
    
        logging.debug(
            "GetAlarmLimitMinimum called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetAlarmLimitMinimum(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetAlarmLimitMinimum(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.SetAlarmLimitMinimum_Responses:
        """
        Executes the unobservable command "Set alarm limit minimum"
            Minimum limit for sensor to trigger alarm situation to be set. Value in percentage.
            Range from 0.00 to 100.
    
        :param request: gRPC request containing the parameters passed:
            request.SetAlarmLimitMinimum (Set alarm limit minimum):
            Minimum limit for sensor to trigger alarm situation to be set. Value in percentage. Range from 0.00 to 100
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command alarm limit minimum
            request.IndexPointing (Index pointing): Index pointing to the first byte in the send message for which the alarm limit minimum status applie
            s
        """
    
        logging.debug(
            "SetAlarmLimitMinimum called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetAlarmLimitMinimum(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetAlarmMode(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.GetAlarmMode_Responses:
        """
        Executes the unobservable command "Get alarm Mode"
            Current alarm mode
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAlarmMode (Current alarm mode): Current alarm mode
        """
    
        logging.debug(
            "GetAlarmMode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetAlarmMode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetAlarmMode(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.SetAlarmMode_Responses:
        """
        Executes the unobservable command "Set alarm mode"
            Set the alarm mode
    
        :param request: gRPC request containing the parameters passed:
            request.SetAlarmMode (Set alarm mode):
            The alarm mode to be set. 255 different mode available to be set.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set alarm mode
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the alarm mode status applies
        """
    
        logging.debug(
            "SetAlarmMode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetAlarmMode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetAlarmOutputMode(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.GetAlarmOutputMode_Responses:
        """
        Executes the unobservable command "Get alarm output mode"
            Alarm release mode during alarm situation.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAlarmOutputMode (Current alarm output mode): Current alarm output mode
        """
    
        logging.debug(
            "GetAlarmOutputMode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetAlarmOutputMode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetAlarmOutputMode(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.SetAlarmOutputMode_Responses:
        """
        Executes the unobservable command "Set alarm output mode"
            Set alarm release activity mode during alarm situation
    
        :param request: gRPC request containing the parameters passed:
            request.SetAlarmOutputMode (Set alarm output mode):
            The alarm output mode to be set. 255 different settable modes available
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set alarm mode
            request.IndexPointing (Index pointing): Index pointing to the first byte in the send message for which the alarm mode status applies
        """
    
        logging.debug(
            "SetAlarmOutputMode called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetAlarmOutputMode(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetAlarmSetpoint(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.GetAlarmSetpoint_Responses:
        """
        Executes the unobservable command "Get alarm setpoint"
            The threshold value or alarm setpoint which triggers an alarm situation
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAlarmSetpointMode (Current alarm setpoint mode): Current alarm setpoint mode
        """
    
        logging.debug(
            "GetAlarmSetpoint called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetAlarmSetpoint(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetAlarmSetpoint(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.SetAlarmSetpoint_Responses:
        """
        Executes the unobservable command "Set alarm setpoint"
            Set new alarm setpoint which triggers an alarm situation. 255 different settable modes available
    
        :param request: gRPC request containing the parameters passed:
            request.SetAlarmSetpoint (Set Alarm Setpoint):
            The new alarm setpoint
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set alarm setpint
            request.IndexPointing (Index pointing): Index pointing to the first byte in the send message for which the alarm setpoint status applies
        """
    
        logging.debug(
            "SetAlarmSetpoint called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetAlarmSetpoint(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)

    
