"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Alarm Service*

:details: AlarmService:
    Allows full control of the alarms and limits of the flowmeter
    By Lukas Bromig and Jose de Jesus Pina, Institute of Biochemical Engineering, Technical University of Munich,
    02.12.2020

:file:    AlarmService_simulation.py
:authors: Lukas Bromig and Jose de Jesus Pina

:date: (creation)          2021-03-19T13:54:01.464463
:date: (last modification) 2021-03-19T13:54:01.464463

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import inspect      # used for status determination
import grpc         # used for type hinting only
from struct import * #used to code binary protocol


# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import AlarmService_pb2 as AlarmService_pb2
# from .gRPC import AlarmService_pb2_grpc as AlarmService_pb2_grpc

# import default arguments
from .AlarmService_default_arguments import default_dict

from sila2lib_implementations.Flowmeter.FlowmeterService.lib import decode_response_read, decode_response_write


# noinspection PyPep8Naming,PyUnusedLocal
class AlarmServiceSimulation:
    """
    Implementation of the *Alarm Service* in *Simulation* mode
        This is a flowmeter service
    """

    def __init__(self, status):
        """Class initialiser"""
        self.status = status
        self.protocol = 'ascii'
        self.node = '03'
        logging.debug('Started server in mode: {mode}'.format(mode='Simulation'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None


        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def _get_function_name(self):
        return inspect.stack()[1][3]

    def GetAlarmLimitMaximum(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.GetAlarmLimitMaximum_Responses:
        """
        Executes the unobservable command "Get alarm limit maximum"
            The current maximum limit for sensor signal to trigger alarm situation.
            Value in percentage. Range between 0 and 100
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAlarmLimitMaximum (Current alarm limit maximum): Current alarm limit maximum value. In percentage.
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '61'        # Process number in hexadecimal
            parameter_query = '21'      # Parameter query in hexadecimal
            process_response = '61'     # Process response in hexadecimal
            parameter_response = '21'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 97          # Process number in decimal
            parameter_query = 33        # Process response in decimal
            process_response = 97        # Parameter query in decimal
            parameter_response = 33      # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    #read = ':06800261217777\r\n'
                    read = ':0403000405\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02a!\x7f\x10\x03'

                type_answer = default_dict['GetAlarmLimitMaximum_Responses']['CurrentAlarmLimitMaximum'].value

                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentAlarmLimitMaximum': silaFW_pb2.Real(
                        value=float('%.2f' % (round((int(response['message']) / 416), 2))))
                }
                return_value = AlarmService_pb2.GetAlarmLimitMaximum_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = AlarmService_pb2.GetAlarmLimitMaximum_Responses(
                **default_dict['GetAlarmLimitMaximum_Responses']
            )
    
        return return_value


    def SetAlarmLimitMaximum(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.SetAlarmLimitMaximum_Responses:
        """
        Executes the unobservable command "Set alarm limit maximum"
            Set maximum limit for sensor to trigger alarm situation
    
        :param request: gRPC request containing the parameters passed:
            request.SetAlarmLimitMaximum (Set alarm limit maximum):
            Maximum limit for sensor to trigger alarm situation to be set. Value in percentage.
            Range from 0.00 to 100
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command alarm limit maximum
            request.IndexPointing (Index pointing): Index pointing to the first byte in the send message for which the alarm limit maximum status applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_in_percent = request.SetAlarmLimitMaximum.value
        input_integer_decimal = 416*input_in_percent
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(4)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'              # 01=write parameter
            process_query = '61'        # Process number in hexadecimal
            parameter_query = '21'      # Parameter query in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1                 # 01=write parameter
            process_query = 97          # Process number in decimal
            parameter_query = 21        # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0480000005\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'
                type_answer = default_dict['SetAlarmLimitMaximum_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = AlarmService_pb2.SetAlarmLimitMaximum_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = AlarmService_pb2.SetAlarmLimitMaximum_Responses(
                **default_dict['SetAlarmLimitMaximum_Responses']
            )

        return return_value

    def GetAlarmLimitMinimum(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.GetAlarmLimitMinimum_Responses:
        """
        Executes the unobservable command "Get alarm limit minimum"
            The current minimum limit for sensor signal to trigger alarm situation.
            Value in percentage. Range between 0,0 and 100,0
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAlarmLimitMinimum (Current alarm limit minimum): Current alarm limit minimum. In percentage
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '61'        # Process number in hexadecimal
            parameter_query = '22'      # Parameter query in hexadecimal
            process_response = '61'     # Process response in hexadecimal
            parameter_response = '22'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 97          # Process number in decimal
            parameter_query = 34        # Process response in decimal
            process_response = 97       # Parameter query in decimal
            parameter_response = 34     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':068002612256F0\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02a!v\x10\x03'

                type_answer = default_dict['GetAlarmLimitMinimum_Responses']['CurrentAlarmLimitMinimum'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentAlarmLimitMinimum': silaFW_pb2.String(
                        value=('%.2f' % (round((int(response['message']) / 416), 2))))
                }
                return_value = AlarmService_pb2.GetAlarmLimitMinimum_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = AlarmService_pb2.GetAlarmLimitMinimum_Responses(
                **default_dict['GetAlarmLimitMinimum_Responses']
            )
    
        return return_value


    def SetAlarmLimitMinimum(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.SetAlarmLimitMinimum_Responses:
        """
        Executes the unobservable command "Set alarm limit minimum"
            Minimum limit for sensor to trigger alarm situation to be set. Value in percentage.
            Range from 0.00 to 100.
    
        :param request: gRPC request containing the parameters passed:
            request.SetAlarmLimitMinimum (Set alarm limit minimum):
            Minimum limit for sensor to trigger alarm situation to be set. Value in percentage. Range from 0.00 to 100
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command alarm limit minimum
            request.IndexPointing (Index pointing): Index pointing to the first byte in the send message for which the alarm limit minimum status applie
            s
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_in_percent = request.SetAlarmLimitMinimum.value
        input_integer_decimal = 416*input_in_percent
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(4)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '61'    # Process number in hexadecimal
            parameter_query = '22'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 97      # Process number in decimal
            parameter_query = 34    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)

        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0480000005\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'
                type_answer = default_dict['SetAlarmLimitMinimum_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = AlarmService_pb2.SetAlarmLimitMinimum_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = AlarmService_pb2.SetAlarmLimitMinimum_Responses(
                **default_dict['SetAlarmLimitMinimum_Responses']
            )
    
        return return_value


    def GetAlarmMode(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.GetAlarmMode_Responses:
        """
        Executes the unobservable command "Get alarm Mode"
            Current alarm mode
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAlarmMode (Current alarm mode): Current alarm mode
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '61'        # Process number in hexadecimal
            parameter_query = '03'      # Parameter query in hexadecimal
            process_response = '61'     # Process response in hexadecimal
            parameter_response = '03'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 97          # Process number in decimal
            parameter_query = 3         # Process response in decimal
            process_response = 97       # Parameter query in decimal
            parameter_response = 3      # Parameter response in decimal

            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    #read = ':050302610302\r\n'
                    read = ':0403000405\r\n'
                else:
                    read =  b'\x10\x02\x01\x03\x04\x02a\x03\x01\x10\x03'
                type_answer = default_dict['GetAlarmMode_Responses']['CurrentAlarmMode'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {

                    'CurrentAlarmMode': silaFW_pb2.String(value=str(response['message']))
                }
                return_value = AlarmService_pb2.GetAlarmMode_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = AlarmService_pb2.GetAlarmMode_Responses(
                **default_dict['GetAlarmMode_Responses']
            )
    
        return return_value

    def SetAlarmMode(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.SetAlarmMode_Responses:
        """
        Executes the unobservable command "Set alarm mode"
            Set the alarm mode
    
        :param request: gRPC request containing the parameters passed:
            request.SetAlarmMode (Set alarm mode):
            The alarm mode to be set. 255 different mode available to be set.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set alarm mode
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the alarm mode status applies
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetAlarmMode.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '61'    # Process number in hexadecimal
            parameter_query = '03'  # Parameter query in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 97      # Process number in decimal
            parameter_query = 3     # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         input_integer_decimal, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0480000003\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'

                type_answer = default_dict['SetAlarmMode_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = AlarmService_pb2.SetAlarmMode_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = AlarmService_pb2.SetAlarmMode_Responses(
                **default_dict['SetAlarmMode_Responses']
            )
    
        return return_value
    
    
    def GetAlarmOutputMode(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.GetAlarmOutputMode_Responses:
        """
        Executes the unobservable command "Get alarm output mode"
            Alarm release mode during alarm situation.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAlarmOutputMode (Current alarm output mode): Current alarm output mode
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '61'        # Process number in hexadecimal
            parameter_query = '04'      # Parameter query in hexadecimal
            process_response = '61'     # Process response in hexadecimal
            parameter_response = '04'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4             # 04=Request parameter
            process_query = 97      # Process number in decimal
            parameter_query = 4     # Process response in decimal
            process_response = 97   # Parameter query in decimal
            parameter_response = 4  # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)

        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302610403\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02a\x04\x01\x10\x03'

                type_answer = default_dict['GetAlarmOutputMode_Responses']['CurrentAlarmOutputMode'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentAlarmOutputMode': silaFW_pb2.String(value=str(response['message']))
                }
                return_value = AlarmService_pb2.GetAlarmOutputMode_Responses(**par_dict)

                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = AlarmService_pb2.GetAlarmOutputMode_Responses(
                **default_dict['GetAlarmOutputMode_Responses']
            )
    
        return return_value
    
    
    def SetAlarmOutputMode(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.SetAlarmOutputMode_Responses:
        """
        Executes the unobservable command "Set alarm output mode"
            Set alarm release activity mode during alarm situation
    
        :param request: gRPC request containing the parameters passed:
            request.SetAlarmOutputMode (Set alarm output mode):
            The alarm output mode to be set. 255 different settable modes available
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set alarm mode
            request.IndexPointing (Index pointing): Index pointing to the first byte in the send message for which the alarm mode status applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetAlarmOutputMode.value
        hex_input = hex(input_integer_decimal)
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '61'    # Process number in hexadecimal
            parameter_query = '04'  # Parameter query in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4             # 01=write parameter
            process_query = 97      # Process number in decimal
            parameter_query = 4     # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         input_integer_decimal, dle, etx)

        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0480000003\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'

                type_answer = default_dict['SetAlarmOutputMode_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = AlarmService_pb2.SetAlarmOutputMode_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = AlarmService_pb2.SetAlarmOutputMode_Responses(
                **default_dict['SetAlarmOutputMode_Responses']
            )
    
        return return_value
    
    
    def GetAlarmSetpoint(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.GetAlarmSetpoint_Responses:
        """
        Executes the unobservable command "Get alarm setpoint"
            The threshold value or alarm setpoint which triggers an alarm situation
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAlarmSetpointMode (Current alarm setpoint mode): Current alarm setpoint mode
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '61'        # Process number in hexadecimal
            parameter_query = '05'      # Parameter query in hexadecimal
            process_response = '61'     # Process response in hexadecimal
            parameter_response = '05'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 97          # Process number in decimal
            parameter_query = 5         # Process response in decimal
            process_response = 97       # Parameter query in decimal
            parameter_response = 5      # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302610504\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02a\x05\x01\x10\x03'
                type_answer = default_dict['GetAlarmSetpoint_Responses']['CurrentAlarmSetpointMode'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentAlarmSetpointMode': silaFW_pb2.String(value=str(response['message']))
                }
                return_value = AlarmService_pb2.GetAlarmSetpoint_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = AlarmService_pb2.GetAlarmSetpoint_Responses(
                **default_dict['GetAlarmSetpoint_Responses']
            )
    
        return return_value
    
    
    def SetAlarmSetpoint(self, request, context: grpc.ServicerContext) \
            -> AlarmService_pb2.SetAlarmSetpoint_Responses:
        """
        Executes the unobservable command "Set alarm setpoint"
            Set new alarm setpoint which triggers an alarm situation. 255 different settable modes available
    
        :param request: gRPC request containing the parameters passed:
            request.SetAlarmSetpoint (Set Alarm Setpoint):
            The new alarm setpoint
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set alarm setpint
            request.IndexPointing (Index pointing): Index pointing to the first byte in the send message for which the alarm setpoint status applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetAlarmSetpoint.value
        hex_input = hex(input_integer_decimal)
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '61'    # Process number in hexadecimal
            parameter_query = '05'  # Parameter query in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{input_integer_decimal}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4             # 01=write parameter
            process_query = 97      # Process number in decimal
            parameter_query = 5     # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         input_integer_decimal, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0403000001\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'

                type_answer = default_dict['SetAlarmSetpoint_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = AlarmService_pb2.SetAlarmSetpoint_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = AlarmService_pb2.SetAlarmSetpoint_Responses(
                **default_dict['SetAlarmSetpoint_Responses']
            )

        return return_value
